;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_survey_plot
;
; Purpose     :	Converts CEPPAD level 0 data to RAL intermediate format for survey plots.
;
; Explanation : IES data with IPS plot for comparison.
;               Prompts for CEPPAD level 0 file names.
;               Converts each level 0 file to level 1 files.
;               Converts the level 1 files to the RAL intermediate format needed for the
;               survey plots.
;               Sets default file name for plots.
;               Optionally produces pedestal data file and plot.
;
; Use         : < if_files = cep_survey_plot ( /L1, /RF, /WINDOWS, /PS, /PNG, /GIF, /BWPS, $
;                                              T0=t0, T1=t1, PEDESTAL=pedestal,
;                                              BF=bf, GZIP=gzip, TYPE=type, REMOVE=remove ) >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : if_files : STRARR containing names of IES intermediate format files used.
;               CEPPAD level 1 and RAL intermediate format files.
;
; Opt. Outputs:	None.
;
; Keywords    : /L1         = Specifies that the l1 files already exist.
;               RF         = Specifies that the if files already exist.
;                             Either Keyword set or STRARR(5,n) containing if files.
;               /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file
;                                if string then assumes it is a file name
;               /PNG        = plots output as PNG file
;                                if string then assumes it is a file name
;               /GIF        = plots output as GIF file
;                                if string then assumes it is a file name
;               /BWPS       = plots output as black and white postscript file
;                                if string then assumes it is a file name
;               /NOCT       = does not load default color table
;               /GZIP       = gzip any files created
;               T0         : STRING giving start time of period of interest in level 1
;                            data. Example format = '10:00:00', ignores day.
;               T1         : STRING giving end   time of period of interest in level 1
;                            data. Example format = '11:00:00', ignores day.
;               PEDESTAL   : Flag indicating pedestal output required.
;               BF         : Flag indicating magnetic field output required.
;               /DEBUG      = STOPs in routine.
;               TYPE        = INT giving type of survey plot required.
;                                0 : IES/IPS standard plot
;                                1 : HISTE/IES/IPS pitch angle plot
;               /REMOVE     = Remove tmp file created.
;               /CHECK      = Check if decompressed file already exists before decompressing.
;               /WEB        = Indicates want WEB type file name : yymmdd.ext.
;               /SHRINK     = Indicates shrink to data.
;               /ALL        = FLAG indicating that all LZ files in specified directory should be processed.
;                                 Avoids use of pickfile.
;               /COPY       = Indicates should copy OR and AT files from CDROM.
;               /FORCE      = forces creation of intermediate files.
;
; Calls       :	cep_survey_files, strip_dirname, datatype, cep_exttime,
;               cep_get_lzday, ies_window_plot, ies_restore_gzip, cep_extract, get_at, get_mf.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 18/9/96
;
; Modified    :	Version 0.1, MKC, 7/11/96
;                 Corrected bug in processing rf keyword.
;                 Added start and end time keywords.
;               Version 0.2, 26/11/96
;                 Changed IPS colour scale.
;               Version 0.3, 4/12/96
;                 Fixed ps output for non standard plots.
;               Version 0.4, 11/12/96
;                 Added QUIET keyword to ies_calibrate.
;               Version 0.5, 16/1/97
;                 Renamed output_info.clean tag to .shift.
;                 Modified default color table.
;                 Modified default survey plot name.
;                 Added AXES button.
;                 Added rng to output_info.
;                 Added full_only to output_info.
;                 Set no bad data.
;                 Added PEDESTAL keyword.
;               Version 0.6, MKC, 21/3/97
;                 Added pedestal plot production.
;               Version 0.7, MKC, 15/4/97
;                 Changed so that .cal, .div and .rat are reversible.
;                 Modified to use ies extdata. This fixes problem with sector plot for cleaned data.
;                 Changed so that picks up multiple l1 and if files.
;                 Fixed pedestal plot range to [-5,10]
;               Version 0.8, MKC, 2/5/97
;                 Created cep_survey_files.
;                 Changed how top page position defined : full_only = 0, pg_pos=0
;               Version 0.9, MKC, 19/8/97
;                 Set ELOG button on for plots.
;               Version 1.0, 28/8/97
;                 Changed so that input times not changed.
;                 Added elo, ehi to output_info
;             	Version 1.1, MKC, 10/9/97.
;                 Added error return to top level.
;                 Added better extraction of lz date.
;             	Version 1.2, MKC, 24/9/97.
;                 Added /L1 and /RF keywords.
;               Version 1.3, MKC, 4/11/97
;                 Removed bottom channel from IES plots by setting lowest energy to 30 keV.
;               Version 1.4, MKC, 27/11/97
;                 Use ies_restore_gzip, cep_extract.
;                 Now displays IPS fluxes.
;                 Removed pedestal plot.
;               Version 1.5, MKC, 18/2/98
;                 Added DEBUG keyword.
;               Version 1.6, MKC, 2/10/98
;                 Fixed 24 bit color problem.
;               Version 1.7, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Extracted KP magnetic field.
;                 Allowed detector sum.
;                 Added TYPE keyword and pitch angle plot.
;                 Added COLTAB keyword to ies_window_plot.
;                 Added REMOVE keyword.
;               Version 1.8, 15/1/99
;                 Added non standard case if T0 or T1 set.
;               Version 1.9, 27/7/99
;                 Added HISTe pitch angle survey plot.
;               Version 2.0, 10/12/00
;                 Added PNG output
;               Version 2.1, 21/02/02
;                 Added IPS to pitch angle survey plot.
;               Version 2.2, 27/02/02
;                 Intercepted error messages from cep_extract.
;                 Added SURVEY_FILES keyword.
;               Version 2.3, 10/06/02
;                 Added check on output_info if not sufficient data.
;               Version 2.4, 10/07/02
;                 Removed COLTAB keywords. Set ncolors.
;                 Added CHECK keyword.
;                 Added JAVA and SHRINK keywords.
;                 Removed SURVEY_FILES keyword.
;               Version 2.5, 23/07/02
;                 Removed code resetting color table..
;               Version 2.6, 02/10/03
;                 Changed JAVA keyword to WEB
;                 Renamed cep survey.
;                 Change so that now set directory as well as filename of plot file here.
;                 Added /ALL keyword.
;                 Added /COPY keyword.
;                 Used ies get dir.
;                 Changed printout.
;               Version 2.7, 21/10/03
;                 Added FORCE keyword.
;
; Version     :	Version 2.7, 21/10/03
;-
;**********************************************************

FUNCTION cep_survey_plot, L1=l1, RF=rf, WINDOWS=windows, PS=postscript, PNG=png, GIF=gif, BWPS=bwps, NOCT=noct, $
                          T0=t0, T1=t1, PEDESTAL=pedestal, BF=bf, GZIP=gzip, TYPE=type, $
                          DEBUG=debug, REMOVE=remove, CHECK=check, WEB=web, ALL=all, COPY=copy, FORCE=force

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information  ** used
   ; data          = structure array containing extracted data               ** used
   ; output_info   = structure containing plotting and processing info       ** used
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, input_header, input_data, header, data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem, mf, at

  ; come back out to top level unless debug set

  IF KEYWORD_SET(debug) THEN ON_ERROR, 0 ELSE ON_ERROR, 1

  ; set colour table and ncolors

  IF NOT KEYWORD_SET(noct) THEN coltab = 39 ; RAINBOW white

  dummy = ies_ncolors(/SET,PRIVATE=coltab)

  ; check type set up

  IF N_ELEMENTS(type) EQ 0 THEN type = 0

  ; get intermediate format files

  PRINT, 'CREATING SURVEY FILES'

  survey_files = cep_survey_files ( L1=l1, RF=rf, PEDESTAL=pedestal, BF=bf, GZIP=gzip, DEBUG=debug, ALL=all, COPY=copy, FORCE=force )

  IF survey_files(0,0) EQ '' THEN BEGIN
    MESSAGE, 'No files found for processing', /CONT
    RETURN, ''
  ENDIF

  ; loop through level 0 files

  PRINT, 'CREATING SURVEY PLOTS'

  FOR f = 0, N_ELEMENTS(survey_files(0,*))-1 DO BEGIN

    ; set up file name variables

    ies_file   = survey_files(0,f)
    ips_file   = survey_files(1,f)
    histe_file = survey_files(2,f)
    histp_file = survey_files(3,f)

    ; print message

    PRINT, 'IES file   : ', ies_file
    PRINT, 'IPS file   : ', ips_file
    PRINT, 'HISTe file : ', histe_file
    PRINT, 'HISTp file : ', histp_file

    ; set flag indicating whether standard survey plot

    IF STRMID(strip_dirname(ies_file),0,28) EQ 'PO_CEP_RAL_CLEAN_SURVEY_IES_' $
       AND (NOT KEYWORD_SET(t0)) AND (NOT KEYWORD_SET(t1)) THEN standard = 1 ELSE standard = 0

    ; set file name for output

    IF NOT KEYWORD_SET(web) THEN BEGIN

      CASE type OF
        0 : ftype = 'cep_'
        1 : ftype = 'cep_pa_'
        2 : ftype = 'cep_histe_pa_'
        3 : ftype = 'cep_histp_pa_'
        ELSE : MESSAGE, 'Invalid TYPE'
      ENDCASE

      IF KEYWORD_SET(postscript) THEN BEGIN
        IF datatype(postscript) NE 'STR' THEN new_postscript=concat_dir(ies_get_dir('PS',/POLAR),ftype+cep_get_lzday(ies_file)+'.ps') ELSE new_postscript = postscript
      ENDIF ELSE IF KEYWORD_SET(bwps) THEN BEGIN
        IF datatype(bwps) NE 'STR' THEN new_bwps=concat_dir(ies_get_dir('PS',/POLAR),ftype+cep_get_lzday(ies_file)+'.ps') ELSE new_bwps = bwps
      ENDIF ELSE IF KEYWORD_SET(png) THEN BEGIN
        IF datatype(png) NE 'STR' THEN new_png=concat_dir(ies_get_dir('PNG',/POLAR),ftype+cep_get_lzday(ies_file)+'.png') ELSE new_png = png
      ENDIF ELSE IF KEYWORD_SET(gif) THEN BEGIN
        IF datatype(gif) NE 'STR' THEN new_gif=concat_dir(ies_get_dir('GIF',/POLAR),ftype+cep_get_lzday(ies_file)+'.gif') ELSE new_gif = gif
      ENDIF

    ENDIF ELSE BEGIN

      IF KEYWORD_SET(postscript) THEN BEGIN
        IF datatype(postscript) NE 'STR' THEN new_postscript=concat_dir(ies_get_dir('WEB',/POLAR),STRMID(cep_get_lzday(ies_file),2,6)+'.ps') ELSE new_postscript = postscript
      ENDIF ELSE IF KEYWORD_SET(bwps) THEN BEGIN
        IF datatype(bwps) NE 'STR' THEN new_bwps=concat_dir(ies_get_dir('WEB',/POLAR),STRMID(cep_get_lzday(ies_file),2,6)+'.ps') ELSE new_bwps = bwps
      ENDIF ELSE IF KEYWORD_SET(png) THEN BEGIN
        IF datatype(png) NE 'STR' THEN new_png=concat_dir(ies_get_dir('WEB',/POLAR),STRMID(cep_get_lzday(ies_file),2,6)+'.png') ELSE new_png = png
      ENDIF ELSE IF KEYWORD_SET(gif) THEN BEGIN
        IF datatype(gif) NE 'STR' THEN new_gif=concat_dir(ies_get_dir('WEB',/POLAR),STRMID(cep_get_lzday(ies_file),2,6)+'.gif') ELSE new_gif = gif
      ENDIF

    ENDELSE

    ; set default plot type

    IF NOT KEYWORD_SET(windows) AND NOT KEYWORD_SET(postscript) AND $
       NOT KEYWORD_SET(png) AND NOT KEYWORD_SET(gif) AND NOT KEYWORD_SET(bwps) THEN windows=1

    ; save current device

    devname = !D.NAME

    CASE type OF

      0 : BEGIN

            ; do IES/IPS spectral plots

            ; restore file

            ies_restore_gzip, ies_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; get ephemeris data for midpoint of range

            ephem = get_ephem ( (start_time+end_time)/2.0 )

            ; zero attitude and magnetic field data
            ; explicitly do not correct for sun on IPS

            at = 0
            mf = 0

            ; extract for detectors 1, 4 and 7

            IF cep_extract([start_time, end_time], [1,0,0,1,0,0,1,0,0]) THEN BEGIN

              ; process data

              IF cep_calibrate(0) THEN BEGIN

                ; do detector 1

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=0, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

                ; do detector 4

                output_info.full_detector = 1

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=1, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink, HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

                ; do detector 7

                output_info.full_detector = 2

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=2, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink, HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; do sector plot

            ; extract for detector 4

            IF cep_extract([start_time, end_time], [0,0,0,1,0,0,0,0,0], FORMAT='SECTOR vs DETECTOR') THEN BEGIN

              ; process data

              IF cep_calibrate(0, FORMAT='SECTOR vs DETECTOR') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=3, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink, HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

                ; restore file

                ies_restore_gzip, ips_file, input_header, input_data, REMOVE=remove, CHECK=check

                ; extract requested time range

                cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

              ENDIF

            ENDIF

            ; extract data for detector C2

            IF cep_extract([start_time, end_time], [0,0,0,0,1,0,0,0,0,0]) THEN BEGIN

              ; process data for detector C2

              IF cep_calibrate(0) THEN BEGIN

                ; fiddle things for bottom plot

                header.xtitle  = 'UT:'

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=4, SHRINK=shrink, $
                                HEADING = 'POLAR/CEPPAD SURVEY PLOT for ') THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

          END

      1 : BEGIN

            ; do pitch angle/detector average HISTe, IES, IPS plots

            ; restore HISTe file

            ies_restore_gzip, histe_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; get ephemeris data for midpoint of range

            ephem = get_ephem ( (start_time+end_time)/2.0 )

            ; get attitude data for midpoint of range

            at = get_at ( (start_time+end_time)/2.0 )

            ; get magnetic field data for midpoint of range

            mf = get_mf ( (start_time+end_time)/2.0 )

            ; extract data for detector 0

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC']) THEN BEGIN

              ; process data for detector 0

              IF cep_calibrate(0, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=0, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                HEADING = 'POLAR/CEPPAD PITCH/ANGLE PLOT for ') THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore IES file

            ies_restore_gzip, ies_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for all detectors for each energy

            IF cep_extract([start_time, end_time], [1,1,1,1,1,1,1,1,1]) THEN BEGIN

              ; process data for all detectors

              IF cep_calibrate(9,RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=1, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, all detectors, summed over energy

            IF cep_extract([start_time, end_time], [1,1,1,1,1,1,1,1,1], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[1.e2, 2.e7], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=2, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore IPS file

            ies_restore_gzip, ips_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; extract requested time range

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for inner 5 detectors

            IF cep_extract([start_time, end_time], [0,0,1,1,1,1,1,0,0,0]) THEN BEGIN

              ; process data for all detectors

              IF cep_calibrate(5, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=3, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, inner 5 detectors, summed over energy

            IF cep_extract([start_time, end_time], [0,0,1,1,1,1,1,0,0,0], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[1.e2, 2.e7], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; fiddle things for bottom plot

                header.xtitle  = 'UT:'

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=4, SHRINK=shrink) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

          END

      2 : BEGIN

            ; do pitch angle/detector average HISTe, IES, IPS plots

            ; restore HISTe file

            ies_restore_gzip, histe_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; get ephemeris data for midpoint of range

            ephem = get_ephem ( (start_time+end_time)/2.0 )

            ; get attitude data for midpoint of range

            at = get_at ( (start_time+end_time)/2.0 )

            ; get magnetic field data for midpoint of range

            mf = get_mf ( (start_time+end_time)/2.0 )

            ; extract data for pitch angle, detector 0, summed over energy

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC'], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[5.e1, 5.e6], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=0, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                HEADING = 'POLAR/CEPPAD PITCH/ANGLE PLOT for ' ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for detector 0

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC']) THEN BEGIN

              ; process data for detector 0

              IF cep_calibrate(0, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=1, /NO_XAXIS, /NO_END, SHRINK=shrink) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore IES file

            ies_restore_gzip, ies_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for all detectors for each energy

            IF cep_extract([start_time, end_time], [1,1,1,1,1,1,1,1,1]) THEN BEGIN

              ; process data for all detectors

              IF cep_calibrate(9,RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=2, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, all detectors, summed over energy

            IF cep_extract([start_time, end_time], [1,1,1,1,1,1,1,1,1], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[1.e2, 2.e7], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=3, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore IPS file

            ies_restore_gzip, ips_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; extract requested time range

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for inner 5 detectors

            IF cep_extract([start_time, end_time], [0,0,1,1,1,1,1,0,0,0]) THEN BEGIN

              ; process data for all detectors

              IF cep_calibrate(5, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=4, $
                                SHRINK=shrink ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

          END

      3 : BEGIN

            ; do pitch angle/detector average HISTe, HISTp, IES, IPS plots

            ; restore HISTe file

            ies_restore_gzip, histe_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; get ephemeris data for midpoint of range

            ephem = get_ephem ( (start_time+end_time)/2.0 )

            ; get attitude data for midpoint of range

            at = get_at ( (start_time+end_time)/2.0 )

            ; get magnetic field data for midpoint of range

            mf = get_mf ( (start_time+end_time)/2.0 )

            ; extract data for detector 0

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC']) THEN BEGIN

              ; process data for detector 0

              IF cep_calibrate(0, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=0, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                HEADING = 'POLAR/CEPPAD PITCH/ANGLE PLOT for ', NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, detector 0, summed over energy

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC'], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[5.e1, 5.e6], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=1, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore IES file

            ies_restore_gzip, ies_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for all detectors for each energy

            IF cep_extract([start_time, end_time], [1,1,1,1,1,1,1,1,1]) THEN BEGIN

              ; process data for all detectors

              IF cep_calibrate(9,RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=2, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink, NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, all detectors, summed over energy

            ; IF cep_extract([start_time, end_time], [1,1,1,1,1,1,1,1,1], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

            ; extract data for pitch angle, central detectors, summed over energy

            IF cep_extract([start_time, end_time], [0,0,0,0,1,0,0,0,0], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[1.e2, 2.e7], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=3, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink, NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore HISTp file

            ies_restore_gzip, histp_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; set up extraction start time and end time

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for detector 0

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC']) THEN BEGIN

              ; process data for detector 0

              IF cep_calibrate(0, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=4, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                NPANELS=9 ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, detector 0, summed over energy

            IF cep_extract([start_time, end_time], [1], MODES = ['ABC'], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[1.e0, 1.e5], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=5, /NO_XAXIS, /NO_END, SHRINK=shrink, $
                                NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; restore IPS file

            ies_restore_gzip, ips_file, input_header, input_data, REMOVE=remove, CHECK=check

            ; extract requested time range

            cep_exttime, input_header, start_time, end_time, T0=t0, T1=t1

            ; extract data for inner 5 detectors

            IF cep_extract([start_time, end_time], [0,0,1,1,1,1,1,0,0,0]) THEN BEGIN

              ; process data for all detectors

              IF cep_calibrate(5, RANGE=[1.e-2, 1.e5]) THEN BEGIN

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=6, /NO_XAXIS, /NO_END, $
                                SHRINK=shrink, NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

            ; extract data for pitch angle, inner 5 detectors, summed over energy

            IF cep_extract([start_time, end_time], [0,0,1,1,1,1,1,0,0,0], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

              ; process data for pitch angle

              IF cep_calibrate(0, RANGE=[1.e2, 2.e7], FORMAT='PITCH ANGLE vs ENERGY') THEN BEGIN

                ; fiddle things for bottom plot

                header.xtitle  = 'UT:'

                ; plot data

                IF ies_window_plot( devname, file, WINDOWS=windows, POSTSCRIPT=new_postscript, PNG=new_png, GIF=new_gif, $
                                BWPS=new_bwps, PG_POS=7, SHRINK=shrink, NPANELS=9, YTOP=ytop, YSIZ=ysiz ) THEN $
                  MESSAGE, 'ies_window_plot error', /TRACEBACK

              ENDIF

            ENDIF

          END

      ELSE : PRINT, 'WARNING, TYPE NOT KNOWN'

    ENDCASE

  ENDFOR

  ; leave with right file name

  files = [ '', ips_file]

  ; set up processing info
  ; if insufficient data then output_info not defined

  IF KEYWORD_SET(output_info) THEN BEGIN

    output_info.minimum = -1
    output_info.maximum = -1

  ENDIF

  ; error printout
  ; NB if windows then file=''

  IF N_ELEMENTS(file) EQ 0 THEN PRINT, 'WARNING, NO OUTPUT PRODUCED'

  ; return IEF file names

  RETURN, survey_files

END
