;+
; Project     :	POLAR - CEPPAD
;
; Name        :	cep_times
;
; Purpose     :	Fixes bad times, puts times into TAI format and selects period of interest.
;               
; Explanation : Times for spins without a corresponding spin header record are estimated by the
;               make_level_1 sw and flagged by negative time values. This routine produces a list of 
;               spins with estimated time values within period of interest and
;               prints out the no. of estimated values.
;               Times are converted to TAI format. Times within period of interest are extracted to nearest 
;               major spin. NB Period of interest does not use the day information.        
;
; Use         : < cep_times, mjulday, spin_times, bad_spin_times, first, last, START_TIME=start_time, END_TIME=end_time >
;
; Inputs      : mjulday        : LONG giving modified Julian day.
;               spin_times     : LONARR giving times of samples in milliseconds from start of day.
;                       Longword array containing the time of 
;                       day in milliseconds at the start of 
;                       acquisition of each spin containing
;                       LEVEL-1 IPS data for the user requested
;                       time range, 'TIME1 - TIME2'
;               no_of_sectors  : INT no. of sectors per spin.
;
; Opt. Inputs : None.
;
; Outputs     : bad_spin_times : LONARR giving spin indeces for bad spin times in period of interest.
;               spin_times     : DBLARR giving TAI times of spins in period of interest.
;               first          : LONG   giving major spin index for start of period of interest.
;               last           : LONG   giving major spin index for end   of period of interest.
;
; Opt. Outputs:	None.
;
; Keywords    : START_TIME : STRING giving start time of period of interest in level 1 
;                            data. Example format = '10:00:00', ignores day.
;               END_TIME   : STRING giving end   time of period of interest in level 1 
;                            data. Example format = '11:00:00', ignores day.
;
; Calls       :	utc2tai, anytim2utc.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, MKC, 13/06/96.
;                 The level 0-1 SW has been modified to correct bad spin times.
;                 The part of cep_times which does this has been removed.
;                 The spin time has been removed from the argument list.
;                 The end of data flag has been hardcoded 100 ms from last spin.
;               Version 0.2, MKC, 18/9/96
;                 Corrected bad spin times list.
;                 Fixed expanding REBIN bug.
;                 Removed extra dummy record. 
;                 Added no_of_sectors to argument list.
;               Version 0.3, MKC, 1/10/96
;                 Split into cep_times and cep_newtimes.
;
; Version     :	Version 0.3, 1/10/96
;-
;**********************************************************

PRO cep_times, mjulday, spin_times, bad_spin_times, first, last, START_TIME=start_time, END_TIME=end_time

  ; set up times

  ; get CDS_INT_TIME structure

  get_utc, dummy

  ; get tai of start of Julian day

  start_of_day = utc2tai ( {CDS_INT_TIME, mjd:mjulday, time:0L} )

  ; convert spin_times to tai

  tai_spin_times = start_of_day + ABS(spin_times)/1000.0D0

  ; get major spin times
  ; select first time of each 16 spin block

  major_spin_times = REBIN( tai_spin_times, N_ELEMENTS(tai_spin_times)/16, /SAMPLE)

  ; find major spin times within start and end time

  first = 0
  last  = N_ELEMENTS(tai_spin_times)/16

  IF KEYWORD_SET ( start_time ) THEN BEGIN

    ; adjust so that only uses time within day

    st_time = anytim2utc(start_time)

    st_time.mjd = mjulday

    list = WHERE ( major_spin_times GE utc2tai(st_time), count)

    IF count EQ 0 THEN MESSAGE, 'Invalid start time interval specified' , /TRACEBACK

    first = list(0)

  ENDIF

  IF KEYWORD_SET ( end_time ) THEN BEGIN

    ; adjust so that only uses time within day

    en_time = anytim2utc(end_time)

    en_time.mjd = mjulday

    list = WHERE ( major_spin_times LE utc2tai(en_time), count)

    IF count EQ 0 THEN MESSAGE, 'Invalid end time interval specified', /TRACEBACK

    last = 1+list(count-1)

  ENDIF

  ; extract relevant data

  spin_times     = spin_times(first*16:last*16-1)
      
  ; get indices of bad times 

  bad_spin_times = WHERE ( spin_times LT 0, bad_spin_count )

  ; set up output data

  spin_times     = tai_spin_times(first*16:last*16-1)
      
  ; printout warning message

  PRINT, 'Bad spin times  = ', STRTRIM(bad_spin_count,1) , '/', STRTRIM(N_ELEMENTS(spin_times),1)

END
