;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_AVERAGE
;
; Purpose     :	Calculates spin averages and detector averages.
;
; Explanation :	Calculates spin averages and detector averages with sector/detector mask appied.
;
; Use         : < array = ies_average (output_array, output_info, header, data) >
;
; Inputs      : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;               output_info   : STRUCTURE containing plotting and processing info
;               header        : STRUCTURE info structure for data.
;               data          : STRARR for each sample point.
;
; Opt. Inputs : None.
;
; Outputs     : array         : FLTARR(*,nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : SPIN_AVERAGE : Sum over sectors.
;               DETECTOR_AVERAGE : Sum over detectors.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/10/98
;                 Adapted from ies calibrate.
;
; Modified    :	Version 0.1, MKC, 11/11/98
;                 Dealt properly with HIST.
;                 Corrected IPS mask i.e masked out whole head rather than detector.
;               Version 0.2, MKC, 2/2/99
;                 Modified old pitch angle calculation which was rather obscure.
;                 Previously, divided each detector/sector bin into smaller bins
;                 to artificially increase the no. of samples. Divided pitch angle
;                 into a number of bins and accumulated histogram. If no pitch angle
;                 for a given bin then gave a zero which could give aliasing problems
;                 in ies_mk_image.
;                 Now, interpolate pitch angle samples to give distribution.
;               Version 0.3, MKC, 22/2/99
;                 Fixed bug if summing sectors over IPS background detector.
;                 Fixed bug averaging over small pixels for IES.
;               Version 0.4, MKC, 5/5/99
;                 Added routine ies pitch angle.
;               Version 0.5, 11/5/99
;                 Removed EARTH and SUN keywords.
;               Version 0.6, 20/8/00
;                 Dealt with RAPID data
;               Version 0.7, 12/07/02
;                 Corrected bug if no magnetic field data.
;               Version 0.8, 07/02/03
;                 Changed to 0-180 degrees pitch angle output.
;                 Set values for which no pitch angle measured equal to -1.
;               Version 0.9, 29/09/03
;                 Changed to selectable no. of pitch angle bins.
;                 Replaced NPITCHS keyword with PITCH_ANGLE keyword.
;                 Used new routine for instrument test.
;                 Fixed odd bug where pitch angle = 180 caused SW to crash.
;                 Can't believe that never encountered this before now ?
;               Version 1.0, 20/10/03
;                 Use routine ies generate masks.
;                 Updated pitch angle histogram calculation and prevented
;                 invalid/masked pitch angles appearing in bin zero.
;                 Checked for pitch angle tag in input.
;                 Moved pitch angle calculation to ies get pitch angle.
;               Version 1.1, 04/12/03
;                 Used cep small pixel.
;
; Version     :	Version 1.1, 04/12/03
;-
;**********************************************************

FUNCTION ies_average, output_array, output_info, header, data, SPIN_AVERAGE=spin_average, DETECTOR_AVERAGE=detector_average

  ; initialize array

  array = 0

  ; check processing required

  IF KEYWORD_SET(spin_average) THEN BEGIN

    ; spin average

    ; check if IPS

    IF (NOT ies_instrument(header.datatype,/IPS)) OR header.nsectors LE 1 THEN BEGIN

      array = TOTAL ( output_array, 1)

      ; get average count rate if rates

      IF output_info.rat THEN array = array / header.nsectors

    ENDIF ELSE BEGIN

      ; IPS and nsectors GT 1 : so apply masks

      ; set up output array

      array = FLTARR ( header.ndetectors, header.nbands, header.npoints)

      ; get masks

      ies_generate_masks, header, data, masks, /EARTH, /SUN

      ; get rid of -1 mask values

      masks = masks > 0

      ; get no of sectors/detectors contibuting

      tmasks = TOTAL(masks,1) > 1

      ; loop through each energy

      FOR e = 0, header.nbands-1 DO BEGIN

        ; NB mask is indexed by sector/detector order
        ; apply mask
        ; get count rate right

        array[*,e,*] = TOTAL(output_array[*,*,e,*]*masks,1)

        IF output_info.rat THEN array[*,e,*] = array[*,e,*]/tmasks

      ENDFOR

      ; bolt on background detector

      dlist = WHERE (header.detectors EQ 10, dcount) ; IPS background detector

      IF dcount GT 0 THEN BEGIN

        ; get average count rate if rates

        array[dlist,*,*] = TOTAL ( output_array[*,dlist,*,*], 1)

        IF output_info.rat THEN array[dlist,*,*] = array[dlist,*,*] / header.nsectors

      ENDIF

    ENDELSE

  ENDIF ELSE IF KEYWORD_SET(detector_average) THEN BEGIN

    ; NB masks are taken into account by doing sector averaging first
    ;    except where all sectors are invalid ie small pixels
    ;    then have to average over less detectors
    ; NB have to ensure IPS background detector not chosen.

    ; detector average

    array = TOTAL ( output_array, 2)  ; sum over detectors

    ; get average count rate if rates
    ; NB this will include background detector if selected

    IF output_info.rat THEN array = array / header.ndetectors

    ; check if IES so can do correct averaging over small pixels

    IF ies_instrument(header.datatype,/IES) AND NOT ies_instrument(header.datatype,/RAPID) THEN BEGIN

      ; get list of small pixels

      list = WHERE ( cep_small_pixel(header, LUT=data.lut), count)

      IF count GT 0 THEN BEGIN

        dlist = WHERE (((header.detectors-1) MOD 3) NE 1, dcount)

        IF dcount GT 0 THEN BEGIN
          array[*,*,list] = TOTAL ( output_array[*,dlist,*,list], 2)
          IF output_info.rat THEN array[*,*,list] = array[*,*,list] / dcount
        ENDIF ELSE BEGIN
          array[*,*,list] = 0
        ENDELSE

      ENDIF

    ENDIF

  ENDIF

  RETURN, array

END
