;+
; Project     :	CLUSTER - RAPID - PULSAUR
;
; Name        :	IES_BANDS
;
; Purpose     :	Widget tool for specifying energy bands selection.
;
; Explanation : Allows the selection of energy bands.
;
; Use         : <st = ies_bands, GROUP=group, GET_CURRENT=get_current, SET_CURRENT=set_current>
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : st : structure containing energy selection information.
;                     nbands    : INT no. of energy bands.
;                     use_bands : INTARR(nbands) : Indicates whether each band to be used.
;                     lb        : INTARR(nbands) : Lower channel bound on each band.
;                     ub        : INTARR(nbands) : Upper channel bound on each band.
;                     botchan   : INT lower channel for display (nbands=0)
;                     topchan   : INT upper channel for display (nbands=0)
;
; Opt. Outputs:	None.
;
; Keywords    : GROUP : Widget ID of parent.
;               GET_CURRENT : returns current band structure.
;               SET_CURRENT : sets current band structure.
;
; Calls       :	ies_do_bands, ies_bands_event.
;
; Common      :	ies_bands_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Command preparation.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 5/6/95
;
; Modified    :	Version 0.1, MKC, 25/2/97
;                 If set_current=-1 then data has been cleaned.
;               Version 0.2, MKC, 16/4/97
;                 Now allows bands to be set if data cleaned (set .editable=1 for nbands=-1
;                 Fixed bug for greater than 9 bands.
;               Version 0.3, MKC, 23/6/97
;                 Fixed behaviour for nchannels = 1.
;               Version 0.4, MKC, 24/2/99
;                 Scrolled window if nbands > 10.
;
; Version     :	Version 0.4, 24/2/99
;-
;**********************************************************
;**********************************************************
; ies_do_bands : widgets for bands input
;**********************************************************

PRO ies_do_bands

   COMMON ies_bands_block, id_nbands, id_bands_parent, id_bands_child, id_use, id_lb, id_ub, id_botchan, id_topchan, $
                           nchannels, bands

  ; must set up child widget first because still want to destroy it later

  IF bands.nbands LE 10 THEN $
    id_bands_child = WIDGET_BASE  ( id_bands_parent, /COLUMN ) $
  ELSE $
    id_bands_child = WIDGET_BASE  ( id_bands_parent, /COLUMN, /SCROLL, X_SCROll_SIZE=600, Y_SCROll_SIZE=512 )

  col = id_bands_child

  ; If nbands=0 then display last energy button
  ; else select channels

  IF bands.nbands GT 0 THEN BEGIN

    id_use = LONARR(bands.nbands)
    id_lb  = LONARR(bands.nbands)
    id_ub  = LONARR(bands.nbands)

    label  = WIDGET_LABEL ( col, VALUE = 'SELECT CHANNELS' )

    FOR k = 0, bands.nbands-1 DO BEGIN

      ch = STRTRIM(k,1)

      row = WIDGET_BASE  ( col, /ROW )

      label  = WIDGET_LABEL ( row, VALUE = 'BAND '+ch )

      ; Make the ON/OFF button:
      ; NB Couldnt get EXCLUSIVE buttons ot work on shaper

      tog    = WIDGET_BASE ( row, /ROW, /NONEXCLUSIVE)

      id_use(k)  = WIDGET_BUTTON (tog, $
  			      VALUE='USE', $	;The label for the button.
  			      UVALUE='US'+ch)           ;The User Value for the button.

      ; make the sliders

      id_lb(k) = WIDGET_SLIDER ( row, $
                                           MINIMUM = 0, $
                                           MAXIMUM = nchannels-1, $
                                           TITLE = 'LOWER', $
                                           UVALUE = 'LB'+ch,$
                                           VALUE  = bands.lb(k), $
                                           XSIZE = 200)

      id_ub(k) = WIDGET_SLIDER ( row, $
                                         MINIMUM = 0, $
                                         MAXIMUM = nchannels-1, $
                                         TITLE = 'UPPER', $
                                         UVALUE = 'UB'+ch,$
                                         VALUE  = bands.ub(k), $
                                         XSIZE = 200)
    ENDFOR

    ; set band buttons

    FOR k = 0, bands.nbands-1 DO WIDGET_CONTROL, id_use(k), SET_BUTTON=0

    list = WHERE ( bands.use_bands, nlist)

    FOR k = 0, nlist-1 DO WIDGET_CONTROL, id_use(list(k)), SET_BUTTON=1

  ENDIF ELSE BEGIN

    ; display first and last bands button

    row = WIDGET_BASE  ( col, /ROW )

    label  = WIDGET_LABEL ( row, VALUE = 'ALL ENERGY CHANNELS USED' )

    label  = WIDGET_LABEL ( row, VALUE = '  BOTTOM CHANNEL ' )

    id_botchan = WIDGET_TEXT ( row, VALUE=STRTRIM(bands.botchan,2), UVALUE= 'BC', /EDITABLE, XSIZE=3, /FRAME )

    label  = WIDGET_LABEL ( row, VALUE = 'TOP CHANNEL ' )

    id_topchan = WIDGET_TEXT ( row, VALUE=STRTRIM(bands.topchan,2), UVALUE= 'TC', /EDITABLE, XSIZE=3, /FRAME )

  ENDELSE

END

;**********************************************************
; ies_bands_event : event handler for main routine
;**********************************************************

PRO ies_bands_event, event

  COMMON ies_bands_block, id_nbands, id_bands_parent, id_bands_child, id_use, id_lb, id_ub, id_botchan, id_topchan, $
                           nchannels, bands

  WIDGET_CONTROL, event.id, GET_UVALUE = uvalue

  CASE STRMID(uvalue,0,2) OF

   'NB'  : BEGIN

             ; get new no. of bands
             ; nbands can be greater than nchannels
             ; nchannels has not changed

             WIDGET_CONTROL, id_nbands, GET_VALUE=value
             nbands = LONG(value(0)) > 0

             ; re-invent bands structure

             IF nbands EQ 0 THEN BEGIN

               bands = { nbands:nbands, use_bands:REPLICATE(1,nchannels), $
                         lb:INDGEN(nchannels), $
                         ub:INDGEN(nchannels), $
                         botchan:bands.botchan,  topchan:bands.topchan, editable:1 }

             ENDIF ELSE BEGIN

               ncommon   = nbands < N_ELEMENTS(bands.lb)
               use_bands = bands.use_bands(0:ncommon-1)
               lb        = bands.lb(0:ncommon-1)
               ub        = bands.ub(0:ncommon-1)

               bands = { nbands:nbands, use_bands:REPLICATE(1,nbands), $
                         lb:INTARR(nbands), $
                         ub:INTARR(nbands), $
                         botchan:bands.botchan,  topchan:bands.topchan, editable:1 }

               bands.use_bands = use_bands
               bands.lb        = lb
               bands.ub        = ub

             ENDELSE

             ; stop updates to screen
             ; this prevents widget jiggling around, saves time and most importantly
             ; stop extra space being created when destroy and recreate child widgets

             WIDGET_CONTROL, id_bands_parent, UPDATE=0

             ; destroy old bands

             WIDGET_CONTROL, id_bands_child, /DESTROY

             ; create new bands

             ies_do_bands

             WIDGET_CONTROL, id_bands_parent, UPDATE=1

           END

   'LB'  : BEGIN

              ch = FIX(STRMID(uvalue,2,STRLEN(uvalue)-2))

              WIDGET_CONTROL, event.id, GET_VALUE = s
              bands.lb(ch) = LONG(s(0))

              bands.ub(ch) = bands.ub(ch) > bands.lb(ch)
              WIDGET_CONTROL, id_ub(ch), SET_VALUE = bands.ub(ch)

            END

   'UB'  : BEGIN

              ch = FIX(STRMID(uvalue,2,STRLEN(uvalue)-2))

              WIDGET_CONTROL, event.id, GET_VALUE = s
              bands.ub(ch) = LONG(s(0))

              bands.lb(ch) = bands.lb(ch) < bands.ub(ch)
              WIDGET_CONTROL, id_lb(ch), SET_VALUE = bands.lb(ch)

            END

    'US'  : BEGIN

              ch = FIX(STRMID(uvalue,2,STRLEN(uvalue)-2))

              bands.use_bands(ch) = 1 - bands.use_bands(ch)

            END

    'TC' : BEGIN
            WIDGET_CONTROL, id_topchan, GET_VALUE=s
            bands.topchan = LONG(s(0))
          END

   'BC' : BEGIN
            WIDGET_CONTROL, id_botchan, GET_VALUE=s
            bands.botchan = LONG(s(0))
          END

   'DN' : BEGIN

            IF bands.nbands EQ 0 THEN BEGIN

              WIDGET_CONTROL, id_botchan, GET_VALUE=s
              bands.botchan = LONG(s(0))

              WIDGET_CONTROL, id_topchan, GET_VALUE=s
              bands.topchan = LONG(s(0))

            ENDIF

            WIDGET_CONTROL, event.top, /DESTROY

          END

    ELSE  : RETURN

  ENDCASE

END
;-----------------------------------------------------------------------------

FUNCTION ies_bands, GROUP=group, GET_CURRENT=get_current, SET_CURRENT=set_current

   COMMON ies_bands_block, id_nbands, id_bands_parent, id_bands_child, id_use, id_lb, id_ub, id_botchan, id_topchan, $
                           nchannels, bands

  ; check if current settings required

  IF KEYWORD_SET(get_current) THEN RETURN, bands

  ; set up values

  ; may be first entry with specified no. of bands : set_current will have 2 elements
  ;   bands will not have been set up
  ;   input_nbands = 0 signifies that all channels used
  ;     set up to not use end channel unless only one channel
  ;   input_nbands = -1 signifies that all channels used
  ;     set up to use end channel
  ;     do not allow nbands to be changed
  ;   input_nbands can be greater than input_nchannels
  ;   set bands = channels up to maximum channels or bands
  ; may be subsequent entry with specified no. of bands : set_current will have 2 elements
  ;   bands will have been set up
  ;   input_nbands = 0 signifies that all channels now used
  ;     want to keep definitions of top and bottom channels if no of channel the same
  ;     set up to not use end channel unless only one channel
  ;   input_nbands = -1 signifies that all channels used
  ;     set up to use end channel
  ;     do not allow nbands to be changed
  ;   input_nbands can be greater than input_nchannels
  ;   if changing the no. of channels then set bands=channels up to maximum
  ;   if no. of channels the same set previous bands as far as possible
  ; may be new data file : set_current will have 1 element
  ;   no. of channels may change
  ;   if changing the no. of channels then set bands=channels up to maximum
  ;   if no. of channels the same set previous bands
  ; may be call to bands from within getset
  ;   no. of channels as previous
  ;   use previous bands

  IF N_ELEMENTS(set_current) EQ 2 THEN BEGIN

    input_nchannels = set_current[0]
    input_nbands    = set_current[1]

    IF N_ELEMENTS(bands) EQ 0 THEN BEGIN

      IF input_nbands EQ -1 THEN BEGIN

        bands = { nbands:0, use_bands:REPLICATE(1,input_nchannels), $
                  lb:INDGEN(input_nchannels), $
                  ub:INDGEN(input_nchannels), $
                  botchan:0,  topchan:(input_nchannels-1), editable:1 }

      ENDIF ELSE IF input_nbands EQ 0 THEN BEGIN

        bands = { nbands:input_nbands, use_bands:REPLICATE(1,input_nchannels), $
                  lb:INDGEN(input_nchannels), $
                  ub:INDGEN(input_nchannels), $
                  botchan:0,  topchan:(input_nchannels-2)>0, editable:1 }

      ENDIF ELSE BEGIN

        bands = { nbands:input_nbands, use_bands:REPLICATE(1,input_nbands), $
                  lb:INDGEN(input_nbands) < (input_nchannels-1), $
                  ub:INDGEN(input_nbands) < (input_nchannels-1), $
                  botchan:0,  topchan:(input_nchannels-2)>0, editable:1 }

      ENDELSE

    ENDIF ELSE BEGIN

      IF input_nbands EQ -1 THEN BEGIN

        bands = { nbands:0, use_bands:REPLICATE(1,input_nchannels), $
                  lb:INDGEN(input_nchannels), $
                  ub:INDGEN(input_nchannels), $
                  botchan:0,  topchan:(input_nchannels-1), editable:1 }

      ENDIF ELSE IF input_nbands EQ 0 THEN BEGIN

        IF input_nchannels EQ nchannels THEN BEGIN

          bands = { nbands:input_nbands, use_bands:REPLICATE(1,input_nchannels), $
                    lb:INDGEN(input_nchannels), $
                    ub:INDGEN(input_nchannels), $
                    botchan:bands.botchan,  topchan:bands.topchan, editable:1 }

        ENDIF ELSE BEGIN

          bands = { nbands:input_nbands, use_bands:REPLICATE(1,input_nchannels), $
                    lb:INDGEN(input_nchannels), $
                    ub:INDGEN(input_nchannels), $
                    botchan:0,  topchan:(input_nchannels-2)>0, editable:1 }

        ENDELSE

      ENDIF ELSE BEGIN

        IF input_nchannels EQ nchannels THEN BEGIN

           ncommon   = input_nbands < N_ELEMENTS(bands.lb)
           use_bands = bands.use_bands(0:ncommon-1)
           lb        = bands.lb(0:ncommon-1)
           ub        = bands.ub(0:ncommon-1)

           bands = { nbands:input_nbands, use_bands:REPLICATE(1,input_nbands), $
                     lb:INTARR(input_nbands), $
                     ub:INTARR(input_nbands), $
                     botchan:bands.botchan,  topchan:bands.topchan, editable:1 }

           bands.use_bands = use_bands
           bands.lb        = lb
           bands.ub        = ub

        ENDIF ELSE BEGIN

          bands = { nbands:input_nbands, use_bands:REPLICATE(1,input_nbands), $
                    lb:INDGEN(input_nbands) < (input_nchannels-1), $
                    ub:INDGEN(input_nbands) < (input_nchannels-1), $
                    botchan:0,  topchan:(input_nchannels-2)>0, editable:1 }

        ENDELSE

      ENDELSE

    ENDELSE

    nchannels = input_nchannels

    RETURN, bands

  ENDIF ELSE IF N_ELEMENTS(set_current) EQ 1 THEN BEGIN

    input_nchannels = set_current(0)

    IF input_nchannels NE nchannels THEN BEGIN

      IF bands.nbands EQ 0 THEN BEGIN

        bands = { nbands:0, use_bands:REPLICATE(1,input_nchannels), $
                  lb:INDGEN(input_nchannels), $
                  ub:INDGEN(input_nchannels), $
                  botchan:0,  topchan:(input_nchannels-2)>0, editable:1 }

      ENDIF ELSE BEGIN

        bands = { nbands:3, use_bands:REPLICATE(1,3), $
                  lb:INDGEN(3) < (input_nchannels-1), $
                  ub:INDGEN(3) < (input_nchannels-1), $
                  botchan:bands.botchan,  topchan:bands.topchan, editable:1 }

      ENDELSE

    ENDIF

    nchannels = input_nchannels

    RETURN, bands

  ENDIF

  ; check if already running

  IF XREGISTERED ("ies_bands") THEN RETURN, bands

  ; ***********************************************************

  ; set up main bands widget

  base = WIDGET_BASE ( TITLE = 'SELECT ENERGY BANDS', /COLUMN)
;  base = WIDGET_BASE ( TITLE = 'SELECT ENERGY BANDS', /COLUMN, GROUP_LEADER=group, /MODAL)

  ; ***********************************************************

  ; set up text widget

  row    = WIDGET_BASE  ( base, /ROW, /FRAME )

  button = WIDGET_BUTTON(row, UVALUE='DN', VALUE='DONE')

  IF bands.editable THEN BEGIN

    label  = WIDGET_LABEL ( row, VALUE='NUMBER OF ENERGY BANDS :')

    ; set up widget ID of nbands text widget

    id_nbands = WIDGET_TEXT  ( row, VALUE=STRTRIM(bands.nbands,2), UVALUE='NB', /EDITABLE, XSIZE=6, /FRAME)

  ENDIF

  ; if nchannels = 1 then dont display any options

  IF nchannels NE 1 THEN BEGIN

    ; set up widget ID of band displays

    id_bands_parent  = WIDGET_BASE  ( base, /COLUMN, /FRAME)

    ; do rest of widgets

    ies_do_bands

    ; ***********************************************************

    WIDGET_CONTROL, base, /REALIZE

    ; ***********************************************************

    XMANAGER, "ies_bands", base,GROUP_LEADER=group, /MODAL
;    XMANAGER, "ies_bands", base

  ENDIF

  ; return

  RETURN, bands

END
