;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_CAL
;
; Purpose     :	Calibrates data and reverses calibration.
;
; Explanation :	Applies count rate calibration depending on LUT mode, converting to flux and reverses calibration.
;
; Use         : < ies_cal, lut_list, count_cal, output_array >
;
; Inputs      : lut_list      : STRUCTURE ARRAY containing LUT run info.
;               count_cal     : FLTARR(nluts) calibration factor for each LUT.
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 26/9/96
;                 Adapted from ies_calibrate.
;
; Modified    :	Version 0.1, MKC, 4/10/96
;                 Added RAT keyword.
;               Version 0.2, 13/11/96
;                 Separate ies div and ies rat.
;             	Version 0.3, 15/4/97
;                 Allowed reverse change.
;             	Version 0.4, 29/4/97
;                 Corrected IPS/HIST wrong calibration factors.
;             	Version 0.5, 18/8/97
;                 Removed fudge for IPS/HIST wrong calibration factors.
;               Version 0.6, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.7, 20/07/01
;                 Used ies runs.
;               Version 0.8, 24/10/03
;                 Changed to lut_list argument.
;
; Version     :	Version 0.8, 24/10/03
;-
;**********************************************************

PRO ies_cal, lut_list, count_cal, output_array, REVERSE=reverse

  ; apply count calibration i.e convert from counts to counts/sr/cm2

  ; loop through runs of each lut

  FOR l = 0L, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = lut_list[l].lut

    ; apply calibration factor

    IF KEYWORD_SET(reverse) THEN BEGIN

      ; reverse calibration for each energy and each detector for this LUT

      output_array(*,*,*,lut_list[l].st:lut_list[l].en) = output_array(*,*,*,lut_list[l].st:lut_list[l].en) / count_cal(lut)

    ENDIF ELSE BEGIN

      ; do calibration for each energy and each detector for this LUT

      output_array(*,*,*,lut_list[l].st:lut_list[l].en) = output_array(*,*,*,lut_list[l].st:lut_list[l].en) * count_cal(lut)

    ENDELSE

  ENDFOR

END
