;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_CALIBRATE
;
; Purpose     :	Applies various calibrations to data.
;
; Explanation :	Allows conversion from counts per energy channel to counts per keV.
;               Applies count rate calibration depending on LUT mode, converting to flux.
;               Converts counts to count rate.
;               Applies clean process to each data sector.
;               Clean implies shift and subtract.
;
; Use         : < res = ies_calibrate() >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : res = Flag indicating whether processing stopped or not.
;               Outputs new data values through COMMON block.
;
; Opt. Outputs:	None.
;
; Keywords    : QUIET = suppress diagnostic output to window.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, 19/9/96
;                 Added SECTOR vs DETECTOR plot option.
;                 Changed bins to keV.
;                 Added CLEAN button.
;                 Extended ies_draw_widget_block.
;                 Added ies_cal and ies_clean.
;                 Moved setting of noutputs and ptitle from ies_do_prelims.
;                 Moved fix of data within min and max.
;                 Added RAT keyword to ies_cal.
;                 Changed to a function.
;               Version 0.2, 11/11/96
;                 Added pedestal subtraction.
;                 Added routine ies docal.
;                 Added div, rat and cal tags.
;               Version 0.3, 11/12/96
;                 Added quiet keyword.
;               Version 0.4, 16/1/97
;                 Renamed ies_clean to ies shift and converted to procedure.
;                 Renamed output_info.clean tag to .shift.
;                 Modified ies docal to include pedestal subtraction and position calculations
;                 and pedestal shift.
;                 Added PROFILES routine.
;                 Modified ENERGY vs SECTORS format to be 8 OR LESS sectors.
;                 Added full_only to output_info.
;                 Changed .ndetectors tag.
;                 Changed definition of output_info.full_sector and added Sum option.
;                 Changed detector sector labelling.
;                 Changed ANGLE/ANGLE labelling.
;               Version 0.5, 15/4/97
;                 Allowed CAL, DIV and RAT to be applied to clean data.
;                 Corrected processing of sector plot fluxes.
;               Version 0.6, 29/8/97
;                 Added setting of elo and ehi.
;               Version 0.7, 24/11/97
;                 Removed sector 0 from IPS spin-average.
;                 Corrected for missing detectors when small pixels used in IES data.
;               Version 0.8, 7/4/98
;                 Added spin average to detector vs sector plot.
;               Version 0.9, 26/8/98
;                 Made small pixel check more explicit.
;               Version 1.0, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Moved setting of band_labels.
;                 Changed values of noutputs -- now no. of windows other than full window.
;               Version 1.1, 22/2/99
;                 Added ies_get_hilo.
;                 Fixed bug in detector vs sector spin_average display.
;               Version 1.2, 17/3/99
;                 Fixed so that hilo works for channels as well as energy bins.
;               Version 1.3, 11/5/99
;                 Removed EARTH and SUN keywords from ies average.
;               Version 1.4, 13/12/99
;                 Added WIN display type.
;               Version 1.5, 20/08/00
;                 Deal with RAPID data.
;               Version 1.6, 1/9/00
;                 Changed so that uses output_info.minimum and maximum without ABS
;               Version 1.7, 7/9/00
;                 Fixed bug where header.nbands used instead of nbands -- causes
;                 problem if shift used.
;               Version 1.8, 03/05/01
;                 Fixed bug for POLAR IIMS I3DD detectors.
;               Version 1.9, 09/07/02
;                 Removed TIMELINE and ANGLE/ANGLE restriction of data betwen min and max.
;               Version 2.0, 07/02/03
;                 Changed so that pitch angle only goes from 0-180 degrees
;                 Change pitch angle bins to 1 degree rather than 2 degrees.
;                 Added OVERLAY plot.
;                 Ensured output_chans is FLTARR.
;               Version 2.1, 26/09/03
;                 Added some comments about full_sector, full_detector.
;                 Changed to variable pitch angle bins.
;                 Used new routine for instrument test.
;               Version 2.2, 20/10/03
;                 Used new routine ies get pitch angle.
;               Version 2.3, 06/12/03
;                 Used routine cep small pixel.
;
; Version     :	Version 2.3, 06/12/03
;-
;**********************************************************

FUNCTION ies_calibrate, QUIET=quiet

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  IF NOT KEYWORD_SET(quiet) THEN BEGIN

    ; put up info widget

    newline = STRING(10B)

    id_ok = 0
    IF N_ELEMENTS(ids) GT 0 THEN IF WIDGET_INFO(ids.infoid, /VALID_ID) THEN id_ok=1

    IF id_ok THEN BEGIN

      temp_infoid = ids.infoid
      temp_logoid = ids.logoid

    ENDIF ELSE BEGIN

      newline = ''
      base    = WIDGET_BASE ( TITLE='PLEASE WAIT ...')

      row = WIDGET_BASE ( base, /ROW)

      ; set up logo

      IF !D.NAME EQ 'X' OR !D.NAME EQ 'WIN' THEN BEGIN

        logo   = ies_logo()
        dims   = SIZE(logo)
        temp_logoid = WIDGET_DRAW ( row, XSIZE=dims(1), YSIZE=dims(2), UVALUE='LOGO', RETAIN=2, /BUTTON_EVENTS)

      ENDIF ELSE temp_logoid = 0L

      ; set up text widget

      temp_infoid = WIDGET_TEXT ( row, VALUE='', XSIZE=60, YSIZE=6)

      ; realize widget

      WIDGET_CONTROL, base, /REALIZE

      ; draw logo if in x windows or win

      IF !D.NAME EQ 'X' OR !D.NAME EQ 'WIN' THEN BEGIN

        WIDGET_CONTROL, temp_logoid, GET_VALUE=win

        WSET, win

        TV, logo

      ENDIF

    ENDELSE

  ENDIF

  IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Calibrate'

  ; process extracted data

  IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Processing data', /APPEND

  ; calibrate data

  ies_docal, header, data, output_array, output_chans, nbands, output_info

  ;  Note that output_array here is not actually count rate but count rate / nsectors.

  IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Formatting data', /APPEND

  output_info.ysamples = nbands

  IF output_info.procname EQ 'SPECTRAL' OR output_info.procname EQ 'LINEPLOT' $
    OR output_info.procname EQ 'PROFILES' OR output_info.procname EQ 'OVERLAY' THEN BEGIN

    ; check data format

    IF output_info.format EQ 'ENERGY vs DETECTOR' THEN BEGIN

      ; fix noutputs

      IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = header.ndetectors

      ; fix ptitle

      output_info.ptitle(0:N_ELEMENTS(header.ptitle)-1) = 'Dt' + header.ptitle

      ; check if spin averaged

      IF output_info.full_sector GT 0 THEN BEGIN

        ; extract sector

        output_array = REFORM ( output_array(output_info.full_sector-1, *, *, *), header.ndetectors, nbands, header.npoints )

        ; add to ptitle

        output_info.ptitle = output_info.ptitle + ' Sc' + STRTRIM(output_info.full_sector-1,2)

      ENDIF ELSE BEGIN

        ; spin average required

        IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Summing over sectors', /APPEND

        output_array = ies_average ( output_array, output_info, header, data, /SPIN_AVERAGE)

        ; reform output array

        output_array = REFORM (output_array, header.ndetectors, nbands, header.npoints)

        ; add to ptitle

        output_info.ptitle = output_info.ptitle + ' Sp av'

      ENDELSE

      ; check if detector average required

      IF output_info.full_detector EQ header.ndetectors THEN BEGIN

        ; get detector average

        output_array = REFORM (output_array, 1, header.ndetectors, nbands, header.npoints)

        temp_array = ies_average ( output_array, output_info, header, data, /DETECTOR_AVERAGE)

        ; combine arrays

        output_array = REFORM (output_array, header.ndetectors, nbands, header.npoints)
        temp_array = REFORM (temp_array, 1, nbands, header.npoints)

        output_array = [output_array, temp_array]

        ; define output_chans for detector sum

        output_chans = output_chans(*,*,[INDGEN(header.ndetectors),0],*)

        ; define ptitle for detector sum

        output_info.ptitle[header.ndetectors] = 'Dt av'
        IF output_info.full_sector GT 0 THEN output_info.ptitle[header.ndetectors] = output_info.ptitle[header.ndetectors] + ' Sp av'

      ENDIF

    ENDIF ELSE IF output_info.format EQ 'ENERGY vs SECTOR' THEN BEGIN

      ; select single detector for multi-sector display
      ; use full detector
      ; use 8 or less sectors
      ; get array of form (nsectors, nbands, npoints)

      IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for multi-sector display', /APPEND

      IF header.nsectors GT 8 THEN sector_list = (INDGEN(8)*header.nsectors)/8 $
                              ELSE sector_list = INDGEN(header.nsectors)

      ; extract sectors

      output_array = REFORM (output_array(sector_list,*,*,*), N_ELEMENTS(sector_list), header.ndetectors, nbands, header.npoints)

      ; set ptitle

      output_info.ptitle(0:N_ELEMENTS(sector_list)-1) = 'Sc' + STRTRIM(sector_list,2)

      ; check if detector averaged

      IF output_info.full_detector LT header.ndetectors THEN BEGIN

        ; extract full detector

        output_array = REFORM (output_array(*,output_info.full_detector,*,*), N_ELEMENTS(sector_list), nbands, header.npoints)

        ; add to ptitle

        output_info.ptitle = output_info.ptitle + ' Dt'+header.ptitle(output_info.full_detector)

      ENDIF ELSE BEGIN

        ; detector average required

        IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Summing over detectors', /APPEND

        output_array = ies_average ( output_array, output_info, header, data, /DETECTOR_AVERAGE)

        ; reform output array

        output_array = REFORM (output_array, N_ELEMENTS(sector_list), nbands, header.npoints)

        ; add to ptitle

        output_info.ptitle = output_info.ptitle + ' Dt av'

      ENDELSE

      ; define positions for all sectors

      output_chans = output_chans(*,*,REPLICATE(0,N_ELEMENTS(sector_list)),*)

      ; fix noutputs and ptitle

      IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = N_ELEMENTS(sector_list)

    ENDIF ELSE IF output_info.format EQ 'SECTOR vs DETECTOR' THEN BEGIN

      ; select single energy band for multi-detector display
      ; get array of form (ndetectors, nsectors, npoints)

      IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for sector vs detector display', /APPEND

      ; check if detector averaged

      output_info.full_detector = output_info.full_detector < (header.ndetectors-1)

      ; select energy band and reform

      output_array = REFORM (output_array[*,*,output_info.full_band,*], header.nsectors,header.ndetectors*header.npoints)

      ; normalize to average counts

  ;    output_array = header.nsectors*output_array / (REPLICATE(1,header.nsectors)#TOTAL(output_array,1))

      ; transpose and reform output array  -> [ndetectors,nsectors,npoints]

      output_array = TRANSPOSE ( REFORM (output_array, header.nsectors, header.ndetectors, header.npoints), $
                    [1,0,2] )

      ; define dummy positions for all LUTS and all detectors

      output_chans = FLTARR( 2, header.nsectors,  header.ndetectors, header.nluts)

      FOR s = 0, header.nsectors-1 DO BEGIN

        output_chans(0,s,*,*) =  (360.0/header.nsectors) * s
        output_chans(1,s,*,*) =  (360.0/header.nsectors) * (s+1)

      ENDFOR

      output_info.ysamples = header.nsectors

      ; fix noutputs and ptitle

      IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = header.ndetectors

      output_info.ptitle(0:N_ELEMENTS(header.ptitle)-1) = 'Dt' + header.ptitle

      ; NB full_sector is not used for SECTOR vs DETECTOR

    ENDIF ELSE IF output_info.format EQ 'SECTOR vs DETECTOR3' THEN BEGIN

      ; select single detector for multi-energy band display
      ; get array of form (nbands, nsectors, npoints)

      IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for sector vs detector display', /APPEND

      ; check if detector averaged

      output_info.full_detector = output_info.full_detector < (header.ndetectors-1)

      ; select detector and reform

      output_array = REFORM (output_array(*,output_info.full_detector,*,*), header.nsectors,nbands*header.npoints)

      ; normalize to average counts

      output_array = header.nsectors*output_array / (REPLICATE(1,header.nsectors)#TOTAL(output_array,1))

      ; transpose and reform output array -> [nbands,nsectors,npoints]

      output_array = TRANSPOSE ( REFORM (output_array, header.nsectors, nbands, header.npoints), $
                    [1,0,2] )

      ; define dummy positions for all LUTS and all energies

      output_chans = FLTARR( 2, header.nsectors,  nbands, header.nluts)

      FOR s = 0, header.nsectors-1 DO BEGIN

        output_chans(0,s,*,*) =  (360.0/header.nsectors) * s
        output_chans(1,s,*,*) =  (360.0/header.nsectors) * (s+1)

      ENDFOR

      output_info.ysamples = header.nsectors

      ; fix noutputs and ptitle

      IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = nbands

      output_info.ptitle(0:nbands-1) = 'Energy '+STRING(INDGEN(nbands))

      ; NB full_sector is not used for SECTOR vs DETECTOR

    ENDIF ELSE IF output_info.format EQ 'DETECTOR vs SECTOR' THEN BEGIN

      ; select single energy band for multi-sector display
      ; get array of form (nsectors, ndetectors, npoints)
      ; use 8 or less sectors
      ; if full_sector = 0 then spin average

      IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for detector vs sector display', /APPEND

      ; check if spin averaged

      IF output_info.full_sector GT 0 THEN BEGIN

        sector_list = (INDGEN(header.nsectors<5)+output_info.full_sector-1) MOD header.nsectors

        ; fix noutputs and ptitle

        IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = N_ELEMENTS(sector_list)

        output_info.ptitle(0:N_ELEMENTS(sector_list)-1) = 'Sc' + STRTRIM(sector_list,2)

        ; reform output array

        output_array = REFORM (output_array(sector_list,*,output_info.full_band,*), N_ELEMENTS(sector_list), header.ndetectors, header.npoints)

        ; define dummy positions for all LUTS and all sectors

        output_chans = FLTARR( 2, header.ndetectors, N_ELEMENTS(sector_list), header.nluts)

      ENDIF ELSE BEGIN

        ; spin average required

        IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Summing over sectors', /APPEND

        output_array = ies_average( output_array[*,*,output_info.full_band,*], output_info, header, data, /SPIN_AVERAGE)

        ; reform output array

        output_array = REFORM (output_array, 1, header.ndetectors, header.npoints)

        ; fix noutputs and ptitle

        IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = 1

        output_info.ptitle(0:0) = 'SC av'

        ; define dummy positions for all LUTS and all sectors

        output_chans = FLTARR( 2, header.ndetectors, 1, header.nluts)

      ENDELSE

      ; do large pixels
      ; NB dummy pixels must have width at least one but are overwritten by lower energy channels
      ; 10 detectors for IPS

      IF ies_instrument(header.datatype,/I3DD) AND ies_instrument(header.datatype,/RAPID) THEN BEGIN

        start_angles = [  0.0,  15.0, 30.0, 45.0, 60.0, 75.0, 90.0,105.0,120.0,135.0,150.0,165.0]
        stop_angles  = [  15.0, 30.0, 45.0, 60.0, 75.0, 90.0,105.0,120.0,135.0,150.0,165.0,180.0]

      ENDIF ELSE BEGIN

        start_angles = [  0.0,  20.0, 40.0, 60.0, 80.0,100.0,120.0,140.0,160.0,160.0]
        stop_angles  = [  20.0, 40.0, 60.0, 80.0,100.0,120.0,140.0,160.0,180.0,180.0]

      ENDELSE

      FOR d = 0, header.ndetectors-1 DO BEGIN

        output_chans(0,d,*,*) =  start_angles(header.detectors(d)-1)
        output_chans(1,d,*,*) =  stop_angles(header.detectors(d)-1)

      ENDFOR

      ; adjust for small pixels

      IF ies_instrument(header.datatype,/IES) AND NOT ies_instrument(header.datatype,/RAPID) THEN BEGIN

        ; no lut map for IPS or HIST and lut map = 1 so count = 0 except for IES but do check anyway

        list = WHERE( cep_small_pixel(header), count)

        IF count GT 0 THEN BEGIN

          ; NB dummy pixels must have width at least one but are overwritten by lower energy channels
          ; -- not if not present

          start_angles = [  0.0, 29.0, 30.0, 60.0, 89.0, 90.0,120.0,149.0,150.0]
          stop_angles  = [ 30.0, 30.0, 60.0, 90.0, 90.0,120.0,150.0,150.0,180.0]

          FOR d = 0, header.ndetectors-1 DO BEGIN

            output_chans(0,d,*,list) =  start_angles(header.detectors(d)-1)
            output_chans(1,d,*,list) =  stop_angles(header.detectors(d)-1)

          ENDFOR

        ENDIF

      ENDIF

      output_info.ysamples = header.ndetectors

      ; NB full_detector is not used for DETECTOR vs SECTOR

    ENDIF ELSE IF output_info.format EQ 'PITCH ANGLE vs ENERGY' THEN BEGIN

      ; get array of form (npitchs, nenergies, npoints)

      ; get pitch angle vector

      IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for pitch angle vs energy display', /APPEND

      ; set up output array -> [nbands,npitchs,npoints]

      output_array = ies_get_pitch_angle ( output_array, output_info, header, data)

      ; set band ptitles
      ; NB none of this makes much sense if energy bands are not the same for each detector and lut
      ; so can use 0th elements

      IF output_info.button_values[0] THEN BEGIN

        ctop = (N_ELEMENTS(output_chans[0,*,0,0]) < N_ELEMENTS(output_info.band_labels)) - 1

        output_info.ptitle = 'Energy '+$
                      STRTRIM(STRING(FORMAT='(I)',output_chans(0,*,0,0)),2) + ':' +$
                      STRTRIM(STRING(FORMAT='(I)',output_chans(1,*,0,0)),2) + ' keV'

      ENDIF ELSE BEGIN

        output_info.ptitle = 'Energy band '+STRTRIM(INDGEN(nbands),2)

      ENDELSE

      ; define dummy positions for all LUTS and all detectors

      output_chans = FLTARR( 2, output_info.npitchs,  nbands, header.nluts)

      FOR s = 0, output_info.npitchs-1 DO BEGIN

        output_chans(0,s,*,*) =  (180.0/output_info.npitchs) * s
        output_chans(1,s,*,*) =  (180.0/output_info.npitchs) * (s+1)

      ENDFOR

      output_info.ysamples = output_info.npitchs

      ; fix noutputs and ptitle, max 10 bands

      IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = nbands < 10

      ; NB full_detector, full_sector are not used for PITCH ANGLE vs ENERGY

    ENDIF ELSE IF output_info.format EQ 'GYRORADIUS vs ENERGY' THEN BEGIN

      ; get array of form (nradii, nenergies, npoints)

      ; set up no. of gyrordaii bins

      nradii = 500

      ; get gyroradii vector

      IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for gyroradii vs energy display', /APPEND

      ; set up output array -> [nbands,nradii,npoints]

      output_array = ies_do_gyroradii( nradii, output_array, output_chans, output_info, header, data)

      ; set band ptitles
      ; NB none of this makes much sense if energy dbands are not the same for each detector and lut
      ; so can use 0th elements

      IF output_info.button_values[0] THEN BEGIN

        ctop = (N_ELEMENTS(output_chans[0,*,0,0]) < N_ELEMENTS(output_info.band_labels)) - 1

        output_info.ptitle = 'Energy '+$
                      STRTRIM(STRING(FORMAT='(I)',output_chans(0,*,0,0)),2) + ':' +$
                      STRTRIM(STRING(FORMAT='(I)',output_chans(1,*,0,0)),2) + ' keV'

      ENDIF ELSE BEGIN

        output_info.ptitle = 'Energy band '+STRTRIM(INDGEN(nbands),2)

      ENDELSE

      ; define dummy positions for all LUTS and all detectors

      output_chans = FLTARR( 2, nradii,  nbands, header.nluts)

      FOR s = 0, nradii-1 DO BEGIN

        ; create 7 decades starting at 0.001 km

        output_chans(0,s,*,*) =  0.001*10^(7.0/nradii * s)
        output_chans(1,s,*,*) =  0.001*10^(7.0/nradii * (s+1))

      ENDFOR

      output_info.ysamples = nradii

      ; fix noutputs and ptitle, max 15 bands

      IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = nbands < 15

      ; NB full_detector, full_sector are not used for GYRORADIUS vs ENERGY

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN

    IF NOT KEYWORD_SET(quiet) THEN $
      WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for sector vs detector vs energy display', /APPEND

    ; fix noutputs and ptitle

    IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = nbands

    output_info.ptitle(0:N_ELEMENTS(header.ptitle)-1) = 'Dt' + header.ptitle

  ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN

    ; select single energy band for sector-detector display
    ; get array of form (nsectors, ndetectors, npoints)
    ; NB very similar to SECTOR vs DETECTOR

    IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Selecting data for sector vs detector display', /APPEND

    ; reform output array

    output_array = REFORM (output_array(*,*,output_info.full_band,*), header.nsectors, header.ndetectors, header.npoints)

    ; define dummy positions for all LUTS and all detectors

    output_chans = FLTARR( 2, header.nsectors,  header.ndetectors, header.nluts)

    FOR s = 0, header.nsectors-1 DO BEGIN

      output_chans(0,s,*,*) =  (360.0/header.nsectors) * s
      output_chans(1,s,*,*) =  (360.0/header.nsectors) * (s+1)

    ENDFOR

    output_info.ysamples = header.nsectors

    ; fix noutputs and ptitle

    IF output_info.full_only THEN output_info.noutputs = 0 ELSE output_info.noutputs = 16

    output_info.ptitle(0:N_ELEMENTS(header.ptitle)-1) = 'Dt' + header.ptitle

    ; cannot use detector average

    output_info.full_detector = output_info.full_detector < (header.ndetectors-1)

  ENDIF ELSE MESSAGE, 'Invalid process name', /TRACEBACK

  ; set up minimum and maximum
  ; ignoring bad counts and times

  IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Calculating minimum and maximum', /APPEND

  ; get ids for minimum and maximum widgets

  IF N_ELEMENTS(ids) EQ 0 THEN BEGIN

    ; widgets not defined yet set to invalid IDs

    minid = 0L
    maxid = 0L
    eloid = 0L
    ehiid = 0L

  ENDIF ELSE BEGIN

    minid = ids.minid
    maxid = ids.maxid
    eloid = ids.eloid
    ehiid = ids.ehiid

  ENDELSE

  ; do min max calculation
  ; NB only calculates if min or max invalid

  ies_get_minmax, output_array, data.flag, minid, maxid, output_info, LOG=output_info.log

  ; check if hi/lo set

  ies_get_hilo, output_chans, eloid, ehiid, output_info, $
    SHOWCH=output_info.button_values[0], ELOG=output_info.button_values[3]

  ; processing completed

  IF NOT KEYWORD_SET(quiet) THEN WIDGET_CONTROL, temp_infoid, SET_VALUE=newline+'Calibrate finished', /APPEND

  IF N_ELEMENTS(base) GT 0 THEN WIDGET_CONTROL, base, /DESTROY

  RETURN, 0

END
