;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_CONCATENATE
;
; Purpose     :	Concatenates intermediate format data structures together.
;
; Explanation :	Concatenates intermediate format data structures together.
;
; Use         : < ies_concatenate, data1, data2, HEADER=header, ERROR=error >
;
; Inputs      : data1   : STRUCTURE ARRAY containing data.
;               data2   : STRUCTURE ARRAY containing data.
;
; Opt. Inputs : None.
;
; Outputs     : data1   : STRUCTURE ARRAY containing concatenated data.
;
; Opt. Outputs:	None.
;
; Keywords    : HEADER : updates intermediate format header structure.
;               ERROR  : STRING reports errors.
;               GROW   : FLAG indicating should grow data array before filling.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 26/9/96
;                 Adapted from ies_getset.
;
; Modified    :	Version 0.1, 13/02/02
;                 Added use of TEMPORARY. 
;                 Added GROW keyword.
;                 Changed behaviour slightly so that input always replaces overlapping 
;                 existing data.
;
; Version     :	Version 0.1, 13/02/02
;-
;**********************************************************

PRO ies_concatenate, data1, data2, HEADER=header, ERROR=error, GROW=grow

  ; check data format

  IF TAG_NAMES(data1,/STRUCTURE_NAME) NE '' AND $
     TAG_NAMES(data2,/STRUCTURE_NAME) NE TAG_NAMES(data1,/STRUCTURE_NAME) THEN BEGIN

    error = 'Incompatible data sets'

    RETURN

  ENDIF

  IF NOT KEYWORD_SET(header) AND KEYWORD_SET(grow) THEN MESSAGE, 'IMPLEMENTATION ERROR'

  ; combine data
  ; NB where two times are equal replace with one with following data

  IF KEYWORD_SET(grow) THEN BEGIN

    n1 = header.npoints

    n2 = N_ELEMENTS(data2)

    IF n1+n2 GT N_ELEMENTS(data1) THEN data1 = [data1,data1]

  ENDIF ELSE BEGIN

    n1 = N_ELEMENTS(data1)

    n2 = N_ELEMENTS(data2)

  ENDELSE

  IF ( data2[0].time GT data1[n1-1].time ) THEN BEGIN

    ; add input to end

    n = n1 + n2

    IF KEYWORD_SET(grow) THEN data1[n1:n-1] = data2 $
                         ELSE data1 = [TEMPORARY(data1[0:n1-1]), data2]

  ENDIF ELSE IF ( data2[n2-1].time LT data1[0].time ) THEN BEGIN

    ; add input to beginning

    data1 = [data2, TEMPORARY(data1[0:n1-1])]

    n = n1 + n2

  ENDIF ELSE IF ( data2[n2-1].time GE data1[n1-1].time ) THEN BEGIN

    ; input replaces overlapping part of existing data

    dummy_list = WHERE ( data1[0:n1-1].time LT data2[0].time, count)

    IF count EQ 0 THEN data1 = data2 ELSE data1 = [TEMPORARY(data1[0:count-1]), data2]

    n = count + n2

  ENDIF ELSE IF ( data2[0].time LE data1[0].time ) THEN BEGIN

    ; part of input precedes existing data
    ; overlapping part replaces existing data

    dummy_list = WHERE ( data1[0:n1-1].time GT data2[n2-1].time, count)

    IF count GT 0 THEN data1 = [data2,TEMPORARY(data1[n1-count:n1-1])]

    n = n2 + count

  ENDIF ELSE BEGIN

    ; input is fully enclosed by existing data

    dummy_list = WHERE ( data1[0:n1-1].time LT data2[0].time, count0)

    after_list = WHERE ( data1[0:n1-1].time GT data2[n2-1].time, count1)

    data1 = [data1[0:count0-1], data2, data1[n1-count1:n1-1]]

    n = count0 + n2 + count1

  ENDELSE

  IF KEYWORD_SET(header) THEN BEGIN

    ; reset header values

    header.npoints = n
    header.tstart  = data1(0).time
    header.tstop   = data1(n-1).time

  ENDIF

END
