;+
; Project     :	POLAR - CEPPAD
;
; Name        :	ies_data_structure
;
; Purpose     :	Returns structure array for data samples.
;
; Explanation : Data is stored as a header structure and a structure array containing the data samples.
;               The data times are stored as a FLTARR of seconds.
;               If KEYWORD <experiment> is set then structure is named. If not set then
;               the structure is anonymous.
;               If anonymous structure created then allow other info to be carried along with structure ie
;               magnetic field data.
;               Also allow other data to be carried along if named structure ie mvals. In this case the structure
;               name is altered to correspond.
;
; Use         : < struct = ies_data_structure ( no_of_points, no_of_detectors, no_of_energies, no_of_sectors, $
;                                               EXPERIMENT=experiment, MODE=mode, MF=mf, PA=pa ) >
;
; Inputs      : no_of_points    : LONG number of time samples.
;               no_of_detectors : LONG number of detectors at each sample
;               no_of_energies  : LONG number of energies at each sample
;               no_of_sectors   : LONG number of sectors at each sample
;
; Opt. Inputs : None.
;
; Outputs     : struct : Structure array for data samples.
;                 .time:DOUBLE TAI time for sample
;                 .data:FLTARR ( nsectors, ndetectors, nenergies ) data for sample
;                         Data is normalized so that has same total no of counts as raw data.
;                 .lut:BYTE LUT number for sample
;                 .flag:BYTE data quality flag for sample
;                     bit 0 : sample occurs at major frame (16 spins)
;                     bit 5 : invalid LUT number
;                     bit 6 : sample contains some invalid counts
;                     bit 7 : time for sample estimated.
;                     bit 8 : last sample in data set
;
; Opt. Outputs:	struct may have optional .mf tag.
;
; Keywords    : EXPERIMENT : STRING indicates should return particular named structure.
;               MODE       : STRING indicates should return particular named structure.
;               EARTH      : LOGICAL flag indicating that should create a .earth tag.
;               MF         : LOGICAL flag indicating that should create a .mf tag.
;               SUN        : LOGICAL flag indicating that should create a .sun tag.
;               MVALS      : LOGICAL flag indicating should add .mvals BYTARR(11) tag to data structure.
;               PA         : LOGICAL flag indicating that should create a .pa tag.
;               GSE        : LOGICAL flag indicating that should create a .gse_xyz tag.
;               GSM        : LOGICAL flag indicating that should create a .gsm_xyz tag.
;               SM         : LOGICAL flag indicating that should create a .sm_xyz tag.
;
; Calls       :	get_utc.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, 29/2/96
;                 Converted to CDS internal time format.
;               Version 0.2, 1/10/96
;                 Changed so that returns no. of sectors in data set rather than for expt.
;                 Added CLEAN_n modes.
;                 Added SPIN_n modes.
;                 Got rid of ndetectors, nenergies.
;                 no_of_detectors, no_of_luts, no_of_energies now set up externally.
;               Version 0.3, 16/4/97
;                 Disallowed no_of_detectors=0.
;               Version 0.4, 9/5/97
;                 Set .lut and .flag to LONG to allow compatability between DECs and SUNs.
;                 Data structure version V2
;                 Interestingly this actually seems to decrease file size !?
;               Version 0.5, 26/2/98
;                 Extended no of energies in clean data to 11 and changed to V3.
;               Version 0.6, 8/10/98
;                 Added MF and AT keywords.
;               Version 0.7, 11/2/00
;                 Added endtime to data structure version 4.
;               Version 0.8, 12/5/00
;                 Added mvals. Did not need to change version no.
;               Version 0.9, 08/02/02
;                 Changed so that mvals is not fixed.
;               Version 1.0, 13/03/02
;                 Added GSE_XYZ.
;               Version 1.1, 15/04/02
;                 Added GSM_XYZ and SM_XYZ.
;               Version 1.2, 06/01/03
;                 Removed GSM_XYZ and SM_XYZ.
;               Version 1.3, 08/02/03
;                 Added total magnetic field for MF tag.
;               Version 1.4, 14/04/03
;                 Added pitch angle tag.
;                 Added GSE_XYZ, GSM_XYZ, SM_XYZ.
;               Version 1.5, 18/04/03
;                 Allowed MF tag to be added to named structure.
;               Version 1.6, 08/10/03
;                 Changed mvals to logical flag.
;                 Changed structure version no. since have changed mvals to BYTARR(11) rather than INTARR(11)
;               Version 1.7, 1/11/03
;                 Changed AT and EPHEM keywords to EARTH and SUN directly.
;
; Version     :	Version 1.7, 1/11/03
;-
;**********************************************************

FUNCTION ies_data_structure, no_of_points, no_of_detectors, no_of_energies, no_of_sectors, $
         EXPERIMENT=experiment, MODE=mode, MF=mf, SUN=sun, EARTH=earth, MVALS=mvals, PA=pa, GSE=gse, GSM=gsm, SM=sm

   ; set structure name variable

  structure_name = ''

  ; deal with experiment and mode keywords

  IF KEYWORD_SET(experiment) THEN structure_name = 'data_' + experiment

  IF KEYWORD_SET(mode) THEN structure_name = structure_name + '_' + mode

  IF structure_name NE '' AND KEYWORD_SET(mvals) THEN structure_name = structure_name + '_mv'

  IF structure_name NE '' AND KEYWORD_SET(pa   ) THEN structure_name = structure_name + '_pa'

  IF structure_name NE '' AND KEYWORD_SET(gse  ) THEN structure_name = structure_name + '_gse'

  IF structure_name NE '' AND KEYWORD_SET(gsm  ) THEN structure_name = structure_name + '_gsm'

  IF structure_name NE '' AND KEYWORD_SET(sm   ) THEN structure_name = structure_name + '_sm'

  IF structure_name NE '' AND KEYWORD_SET(mf   ) THEN structure_name = structure_name + '_mf'

  IF structure_name NE '' THEN structure_name = structure_name + '_V5,'

  xtra = ''

  ; only create sun and earth tags in temporary unnamed data structure

  IF structure_name EQ '' AND KEYWORD_SET(sun) THEN xtra = xtra + ', sun:[0.0,0.0,0.0]'

  IF structure_name EQ '' AND KEYWORD_SET(earth) THEN xtra = xtra + ', earth:[0.0,0.0,0.0]'

  IF KEYWORD_SET(mvals) THEN xtra = xtra + ', mvals:BYTARR(11)'

  IF KEYWORD_SET(pa)    THEN xtra = xtra + ', pa:FLTARR(no_of_sectors,no_of_detectors)'

  IF KEYWORD_SET(gse)   THEN xtra = xtra + ', gse_xyz:FLTARR(3)'

  IF KEYWORD_SET(gsm)   THEN xtra = xtra + ', gsm_xyz:FLTARR(3)'

  IF KEYWORD_SET(sm )   THEN xtra = xtra + ', sm_xyz:FLTARR(3)'

  IF KEYWORD_SET(mf)    THEN xtra = xtra + ', mf:[0.0,0.0,0.0]'

  structure = 0

  ; return initialized structure
  ; NB structure name contains a ','

  IF EXECUTE ( 'structure={' + structure_name + $
     'time:0.0D0, endtime:0.0D0, data:FLTARR(no_of_sectors,no_of_detectors,no_of_energies), lut:0L, flag:0L' + $
     xtra + ' }' ) THEN $
     RETURN, REPLICATE ( structure, no_of_points)

  MESSAGE, 'Execute did not work', /TRACEBACK

END

