;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_DETECTOR_POLAR_ANGLE
;
; Purpose     :	Sets polar angle coverage of detector.
;
; Explanation :	Sets polar angle coverage of detector.
;
;               On POLAR the electron sensor numbering gives sensors from
;               anti-parallel to parallel to the SC spin axis for the small pixels and the opposite
;               way around for the large pixels. Similarly IPS detector
;               numbering as used in header.detectors gives detectors from anti-parallel
;               to parallel.
;
;               Believe that SAT table gives small pixel orientation so that detectors 0-8 are
;               anti-parallel to spin axis.
;
;               For CEPPAD/IES large pixels dpositions = 8->0 so that
;               CEPPAD/IES large detector 0 is parallel to spin axis.
;
;               For CEPPAD/IPS dpositions = header.detectors-1.
;               header.detectors indexes detectors from anti-parallel to parallel to spin axis.
;
;
; Use         : < ies_detector_polar_angle, header, lut, dlist, dcount, dsize, dpositions >
;
; Inputs      : header       : STRUCTURE giving header information
;
; Opt. Inputs : lut          : INT lut no indicating small or large pixel for CEPPAD
;
; Outputs     : dlist        : FLTARR(ndetectors) indexes position in data array and header.detectors ie detectors used
;               dcount       : INT no. of detectors used
;                              NB IPS background detector not used.
;               dsize        : INT no. of detectors spanning the 180 degrees of polar angle
;               dpositions   : INTARR(dcount) gives position of detector relative to anti-spin direction
;                              where 0->180 is 0->dsize-1
;                              Also indexes into mask
;                              dposition 0 is antiparallel to spin
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 23/10/98
;                 Adapted from ies_plot.
;
; Modified    :	Version 0.0, 07/02/03
;                 Adapted from ies plot img and ies pitch angle.
;                 Corrected bug if IPS background detector not selected.
;               Version 0.1, 26/09/03
;                 Used new routine for instrument test.
;               Version 0.2, 14/10/03
;                 Allowed lut map tag to be missing.
;                 .lut map is usually set up in ies extdata even if not
;                 present in IF file (ie for IPS). However, may also use
;                 routine directly on IF data set.
;               Version 0.3, 04/12/03
;                 Used cep small pixel.
;
; Version     :	Version 0.3, 04/12/03
;-
;**********************************************************

PRO ies_detector_polar_angle, header, lut, dlist, dcount, dsize, dpositions

  ; dposition 0 is antiparallel to spin

  ; header.detectors gives the position of each detector relative to spin or anti-spin direction
  ; header.ptitle gives the name of each detector
  ;  ;
  ; for IPS  :
  ;  detectors = [9,4,3,5,6,7,8,2,1,0] (cep reorg)
  ;  detectors = [10,9,4,3,5,6,7,8,2,1] (ies extdata)
  ;  ptitle = [ 'T1/C1/B1', 'C2', 'C5', 'C0', 'B5', 'B2', 'B0', 'T5', 'T2', 'T0'] (cep reorg)
  ;
  ; for IES  :
  ;  detectors = [1,2,3,4,5,6,7,8,9] (ies_header_structure)
  ;  ptitle = ['1','2','3','4','5','6','7','8','9'] (ies_header_structure)
  ;

  IF ies_instrument(header.datatype,/RAPID) THEN BEGIN

    ; header.detectors gives the position of each detector relative to the spin direction
    ; invert dpositions

    IF ies_instrument(header.datatype,/IES) THEN dsize = 9 ELSE $
    IF ies_instrument(header.datatype,/IIMS) THEN dsize = 12 ELSE dsize = 1

    dcount = header.ndetectors
    dlist = INDGEN(header.ndetectors)
    dpositions = header.ndetectors - header.detectors

  ENDIF ELSE IF ies_instrument(header.datatype,/IES) THEN BEGIN

    ; if IES then lut map tag should be present

    IF cep_small_pixel(header, LUT=lut) THEN BEGIN

      ; IES small pixels
      ; header.detectors gives the position of each small detector relative to the anti-spin direction

      ; header.detectors = 1->9
      ; select dlist = 0,2,3,5,6,8 (index into header.detectors)
      ; set dpositions = 0,1,2,3,4,5

      dlist = WHERE (((header.detectors-1) MOD 3) NE 1, dcount)
      dsize = 6
      IF dcount GT 0 THEN dpositions = (((header.detectors[dlist]-1)/3)*2 + ((header.detectors[dlist]-1) MOD 3)/2)

    ENDIF ELSE BEGIN

      ; NB IES detector numbering goes 1-9
      ; header.detectors gives position of detectors relative to spin axis for large detectors
      ; NB this really disagrees with the SAT tables though they are slightly ambiguous
      ; invert dpositions
      ; i.e set dpositions = [8,7,6,5,4,3,2,1,0] if all detectors present

      dlist = INDGEN(header.ndetectors)
      dcount = header.ndetectors
      dsize = 9
      dpositions = dsize-header.detectors

    ENDELSE

  ENDIF ELSE IF ies_instrument(header.datatype,/HIST) THEN BEGIN

    dlist = [0]
    dcount = 1
    dsize = 7
    dpositions = 3 ; actually 26 deg not 25.71 deg

  ENDIF ELSE IF ies_instrument(header.datatype,/IPS) THEN BEGIN

    ; NB have adjusted IPS detector numbering so that goes 1-10
    ; set dpositions = header.detectors-1
    ; header.detectors gives position of detectors relative to anti-parallel spin axis

    dlist = WHERE (header.detectors NE 10, dcount) ; avoid IPS background detector
    dsize = 9
    IF dcount GT 0 THEN dpositions = (header.detectors[dlist]-1)

  ENDIF ELSE BEGIN

    ; set dpositions = 0,1,2,3,4,5,6,7,8

    dlist = INDGEN(header.ndetectors)
    dcount = header.ndetectors
    dsize = header.ndetectors
    dpositions = INDGEN(header.ndetectors)

  ENDELSE

END