;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_DIV
;
; Purpose     :	Divides data by channel sizes to convert to counts per keV and the reverse.
;
; Explanation :	Allows conversion from counts per energy channel to counts per keV and the reverse.
;
; Use         : < ies_div, lut_list, nbands, ndetectors, ch_positions, output_array, REVERSE=reverse >
;
; Inputs      : lut_list      : STRUCTURE ARRAY containing LUT run info.
;               nbands        : INT no. of energy channels.
;               ndetectors    : INT no. of detectors.
;               ch_positions  : INTARR(2,nbands,ndetectors,nluts) containing energy channel positions for each detector and LUT
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/96
;                 Adapted from ies_cal.
;
; Modified    :	Version 0.1, 15/4/97
;                 Allowed reverse change.
;               Version 0.2, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.3, 20/07/01
;                 Used ies_runs.
;               Version 0.4, 24/10/03
;                 Changed to lut_list argument.
;
; Version     :	Version 0.4, 24/10/03
;-
;**********************************************************

PRO ies_div, lut_list, nbands, ndetectors, ch_positions, output_array, REVERSE=reverse

  ; take into account channel sizes
  ; convert from counts per channel to count per keV

  ; loop through runs of each lut

  FOR l = 0L, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = lut_list[l].lut

    ; get channel sizes in keV for each detector for this lut
    ; this gives an array (nbands,ndetectors)

    ch_sizes = ch_positions(1,*,*,lut) - ch_positions(0,*,*,lut)

    ch_sizes = REFORM(ch_sizes, nbands, ndetectors)

    ; get array (ndetectors,nbands)
    ; NB output_array is array (noutputs,nenergies,npoints)

    IF KEYWORD_SET(reverse) THEN BEGIN

      ; do channel size multiplication for each energy and each detector and all sectors

      FOR d = 0L, ndetectors-1 DO FOR e = 0L, nbands-1 DO $
        output_array(*,d,e,lut_list[l].st:lut_list[l].en) = output_array(*,d,e,lut_list[l].st:lut_list[l].en) * ch_sizes(e,d)

    ENDIF ELSE BEGIN

      ; do channel size division for each energy and each detector and all sectors

      FOR d = 0L, ndetectors-1 DO FOR e = 0L, nbands-1 DO $
        output_array(*,d,e,lut_list[l].st:lut_list[l].en) = output_array(*,d,e,lut_list[l].st:lut_list[l].en) / ch_sizes(e,d)

    ENDELSE

  ENDFOR

END
