;+
; Project     :	POLAR - CEPPAD
;
; Name        :	IES_DO_AXES
;
; Purpose     :	Sets up axes for display.
;
; Explanation :	Plots time along x axis. Picks suitable ticks for display.
;
; Use         : < ies_do_axes, time_start, time_end, ylo, yhi, xmargin, ymargin, $
;                              TITLE=title, XTITLE=xtitle, YTITLE=ytitle, SAVE_CC=save_cc, $
;                              NO_XAXIS=no_xaxis, NO_YAXIS=no_yaxis, NO_ERASE=no_erase, LOG=log, SHRINK=shrink >
;
; Inputs      : time_start : DOUBLE TAI start time of data.
;               time_end   : DOUBLE TAI end time of data.
;               ylo        : FLOAT  minimum y value in plot window
;               yhi        : FLOAT  maximum y value in plot window
;               xmargin    : INTARR(2) character size of left and right margins.
;               ymargin    : INTARR(2) character size of top and bottom margins.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs:	None.
;
; Keywords    : TITLE       : STRING Title of plot.
;               XTITLE      : STRING Plot x title.
;               YTITLE      : STRING Plot y title.
;               STITLE      : STRING Title of plot to be printed in the right margin.
;               SAVE_CC     : INT flag indicating that should save coordinate conversion to named variable.
;               NO_XAXIS    : INT flag preventing x axis display.
;                                1 = no tick labels
;                                2 = no tick labels or marks
;               NO_YAXIS    : INT flag preventing y axis display.
;               NO_ERASE    : INT flag preventing erase of plot device initially.
;               LOG         : INT flag indicating logarithmic display required.
;               SHRINK      : INT flag indicating whether to allow x axis space around plotted data
;               OVERLAY     : INT flag indicating overlay plot -- plots yaxis on RHS
;               _EXTRA      : Plot keywords.
;
; Calls       :	ies_xticks, ies_do_x_axis, get_chsize.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 31/1/96
;               Adapted from ies_sp_plot.
;
; Modified    :	Version 0.1, MKC, 21/10/96
;                 Replaced charsize with !P.CHARSIZE.
;                 Prevented time_start and time_end from being adjusted.
;                 NB !X.CRANGE = [tstart,tend]
;               Version 0.2, 11/11/96
;                 Added no_yaxis keyword.
;                 Added total suppression of x axis for no_xaxis=2
;                 Adjusted y axis origin in linear plots.
;               Version 0.3, 13/2/97
;                 Added ylo to ies_do_x_axis.
;                 Added _EXTRA keyword.
;               Version 0.4, 26/1/99
;                 Added OVERLAY keyword.
;                 Adjusted use of YLOG keyword in PLOT.
;               Version 0.5, 11/2/00
;                 Always save coord conversion.
;               Version 0.6, 20/7/00
;                 Added keyword STITLE.
;               Version 0.7, 14/03/01
;                 Fixed charsize used in y axis.
;               Version 0.8, 14/11/01
;                 Prevented y axis title being displayed if NO_YAXIS keyword set.
;
; Version     :	Version 0.8, 14/11/01
;-
;**********************************************************

PRO ies_do_axes, time_start, time_end, ylo, yhi, xmargin, ymargin, $
                 TITLE=title, XTITLE=xtitle, YTITLE=ytitle, STITLE=stitle, SAVE_CC=save_cc, $
                 NO_XAXIS=no_xaxis, NO_YAXIS=no_yaxis, NO_ERASE=no_erase, LOG=log, SHRINK=shrink, $
                 OVERLAY=overlay, _EXTRA=extra

  ; explicitly zero titles if not defined
  ; for some reason plot objects otherwise with
  ; % PLOT: Variable is undefined: Conversion source variable.

  IF NOT KEYWORD_SET(title)  THEN title  = ''
  IF NOT KEYWORD_SET(xtitle) THEN xtitle = ''
  IF NOT KEYWORD_SET(ytitle) THEN ytitle = ''
  IF NOT KEYWORD_SET(no_erase) THEN no_erase=0
  IF NOT KEYWORD_SET(no_xaxis) THEN no_xaxis=0
  IF NOT KEYWORD_SET(no_yaxis) THEN no_yaxis=0
  IF NOT KEYWORD_SET(overlay) THEN overlay=0

  ; get character sizes in normal coords

  x_chsize = get_chsize( /X )
  y_chsize = get_chsize( /Y )

  ; get adjustable copies of time_start and time_end

  tstart = time_start
  tend   = time_end

  ; fix xticks
  ; tstart and tend may be adjusted within routine

  ies_xticks, tstart, tend, 1.0/x_chsize, xticks, xminor, xtickname, xtickv, SHRINK=shrink

  ; adjust plot window by no. of characters
  ; allow room for color bar
  ; xmargin = [10,15], ymargin=[4,2]

  !P.POSITION(0) = !P.REGION(0) + xmargin(0)*x_chsize
  !P.POSITION(1) = !P.REGION(1) + ymargin(0)*y_chsize
  !P.POSITION(2) = !P.REGION(2) - xmargin(1)*x_chsize
  !P.POSITION(3) = !P.REGION(3) - ymargin(1)*y_chsize

  ; set so that y data fits in plot box exactly.

  ystyle=1

  ; check if y axis display required

  IF KEYWORD_SET(no_yaxis) THEN ystyle = ystyle OR 4

  ; set plotting coordinate system
  ; suppress x axis

;  PLOT, [0,1], XRANGE=[tstart, tend], YRANGE=[ylo, yhi], /NODATA, YTICKLEN=-x_chsize, XSTYLE=5, YSTYLE=ystyle, $
;             TITLE=title, NOERASE=no_erase, CHARSIZE=adjust*!P.CHARSIZE, YLOG=log, _EXTRA=extra

  PLOT, [1,2], /NODATA, XSTYLE=5, YSTYLE=5, TITLE=title, NOERASE=no_erase, $
    XRANGE=[tstart, tend], _EXTRA=extra

  ; adjust character size for y axis ticks if more than 3 digits

  IF yhi GE 1000 OR (KEYWORD_SET(log) AND ylo LT 1) THEN adjust = 0.8 ELSE adjust = 1.0

  ; do Y axis

  AXIS, YLOG=log, YRANGE=[ylo, yhi], YTICKLEN=-x_chsize, YSTYLE=ystyle, YAXIS=overlay, /SAVE, CHARSIZE=adjust*!P.CHARSIZE

  ; fix position of y axis label
  ; NB cannot simply swap x and y normal co-ords

  IF KEYWORD_SET(NO_YAXIS) THEN BEGIN

    ; dont plot yaxis title

  ENDIF ELSE IF KEYWORD_SET(overlay) THEN BEGIN

    XYOUTS, !P.POSITION[2]+x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.1, $
            (!P.POSITION(1)+!P.POSITION(3)-STRLEN(ytitle)*y_chsize*!D.X_CH_SIZE/!D.Y_CH_SIZE)/2.0, ytitle, $
            ORIENTATION=90, CHARSIZE=!P.CHARSIZE, /NORMAL

  ENDIF ELSE BEGIN

    IF KEYWORD_SET(stitle) THEN BEGIN

      ; fix position of title if plotted in y axis margin

;      XYOUTS, !P.REGION(0)+x_chsize*0.5, $
;              !P.REGION(3)-1.1*y_chsize, stitle, CHARSIZE=!P.CHARSIZE, /NORMAL

;      XYOUTS, !P.REGION(0)+x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.1, $
;              (!P.POSITION(1)+!P.POSITION(3)-STRLEN(ytitle)*y_chsize*!D.X_CH_SIZE/!D.Y_CH_SIZE)/2.0, ytitle, $
;              ORIENTATION=90, CHARSIZE=!P.CHARSIZE, /NORMAL

      XYOUTS, !P.REGION(0)+x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.1, $
              (!P.POSITION(1)+!P.POSITION(3)-STRLEN(stitle)*y_chsize*!D.X_CH_SIZE/!D.Y_CH_SIZE)/2.0, stitle, $
              ORIENTATION=90, CHARSIZE=!P.CHARSIZE, /NORMAL

      XYOUTS, !P.REGION(0)+2*x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.1, $
              (!P.POSITION(1)+!P.POSITION(3)-STRLEN(ytitle)*y_chsize*!D.X_CH_SIZE/!D.Y_CH_SIZE)/2.0, ytitle, $
              ORIENTATION=90, CHARSIZE=!P.CHARSIZE, /NORMAL

    ENDIF ELSE BEGIN

      XYOUTS, !P.REGION(0)+x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.1, $
              (!P.POSITION(1)+!P.POSITION(3)-STRLEN(ytitle)*y_chsize*!D.X_CH_SIZE/!D.Y_CH_SIZE)/2.0, ytitle, $
              ORIENTATION=90, CHARSIZE=!P.CHARSIZE, /NORMAL

    ENDELSE

  ENDELSE

  ; plot x axis

  IF no_xaxis LT 2 THEN ies_do_x_axis, xticks, xminor, xtickname, xtickv, ylo, XTITLE=xtitle, NO_XAXIS=no_xaxis

  ; save coordinate conversion to device coords.

  save_cc = !X.S

END
