;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_DO_CENTROID
;
; Purpose     : Locates pedestal bin position and optionally subtracts pedestal from each sample.
;
; Explanation :	This routine calculates pedestal shift to a fraction of a channel using the centroid of the
;               pedestal.
;               The pedestal contribution may then be subtracted if required.
;
; Use         : < index = ies_do_centroid(npeds, array, max_sample_lists, samples, tot_exp_counts, channel_centres) >
;
; Inputs      : npeds         : INT no of pedestal channels
;               array         : FLTARR(nbands) containing input data for each energy channel
;                               May be changed on output if SUB flag set.
;               ch_positions  : INTARR(2,nbands) containing energy channel bin positions for given detector and LUT
;               pedestal_posn : FLOAT position of pedestal
;                               NB This is calculated putting data on bin centres.
;               pedestal_width: FLOAT FWHM of pedestal in bins.
;
; Opt. Inputs : None.
;
; Outputs     : index         : LONG centroid position
;
; Opt. Outputs:	None.
;
; Keywords    : PED     : Named variable returning pedestal on output.
;               MAX_POS : Named variable returning channel position of peak on output.
;
; Written     :	Version 0.0, 25/11/03
;                 Separated from cep pedestals.
;
; Modified    : Version 0.1, 20/01/04
;                 Moved pedestal info calculation and added PED and MAX_POS keywords.
;                 Changed so that return centroid position as function value.
;                 Removed SUB, POLAR, CHISQ keywords.
;
; Version     :	Version 0.1, 20/01/04
;-
;**********************************************************

FUNCTION ies_do_centroid, npeds, array, ch_positions, pedestal_posn, pedestal_width, PED=ped, MAX_POS=max_pos

  ; get pedestal position from centroid

  ; ped_array is pedestal channels for given sector and given sample [npeds]

  ped_array = array[0:npeds-1]

  ; find pedestal channel maximum

  max_ped_array = MAX(ped_array,max_pos)

  ; use channels either side of max channel

  ; get indices for 2n+1 samples
  ; get n samples either side of peak

  n = 1
  istart = (max_pos-n) > 0
  iend   = (max_pos+n)<(npeds-1)
  inds   = istart+INDGEN(iend-istart+1)

  ; get samples surrounding peak

  ped_array = ped_array[inds]

  ; calculate centroid of data samples

  posns  = TOTAL(ch_positions[*,inds],1)/2.0

  ; tot_ped_array is the sum of counts in ped_array

  tot_ped_array = TOTAL(ped_array)

  IF tot_ped_array GT 0 THEN pos = TOTAL(ped_array*posns)/tot_ped_array ELSE pos = pedestal_posn

  ; fix pp as for cross_correlation

  pp = 10*(pedestal_posn+5-pos)

  ; get synthetic pedestal

  ; get distances of band boundaries from centroid

  dists_0 = REFORM(ch_positions[0,0:npeds-1])-pos

  dists_1 = REFORM(ch_positions[1,0:npeds-1])-pos

  ; normalize distance to pedestal Gaussian widths

  dists_0 = dists_0/pedestal_width

  dists_1 = dists_1/pedestal_width

  ; get areas under Gaussian from -inf

  int_0 = GAUSSINT(dists_0)

  int_1 = GAUSSINT(dists_1)

  ; get Gaussian contribution to band

  gs = int_1 - int_0

  ; get required data samples

  ssamples = gs[inds]

  ; normalize required data samples

  normalization = ies_get_normalization(ped_array,ssamples)

  ; calculate pedestal contribution

  ped = gs*normalization[0]

  ; return position

  RETURN, pp

END