;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_DO_CLEAN
;
; Purpose     :	Applies various corrections to data.
;
; Explanation :	Calculates pedestal positions and subtracts pedestal from data.
;               Applies pedestal offset correction and resamples data.
;
; Use         : < ies_do_clean, lut_list, header, data, output_array, output_chans, nbands, output_info >
;
; Inputs      : lut_list     :
;               header       : intermediate format header structure.
;               data         : uncalibrated intermediate format data structure array.
;               output_array : uncorrected data
;               output_chans : original energy channel definitions in keV
;               output_info  : structure containing plotting and processing info
;
; Opt. Inputs : None.
;
; Outputs     : output_array : corrected data
;               output_chans : energy channel definitions in keV
;               nbands       : no. of energy channels
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Prev. Hist. :	Adapted from ies_calibration.
;
; Written     :	Version 0.0, Martin Carter, RAL, 15/11/96
;
; Modified    :	Version 0.1, 16/1/97.
;                 Changed to a procedure.
;                 Added new pedestal subtraction.
;                 Added pedestal shift routine.
;               Version 0.2, 16/1/97.
;                 Changed .ndetectors tag.
;                 Added pedestal shift to ies pedestals.
;               Version 0.3, 11/3/97.
;                 Changed pedestal_shift to pedestal_info.
;               Version 0.4, 15/4/97.
;                 Added output_info to argument list and removed KEYWORDs.
;               Version 0.5, 18/8/97.
;                 Added energy efficiency.
;                 Changed ies pedestals argument list.
;               Version 0.6, 10/10/97
;                 Added energy_calibration offset.
;               Version 0.7, 25/2/98
;                 Made 10 th channel an integral channel.
;                 Changed so that ies shift works in bins rather than keV.
;                 Changed so that ies shift does not update nbands.
;               Version 0.8, 28/10/98
;                 Changed so that if rates then converts to counts per sector.
;               Version 0.9, 7/9/00
;                 Fixed bug where header.nbands was used instead of bands -- this causes problems
;                 if shift used.
;                 Force use of CENTROID algorithm for RAPID data
;               Version 1.0, 20/9/00
;                 Moved set up of pedestal calibration data to here.
;                 Added RAPID keyword to ies pedestals.
;               Version 1.1, 13/10/00
;                 Passed lut map in ies pedestals argument list instead of picking up from calibration file.
;               Version 1.2, 08/05/01
;                 Added ENDTIMES keyword to ies_rat.
;               Version 1.3 12/07/02
;                 Used ies_tag_exists.
;               Version 1.4 08/02/03
;                 Ensured output_chans is a float array.
;               Version 1.5 14/04/03
;                 Changed argument list of ies_rat.
;               Version 1.6, 26/09/03
;                 Used new routine for instrument test.
;                 Check do not use ies pedestals for RAPID data.
;                 Changed argument list of ies get pedestal calibration
;                 Renamed routine ies pedestals.
;                 Used ies get dir.
;               Version 1.7, 24/10/03
;                 Used ies lut list.
;               Version 1.8, 14/11/03
;                 Used ies_bins_to_kev.
;               Version 1.9, 20/11/03
;                 Separated routine ies docal into ies do clean and ies do cal.
;                 Added LUT_LIST keyword.
;                 Changed ERRORS keyword to CHISQ in cep_pedestals.
;                 Moved ies get pedestal calibration to cep pedestals.
;               Version 2.0, 17/12/03
;                 Used ies set ch positions..
;
; Version     :	Version 2.0, 17/12/03
;-
;**********************************************************

PRO ies_do_clean, lut_list, header, data, output_array, output_chans, nbands, output_info

  ; calculate pedestal positions and subtract pedestals
  ; NB works in bins
  ; NB .clean is a version no.

  IF output_info.subtract OR (output_info.shift AND header.clean EQ 0) THEN BEGIN

    ; NB Only for POLAR IES

    IF NOT ies_instrument(header.datatype,/POLAR) THEN MESSAGE, 'IMPLEMENTATION ERROR', /TRACEBACK

    pedestals = cep_pedestals ( output_array, data.time, lut_list, header.detectors, $
                                header.nsectors, header.ndetectors, nbands, $
                                header.ch_positions, header.lut_map, pedestal_info, SUB=output_info.subtract, $
                                CHISQ=NOT (header.div OR header.cal OR header.rat) )

  ENDIF

  ; clean up data, ensure pass by reference
  ; NB ch_positions in bins
  ;    output_chans in kev

  IF (KEYWORD_SET(output_info.shift) AND header.clean EQ 0) AND N_ELEMENTS(pedestals) GT 0 THEN BEGIN

    ; convert to counts per channel if necessary
    ; NB need channel sizes in keV

    IF header.div THEN ies_div, luts, nbands, header.ndetectors, output_chans, output_array, /REVERSE

    ; do resampling
    ; NB needs channels defined in terms of bins
    ; NB both output_array and output_chans are updated

    new_bands = 10

    ; set up ch_positions in bins
    ; NB uses POLAR energy channels

    new_ch_positions = ies_set_ch_positions(new_bands, header.ndetectors, N_ELEMENTS(header.ch_positions[0,0,0,*]) )

    ; do shift

    ies_shift, lut_list, header.nsectors, new_bands, header.ndetectors, nbands, header.ch_positions, new_ch_positions, pedestals, output_array

    ; set up (new) energies in keV of channel boundaries

    output_chans = ies_bins_to_kev(header.energy_cal, new_ch_positions)

    ; set up new nbands

    nbands = new_bands

    ; convert back to counts per keV if necessary
    ; NB need channel sizes in keV

    IF header.div THEN ies_div, lut_list, nbands, header.ndetectors, output_chans, output_array

  ENDIF

  ; reform output array

  output_array = REFORM ( output_array, header.nsectors, header.ndetectors, nbands, header.npoints)

END
