;+
; Project     :	CLUSTER - RAPID, POLAR _ CEPPAD
;
; Name        :	IES_DO_COLORBAR
;
; Purpose     :	Draws color bar.
;
; Explanation :	Draws color bar.
;
; Use         : < ies_do_colorbar, ymin, ymax, TITLE=title, CTITLE=ctitle, NO_XAXIS=no_xaxis, LOG=log >
;
; Inputs      :
;
; Opt. Inputs : None.
;
; Outputs     : Draws plot on current graphics device.
;
; Opt. Outputs:	None.
;
; Keywords    : TITLE       : Title of plot.
;               CTITLE      : Title for colorbar.
;
; Calls       :	None..
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, 29/10/96
;                 Added plot title.
;               Version 0.2, 3/3/97
;                 Modified y title size
;               Version 0.3, 2/11/98
;                 Fixed ymin in linear plot.
;               Version 0.4, 26/1/99
;                 Adjusted use of YLOG keyword in AXIS.
;               Version 0.5, 11/2/00
;                 Under windows exp has form e00n rather than e0n.
;                 Requires extra character for field.
;               Version 0.6, 14/11/01
;                 Corrected title size adjustement.
;               Version 0.7, 17/01/02
;                 Added FIXED_CHARSIZE keyword.
;               Version 0.8, 12/04/02
;                 Added XCHARSTEP keyword.
;               Version 0.9, 09/07/02
;                 Removed ncolors from argument list.
;               Version 1.0, 03/03/03
;                 Found if 4 panels used then not enough space left for colorbar and crashed.
;                 Allow for this problem.
;
; Version     :	Version 1.0, 03/03/03
;-
;**********************************************************

PRO ies_do_colorbar, ymin, ymax, TITLE=title, CTITLE=ctitle, $
  NO_XAXIS=no_xaxis, LOG=log, FIXED_CHARSIZE=fixed_charsize, XCHARSTEP=xcharstep

  ; draw color bar

  ; save plot window and axis range

  plot_window = !P.POSITION
  axis_crange = !X.CRANGE

  ; get character sizes in normal coords

  x_chsize = get_chsize( /X )
  y_chsize = get_chsize( /Y )

  ; leave room for plot title if necessary

  IF N_ELEMENTS(title) GT 0 THEN BEGIN

    ch_factor = 1.5

    ch_factor = ch_factor/((1.2*ch_factor*STRLEN(title)*x_chsize/(!P.REGION(2)-!P.POSITION(2)-2*x_chsize)) > 1.0)

    IF KEYWORD_SET(fixed_charsize) THEN ch_factor = 1

    !P.POSITION(3) = !P.POSITION(3) - 1.67*y_chsize*ch_factor

    XYOUTS, (2*x_chsize+!P.POSITION(2)+!P.REGION(2) - STRLEN(title)*ch_factor*x_chsize)/2.0, !P.POSITION(3)+y_chsize, title, /NORMAL, $
             CHARSIZE=ch_factor*!P.CHARSIZE

  ENDIF

  ; fix position of y axis label
  ; NB cannot simply swap x and y normal co-ords

  XYOUTS, !P.POSITION(2)+x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.5, $
          ((!P.POSITION(1)+!P.POSITION(3)-STRLEN(ctitle)*y_chsize*!D.X_CH_SIZE/!D.Y_CH_SIZE)/2.0) > !P.POSITION(1), $
          ctitle, ORIENTATION=90, CHARSIZE=!P.CHARSIZE, /NORMAL

  ; a space has been left to the left of the plot window
  ; reset plot window and region to point at this space
  ; adjust plot window by no. of characters

  IF NOT KEYWORD_SET(xcharstep) THEN xcharstep = 6

  pre = x_chsize*!D.Y_CH_SIZE/!D.X_CH_SIZE*1.5 + xcharstep*x_chsize

  post = 1*x_chsize

  ; Found if 4 panels used then not enough space left for colorbar and crashed.
  ; Ensure colorbar at least on character wide.

  IF pre+post GT !P.REGION[2]-!P.POSITION[2]-x_chsize THEN BEGIN

    factor = (!P.REGION[2]-!P.POSITION[2]-x_chsize)/(pre+post)

    pre  = factor*pre

    post = factor*post

  ENDIF

  !P.POSITION(0) = !P.POSITION(2) + pre
  !P.POSITION(2) = !P.REGION(2) - post

  ; set plot axes in device coords, suppress x and y axes

  PLOT, [0,1], /NODATA, XSTYLE=4, YSTYLE=4, /NOERASE

  ; write out colorbar title

  IF NOT KEYWORD_SET(no_xaxis) THEN BEGIN

    color = 'Color'

    XYOUTS, (!P.POSITION(0)+!P.POSITION(2) - STRLEN(color)*x_chsize)/2.0, !P.POSITION(1)-y_chsize, color, /NORMAL, CHARSIZE=!P.CHARSIZE

  ENDIF

  ; set tick format
  ; if max ge 1.e4 then use default
  ; >= 1e4 default is a.b*10n
  ; <=1e-4 default is a.b*10-n
  ; 1e-4<x<1e4 default is ab0 or 0.00ab
  ; previously forced use of format e6.0 below 1e4
  ; this does not give sufficient space under windows due to long exponent
  ; not clear why this was done so remove for now

  ; NB can produce up to 8 character label for range 1e-4->1e4 i.e 0.001-> 1000.000


  IF ymax LT 1.e4 THEN BEGIN
    ;ytickformat = '(e5.0)'
    ytickformat = ''
  ENDIF ELSE BEGIN
    ytickformat = ''
  ENDELSE

  ; draw y axis
  ; for some reason the y charsize does not default to the expected value
  ; yticklen's seem to need to be this big to be visible

  IF KEYWORD_SET(fixed_charsize) THEN ch_factor = 1 ELSE ch_factor = 0.6

  AXIS, YLOG=log, YAXIS=0, YRANGE=[ymin,ymax], YTICKLEN=-5*x_chsize, YSTYLE=1, $
        CHARSIZE=ch_factor*!P.CHARSIZE, YTICKFORMAT=ytickformat

  ; get device coords of plot window

  vector = CONVERT_COORD ( !X.WINDOW, !Y.WINDOW, /NORMAL, /TO_DEVICE )

  ; get origin and size in device co-ords (inclusive)

  xorig = vector(0,0)
  yorig = vector(1,0)
  xsize = vector(0,1) - xorig + 1
  ysize = vector(1,1) - yorig + 1

  ; plot array, checking if scaleable output
  ; NB must be after plot if plot erases

  IF !D.FLAGS AND 1 THEN $
    TV, FINDGEN(1,ies_ncolors()), xorig, yorig, XSIZE=xsize, YSIZE=ysize, /DEVICE, ORDER=0 $
  ELSE $
    TV, REPLICATE(1,xsize)#FINDGEN(ysize)*ies_ncolors() /FLOAT(ysize), xorig, yorig, ORDER=0

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

 ; restore plot window

 !P.POSITION = plot_window

 !X.CRANGE   = axis_crange

END

