;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_DO_CROSS_CORRELATION
;
; Purpose     : Locates pedestal bin position and optionally subtracts pedestal from each sample.
;
; Explanation :	This routine calculates pedestal shift to a fraction of a channel by cross-correlating the known
;               pedestal profile with the sampled data.
;               The pedestal contribution may then be subtracted if required.
;               The routine uses a slightly different algorithm for CEPPAD and RAPID data because of the different
;               pedestal information available : for CEPPAD uses chanels surrounding the peak pedestal channel; for RAPID
;               uses all the pedestal channels. For CEPPAD normalizes the shifted histogram data using minimum chisq or
;               minimum error. For RAPID normalizes the shifted histogram data to the expected number of pedestal counts.
;
; Use         : < index = ies_do_cross_correlation(npeds, array, max_sample_lists, samples, pedestal_posn, $
;                               channel_centres, tot_exp_counts) >
;
; Inputs      : npeds            : INT no of pedestal channels
;               array            : FLTARR(nbands) containing input data for each energy channel
;                                  May be changed on output if SUB flag set.
;               ch_positions     : INTARR(2,nbands) containing energy channel bin positions for given detector and LUT
;               max_samples_list : FLTARR(1+nshifts, npeds) indices of shifts with peak in channel 1+channel.
;                                  Element 0 contains no of indices. max_samples contains the index of the peak
;                                  for each shift. This is used to restrict the search for max cross-correlation
;                                  to samples with the same peak position as the data
;               samples          : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data. Only the first npeds energy channels are kept.
;                                  Whole spectrum is normalized to one.
;               tot_exp_counts   : LONG total no. of counts expected for integration time (if no compression).
;
; Opt. Inputs : None.
;
; Outputs     : index            : LONG index into samples array of best correlation.
;
; Opt. Outputs:	None.
;
; Keywords    : POLAR   : FLAG indicating should do POLAR type processing.
;               CHISQ   : FLAG indicating should use chisq for calculating normalization factor.
;               PED     : Named variable returning pedestal on output.
;               MAX_POS : Named variable returning channel position of peak on output.
;
; Written     :	Version 0.0, 17/09/00
;
; Modified    :	Version 0.1, 9/2/01
;                 Converted occupancy and missing counts to percentage and out of 10 respectively.
;               Version 0.2, 20/07/01
;                 Used ies_runs.
;               Version 0.3, 14/05/02
;                 Used rcs_check_ies to get expected counts.
;               Version 0.4, 24/10/03
;                 Changed to lut_list argument.
;               Version 0.5, 20/11/03
;                 Separated calibration info -- removed keywords CH_POSITIONS, COUNT_CAL, ENERGY_CAL, ENERGY_EFF
;                 Used rcs pedestal samples.
;                 Added BM keyword.
;                 Removed ped_chans.
;                 Separated from rcs pedestals.
;                 Incorporated cep pedestals.
;                 Corrected bug in cep pedestals if ERRORS flag not used.
;                 Change ERRORS keyword to CHISQ.
;                 Used ies get normalization.
;                 Used cep pedestal info and rcs pedestal info.
;               Version 0.6, 20/01/04
;                 Moved pedestal info calculation and added PED and MAX_POS keywords.
;                 Changed so that return index into samples as function value.
;                 Removed pedestal_posn, pedestal_centre, channel_centres arguments.
;                 Removed SUB keyword.
;
; Version     :	Version 0.6, 20/01/04
;-
;**********************************************************

FUNCTION ies_do_cross_correlation, npeds, array, ch_positions, max_sample_lists, samples, tot_exp_counts, $
                                   POLAR=polar, CHISQ=chisq, PED=ped, MAX_POS=max_pos

  ; get pedestal position from cross-correlation

  ; ped_array is pedestal channels for given sector and given sample [npeds]

  ped_array = array[0:npeds-1]

  ; find pedestal channel maximum

  max_ped_array = MAX(ped_array,max_pos)

  ; find histogram samples with peak at max_pos

  max_count = max_sample_lists[0,max_pos]

  IF max_count GT 0 THEN max_list = max_sample_lists[1:max_count,max_pos] ELSE max_list = [0]

  ; samples  : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
  ;            data sampled as for the real data.
  ;            These have been normalized to unity for the whole spectrum.
  ; ssamples : FLTARR[npeds,max_count] are the shifted samples for pedestal channels with same peak as the data

  ssamples = samples[*,max_list]

  ; check if POLAR type processing

  IF KEYWORD_SET(POLAR) THEN BEGIN

    ; use channels either side of max channel

    ; get indices for 2n+1 samples
    ; get n samples either side of peak

    n = 1
    istart = (max_pos-n) > 0
    iend   = (max_pos+n)<(npeds-1)
    inds   = istart+INDGEN(iend-istart+1)

    ; get samples surrounding peak

    ped_array = ped_array[inds]

    ; extract array

    ssamples = ssamples[inds,*]

  ENDIF

  ; tot_ped_array is the sum of counts in ped_array

  tot_ped_array = TOTAL(ped_array)

  ; form a 2-D array replicating ped_array for each shifted sample with same peak
  ; ped2d_array  : FLTARR[npeds,max_count] pedestal data replicated

  ped2d_array = ped_array#REPLICATE(1,max_count>1)

  ; normalize ssamples
  ; check if POLAR OR CLUSTER type processing

  IF KEYWORD_SET(POLAR) THEN BEGIN

    ; get normalization

    normalization = ies_get_normalization( ped2d_array, ssamples, CHISQ=chisq)

    ; apply normalization factors
    ; normalization : FLTARR[max_count] ie different factor for each shift
    ; ssamples : FLTARR[ninds,max_count]
    ; this just multiplies each row of ssamples by a different normalization factor.

    ssamples = ssamples * ( REPLICATE(1,N_ELEMENTS(inds)) # normalization )

  ENDIF ELSE BEGIN

    ; get total no of counts in pedestal and signal channels

    tot_array = TOTAL(array)

    ; normalize ssamples to expected no of counts taking into account signal counts
    ; reduce counts in predicted pedestal samples by observed signal counts
    ; NB do not have actual total counts

    normalization = tot_exp_counts-tot_array+tot_ped_array

    ; NB ssamples : FLTARR[npeds,max_count], normalization : FLOAT

    ssamples = ssamples*normalization

  ENDELSE

  ; take residuals with sampled pedestal array

  residuals =  ped2d_array - ssamples

  ; minimize sum of chisq over the pedestal channels where chisq is the square of difference between the
  ; actual counts and the predicted counts for each shift divided by the actual counts

  smsqs  = TOTAL((residuals*residuals)/(ped2d_array>1),1)  ; takes into account Poisson errors

  ; find min

  dummy = MIN ( smsqs, p )

  ; get index into samples

  pp = max_list[p]

  ; get pedestal

  IF KEYWORD_SET(POLAR) THEN BEGIN

    ; NB ssamples are reduced normalized pedestal channels for POLAR

    ped = samples[*,pp]*normalization[p]

  ENDIF ELSE BEGIN

    ped = ssamples[*,p]

  ENDELSE

  ; return index into samples

  RETURN, pp

END