;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_DO_GYRORADII
;
; Purpose     :	Calculates gyroradii projection.
;
; Explanation :	Calculates gyroradii projection.
;               A given energy and detector samples nsectors pitch angles.
;               Each of these samples has an associated count rate and unique picth angle.
;               When projected onto gyroradii there is always a sample anti-parallel to the
;               B field with the same gyroradius as a sample parallel to the B field.
;               Thus there are nsectors/2 unique gyroradii positions, each with two associated
;               count rates corresponding to parallel and anti-parallel directions.
;
; Use         : < array = ies_do_gyroradii (nradii, output_array, output_info, header, data) >
;
; Inputs      : nradii        : INT.
;               output_info   : STRUCTURE containing plotting and processing info
;               header        : STRUCTURE info structure for data.
;               data          : STRARR for each sample point.
;
; Opt. Inputs : None.
;
; Outputs     : array         : FLTARR(*,nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    :
;               SPIN_AVERAGE : Sum over sectors.
;               DETECTOR_AVERAGE : Sum over detectors.
;               GYRORADII : FLAG indicating gyroradii output,
;
; Modified    :	Version 0.1, 29/09/03
;                 Used new routine for instrument test.
;               Version 0.2, 20/10/03
;                 Used routine ies generate masks.
;
; Version     :	Version 0.2, 20/10/03
;-
;**********************************************************

FUNCTION ies_do_gyroradii, nradii, output_array, output_chans, output_info, header, data

  ; get gyroradii histogram

  array = REPLICATE(-1.0, header.nbands, nradii, header.npoints)

  ; get pitch angles

  ies_pitch_angle, header, data, pas

  ; generate mask values

  ies_generate_masks, header, data, pas, EARTH=ies_instrument(header.datatype,/IPS), SUN=ies_instrument(header.datatype,/IPS)

  ; check magnetic field available

  IF N_ELEMENTS(pas) EQ 0 THEN RETURN, array

  ; loop through each sample point
  ; use duplicated pitch angle trick

  FOR p = 0, header.npoints-1 DO BEGIN

    bt = data[p].mf[2]

    FOR e = 0, header.nbands-1 DO FOR s = 0, header.nsectors/2-1 DO FOR d = 0, header.ndetectors-1 DO BEGIN

      ; cehck if valid pitch angle or masked off

      IF pas[s,d,p] GE 0 THEN BEGIN

        rad = ies_gyro_radius(TOTAL(output_chans[*,e,d,data[p].lut])/2,bt,pas[s,d,p],$
                e=ies_instrument(header.datatype,/IES) OR ies_instrument(header.datatype,/EHIST), $
                P=ies_instrument(header.datatype,/IPS) OR ies_instrument(header.datatype,/PHIST) OR $
                  ies_instrument(header.datatype,/I3DD_H))/500.*nradii

        ; create 7 decades starting at 0.001 km

        rad = (ALOG10(rad)+3)/7.0*nradii

        rad = 0 > rad < (nradii-1)

        array[e,rad,p] = MAX([array[e,rad,p],output_array[s,d,e,p],output_array[s+header.nsectors/2,d,e,p]])

        ; check how many samples there are for each gyroradii bin
        ; if array[e,rad,p] lt 0 then array[e,rad,p] = 1 ELSE array[e,rad,p] = array[e,rad,p] +1

      ENDIF

    ENDFOR

  ENDFOR

  RETURN, array

END
