;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_DO_PLOT
;
; Purpose     :	Plot data in desired format.
;
; Explanation : Does plot according to specified output format.
;
; Use         : < ies_do_plot, pos, procedure, xmargin, ymargin, /NO_XAXIS, /FULL, /WIDGET, /ANGLEANGLE, /SAVE_CC>
;
; Inputs      : pos       : INT no. for plot.
;               procedure : STRING prodecure name for doing plot.
;
; Opt. Inputs : xmargin   : FLTARR(2) xmargin sizes
;               ymargin   : FLTARR(2) ymargin sizes
;
; Outputs     : Displays plots in window.
;
; Opt. Outputs:	xmargin   : FLTARR(2) xmargin sizes
;               ymargin   : FLTARR(2) ymargin sizes
;
; Keywords    : /NO_XAXIS   = controls whether x axis displayed
;               /FULL       = Indicates whether full detector display.
;               /WIDGET     = Indicates data displayed in draw widget.
;               /ANGLEANGLE = Indicates ANGLE/ANGLE data format.
;               /TIMELINE   = Indicates TIMELINE data format.
;               /SAVE_CC    = Saves coordinate conversion in common block.
;               /SHRINK     = Allows time axis to be adjusted.
;
; Calls       :	ies_sp_plot, ies_sinplot, get_chsize, ies_ctitle, ies_band_labels.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : Display.
;
; Prev. Hist. :	Adapted from ies_window_plot.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/10/96
;
; Modified    : Version 0.1, 8/11/96
;                 Added PROFS keyword.
;                 Removed cal keyword from CALL_PROCEDURE call.
;               Version 0.2, 16/1/97
;                 Renamed output_info.clean tag to .shift.
;                 Removed PROFS keyword.
;                 Added DETECTOR vs SECTOR format.
;                 Moved detector sort to ies extdata.
;                 Added margins to argument list.
;               Version 0.3, 2/4/97
;                 Added minbin to procedure call.
;               Version 0.4, 15/4/97
;                 Changed so that DIV, CAL and RAT may be applied to clean data.
;                 Changed so that gets full angle/angle plot rates right.
;               Version 0.5, 12/5/97
;                 Added tstart and tstop to argument list of ies_sp_plot and ies_sinplot
;               Version 0.6, 29/8/97
;                 Set ylo, yhi for ies_sp_plot
;               Version 0.7, 23/11/97
;                 Omitted sector 0 from spin-average if IPS.
;               Version 0.8, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;               Version 0.9, 9/10/98
;                 Added ncolors to argument list.
;               Version 1.0, 26/10/98
;                 Added pitch angle plot.
;                 Added detector sum.
;               Version 1.1, 10/11/98
;                 Added ies_ctitle.
;                 Moved setting of band_labels from ies_calibrate to ies_band_labels.
;               Version 1.2, 22/2/99
;                 Allowed negative hi and lo.
;               Version 1.3, 17/3/99
;                 Allowed channels and hilo.
;               Version 1.4, 1/9/00
;                 Changed so that uses output_info.minimum and maximum without ABS,
;               Version 1.5, 14/2/01
;                 Added ENDTIME keyword to procedure call.
;               Version 1.6, 15/1/02
;                 Changed so that pitch angle plots energy band 0 as full.
;               Version 1.7, 08/05/02
;                 Changed so that pitch angle plots full_band as full.
;               Version 1.8, 09/07/02
;                 Removed ncolors from argument list.
;               Version 1.9, 08/02/03
;                 Added GYRORADIUS vs ENERGY format.
;
; Version     :	Version 1.9, 08/02/03
;-
;**********************************************************

PRO ies_do_plot, pos, procedure, xmargin, ymargin, NO_XAXIS=no_xaxis, FULL=full, WIDGET=widget, $
                                 ANGLEANGLE=angleangle, TIMELINE=timeline, SAVE_CC=save_cc, SHRINK=shrink

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information
  ; data          = structure array containing extracted data
  ; output_info   = structure containing plotting and processing info
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem, mf, at

  ; element is index into output_array(element,*,npoints)
  ; detector is index into output_chans(2,*,detector,nluts)

  IF output_info.format EQ 'ENERGY vs DETECTOR' THEN BEGIN  ; NB format for ANGLE/ANGLE

    IF KEYWORD_SET(full) THEN element = output_info.full_detector ELSE element  = pos

    detector = element

    IF KEYWORD_SET(output_info.button_values(0)) THEN ytitle = header.ytitle + ' [keV]' ELSE ytitle = header.ytitle + ' [channels]'

  ENDIF ELSE IF output_info.format EQ 'ENERGY vs SECTOR' THEN BEGIN

    IF KEYWORD_SET(full) THEN element  = output_info.full_sector ELSE element  = pos

    detector = element

    ytitle = header.ytitle

  ENDIF ELSE IF output_info.format EQ 'SECTOR vs DETECTOR' THEN BEGIN  ; NB format for TIMELINE

    IF KEYWORD_SET(full) THEN element  = output_info.full_detector ELSE element  = pos

    detector = output_info.full_detector

    IF KEYWORD_SET(output_info.button_values(0)) THEN ytitle = 'Sectors [degrees]' ELSE ytitle = 'Sectors [number]'

  ENDIF ELSE IF output_info.format EQ 'DETECTOR vs SECTOR' THEN BEGIN

    IF KEYWORD_SET(full) THEN element  = 0 ELSE element  = pos

    detector = element

    IF KEYWORD_SET(output_info.button_values(0)) THEN ytitle = 'Detector [degrees]' ELSE ytitle = 'Detector [number]'

  ENDIF ELSE IF output_info.format EQ 'PITCH ANGLE vs ENERGY' THEN BEGIN

    ;IF KEYWORD_SET(full) THEN element  = header.nbands-1 ELSE element  = pos
    IF KEYWORD_SET(full) THEN element  = output_info.full_band ELSE element  = pos

    detector = element

    IF KEYWORD_SET(output_info.button_values(0)) THEN ytitle = 'Pitch Angle [degrees]' ELSE ytitle = 'Pitch angle [bin]'

  ENDIF ELSE IF output_info.format EQ 'GYRORADIUS vs ENERGY' THEN BEGIN

    IF KEYWORD_SET(full) THEN element  = output_info.full_band ELSE element  = pos

    detector = element

    IF KEYWORD_SET(output_info.button_values(0)) THEN ytitle = 'Gyro-radius [km]' ELSE ytitle = 'Gyro-radius [bin]'

  ENDIF

  ; set band labels

  band_labels = ies_band_labels(header, output_info, output_chans)

  ; define z title

  ztitle = ies_ctitle(header,output_info)

  IF KEYWORD_SET(widget) THEN BEGIN

    ; check if margins defined

    IF N_ELEMENTS(xmargin) EQ 0 THEN BEGIN
      xmargin = [11,12]
    ENDIF

    IF N_ELEMENTS(ymargin) EQ 0 THEN BEGIN
      IF KEYWORD_SET(full) THEN ymargin = [2.,3] ELSE ymargin = [1.5,1]
      IF KEYWORD_SET(no_xaxis) THEN ymargin[0] = 0.5
    ENDIF

    IF KEYWORD_SET(angleangle) THEN BEGIN

      ; NB ANGLEANGLE is called with DETECTOR vs ENEGRY format
      ;  Note that output_array is count rate here.

      factor = 1
      IF output_info.full_sector EQ 0 THEN BEGIN

        ; spin average
        ; dont bother to take into account mask since only used to give a time axis

        array = TOTAL ( output_array(*,*,0:output_info.ysamples-1,*), 1)

        IF output_info.rat THEN array = array/header.nsectors ELSE factor = header.nsectors

      ENDIF ELSE $
        array = output_array(output_info.full_sector-1,*,0:output_info.ysamples-1,*)

      array = REFORM(array, header.ndetectors, output_info.ysamples, header.npoints)

      IF output_info.full_detector EQ header.ndetectors THEN BEGIN

        ; detector average
        ; dont bother to take into account mask since only used to give a time axis

        array = TOTAL ( array, 1)

        IF output_info.rat THEN array = array/header.ndetectors ELSE factor = factor*header.ndetectors

        detector = 0

      ENDIF ELSE $
        array = array(element,*,*)

      CALL_PROCEDURE, procedure, data.time, data.endtime, $
                      REFORM(array, output_info.ysamples, header.npoints), $
                      data.lut, data.flag, output_chans(*,*,detector,*), $
                      factor*output_info.minimum, factor*output_info.maximum, $
                      header.count_cal, header.npoints, output_info.ysamples, $
                      output_info.elow, output_info.ehigh, $
                      header.tstart, header.tstop, $
                      XTITLE=header.xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
                      BAND_LABELS=band_labels, BUTTONS=output_info.button_values, $
                      SAVE_CC=save_cc, XMARGIN=xmargin, YMARGIN=ymargin, NO_XAXIS=no_xaxis, $
                      LOG=output_info.log, SHRINK=shrink

    ENDIF ELSE IF KEYWORD_SET(TIMELINE) THEN BEGIN

     CALL_PROCEDURE, procedure, data.time, data.endtime, $
                     REFORM(output_array(*,element,*), output_info.ysamples, header.npoints), $
                     data.lut, data.flag, output_chans(*,*,detector,*), $
                     output_info.minimum, output_info.maximum, $
                     header.count_cal, header.npoints, output_info.ysamples,  $
                     output_info.elow, output_info.ehigh, $
                     header.tstart, header.tstop, $
                     XTITLE=header.xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
                     BAND_LABELS=band_labels, BUTTONS=output_info.button_values, $
                     SAVE_CC=save_cc, XMARGIN=xmargin, YMARGIN=ymargin, NO_XAXIS=no_xaxis, $
                     LOG=output_info.log, SHRINK=shrink

   ENDIF ELSE BEGIN

     CALL_PROCEDURE, procedure, data.time, data.endtime, $
                     REFORM(output_array(element,0:output_info.ysamples-1,*), output_info.ysamples, header.npoints), $
                     data.lut, data.flag, output_chans(*,*,detector,*), $
                     output_info.minimum, output_info.maximum, $
                     header.count_cal, header.npoints, output_info.ysamples,  $
                     output_info.elow, output_info.ehigh, $
                     header.tstart, header.tstop, $
                     XTITLE=header.xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
                     BAND_LABELS=band_labels, BUTTONS=output_info.button_values, $
                     SAVE_CC=save_cc, XMARGIN=xmargin, YMARGIN=ymargin, NO_XAXIS=no_xaxis, $
                     LOG=output_info.log, SHRINK=shrink

   ENDELSE

 ENDIF ELSE BEGIN

    ; check if margins defined

    IF N_ELEMENTS(xmargin) EQ 0 THEN BEGIN

     xmargin = [10.0,16.0]

     ; get standard character sizes in normal coords

     x_chsize = get_chsize(/X)
     y_chsize = get_chsize(/Y)

     ; fix margins so that in postcript format plot size is 16 cm
     ; in postscript format plot xsize is 19.9 cm
     ; xmargin is measured in units of actual character size
     ; so xmargin required size is 3.9 cm or 3.9/19.9 in normal units
     ; current x margin size in normal units is x_chsize*TOTAL(xmargin)
     ; adjust xmargin by required factor

     xmargin = xmargin * (3.9/19.9) / (x_chsize*TOTAL(xmargin))

   ENDIF

   IF N_ELEMENTS(ymargin) EQ 0 THEN BEGIN

     ymargin = [0,0.5]

   ENDIF

   CALL_PROCEDURE, procedure, data.time, data.endtime, $
                   REFORM(output_array(element,0:output_info.ysamples-1,*), output_info.ysamples, header.npoints), $
                   data.lut, data.flag, output_chans(*,*,detector,*), $
                   output_info.minimum, output_info.maximum, $
                   header.count_cal, header.npoints, output_info.ysamples,  $
                   output_info.elow, output_info.ehigh, $
                   header.tstart, header.tstop, $
                   TITLE=output_info.ptitle(element), $
                   XTITLE=header.xtitle, YTITLE=ytitle, ZTITLE=ztitle, $
                   BAND_LABELS=band_labels, BUTTONS=output_info.button_values, $
                   SAVE_CC=save_cc, XMARGIN=xmargin, YMARGIN=ymargin, /NO_ERASE, /NO_MODES, NO_XAXIS=no_xaxis,$
                   LOG=output_info.log, SHRINK=shrink

  ENDELSE

END

