;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	IES_DO_WIDGET
;
; Purpose     :	Sets up draw widgets for IES_PLOT display.
;
; Explanation : Displays data from a number of detectors in different draw widgets.
;               Displays detector data in scrolling windows.
;               Displays full range of data from first detector.
;               Scrolls each detector plot in synchronization with cursor position in
;               full window plot.
;               Found display difference between machines.
;               Change size of scrolling windows as number of windows increases.
;               Fix size of full window.
;
; Use         : < ies_do_widget, col >
;
; Inputs      : col  : widget ID of parent widget.
;
; Opt. Inputs : None.
;
; Outputs     : Sets up draw widgets.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	ies_machine, ies_band_labels.
;
; Common      :	ies_draw_widget_block..
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 16/1/96
;
; Modified    :	Version 0.1, MKC, 16/9/96
;                 Added ies_machine.
;                 Fix size of full window display.
;                 Adjust placement algorithm.
;                 Added detector sort.
;                 Extended ies_draw_widget_block.
;                 Removed MOTION_EVENTS from ids.draw_ids(0).
;               Version 0.2, 20/1/97
;                 Added PROFILES output.
;                 Moved detector sort to ies extdata.
;                 Added full_only to output_info.
;                 Allowed for SHIFT in ANGLE/ANGLE plot.
;                 Added time box to SPECTRAL, LINEPLOT and PROFILES.
;               Version 0.3, 28/4/97
;                 Set maximum size of 10000 for draw window.
;               Version 0.4, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Corrected title of full window in DETECTOR vs SECTOR
;                 Allowed detector sum.
;               Version 0.5, 12/11/98
;                 Moved setting of band_labels from ies_calibrate.
;                 Added title_ids.
;               Version 0.6, 6/9/00
;                 Generalized grp0, grp1 algorithm.
;               Version 0.7, 08/02/03
;                 Added OVERLAY plot.
;                 Removed title_ids from PROFILES plot.
;               Version 0.8, 26/09/03
;                 Corrected labelling used to full_band for some plots.
;               Version 0.9, 06/10/03
;                 Made title[0] in TIMELINE dynamically resizeable.
;
; Version     :	Version 0.9, 06/10/03
;-
;**********************************************************

PRO ies_do_widget, col

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  ; set band labels

  band_labels = ies_band_labels(header, output_info, output_chans)

  ; choose plot type

  IF output_info.procname EQ 'SPECTRAL' OR output_info.procname EQ 'LINEPLOT' THEN BEGIN

    ; get machine dependent values

    ies_machine, output_info.noutputs, header.npoints, xsize, wsize, y_adjust, ysizes

    ; y draw view size
    ; NB if y size too big then will omit plots
    ;    if full_only gives 0th element

    IF output_info.full_only THEN ysize = ysizes(0) ELSE ysize = ysizes((output_info.noutputs-1)/2)

    IF output_info.format EQ 'ENERGY vs DETECTOR' OR output_info.format EQ 'SECTOR vs DETECTOR' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_detector) ) $
    ELSE IF output_info.format EQ 'ENERGY vs SECTOR' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_sector) ) $
    ELSE IF output_info.format EQ 'DETECTOR vs SECTOR' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_band) ) $
    ELSE IF output_info.format EQ 'PITCH ANGLE vs ENERGY' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_band) ) $
    ELSE IF output_info.format EQ 'GYRORADIUS vs ENERGY' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_band) )

    ; set up full draw widget
    ; the /FRAME makes the scroll windows and full window about the same size
    ; let draw window decide x size on its own (cant force it anyway)

    ids.draw_ids(0) = WIDGET_DRAW ( col, XSIZE=1100, YSIZE = 200, UVALUE='FULL', /FRAME, RETAIN=2 )

    IF NOT output_info.full_only THEN BEGIN

      ; set up time widget

      IF output_info.track THEN $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='MOVE CURSOR INTO FULL WINDOW ...', /DYNAMIC_RESIZE) $
      ELSE $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='CLICK MOUSE IN FULL WINDOW ...', /DYNAMIC_RESIZE )

       ; set up other widgets
      ; xsize, wsize specify the width and height of the drawing widget
      ; x_scroll_size, y_scroll_size specify the width and height of the view of the drawing widget
      ; to avoid the vertical scroll bars need to leave sufficient space for the horizontal scroll bars

      ; make row group

      row  = WIDGET_BASE ( col, /ROW)

      ; make column groups

      grp0 = WIDGET_BASE ( row, /COLUMN)
      grp1 = WIDGET_BASE ( row, /COLUMN)

      grp = grp0 ; set current group to be first column

      FOR k = 1, output_info.noutputs DO BEGIN

        ; check if need to put in second column

        IF k GT (output_info.noutputs+1)/2 THEN grp = grp1

        ; make column group

        ids.title_ids[k] = WIDGET_LABEL ( grp, VALUE=output_info.ptitle(k-1) )

        ; RETAIN=2 forces the widget to have backing store which means that the
        ; full window is maintained in store even though only part may be visible
        ; through the scroll window

        ids.draw_ids(k) = WIDGET_DRAW ( grp, XSIZE = wsize, YSIZE = ysize, UVALUE=STRTRIM(k,1), $
                               X_SCROLL_SIZE = xsize, Y_SCROLL_SIZE = ysize+y_adjust, RETAIN=2)

      ENDFOR

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'PROFILES' THEN BEGIN

    ; x/y draw view size

    CASE 1 OF
      output_info.noutputs EQ 1  : BEGIN
             num   = 1
             xsize = 900
             ysize = 600
           END
      output_info.noutputs EQ 2  : BEGIN
             num   = 2
             xsize = 540
             ysize = 300
           END
      output_info.noutputs EQ 3 OR output_info.noutputs EQ 6  : BEGIN
             num   = 3
             xsize = 350
             ysize = 220
           END
      output_info.noutputs EQ 4 OR output_info.noutputs EQ 7 OR output_info.noutputs EQ 8 : BEGIN
             num   = 4
             xsize = 260
             ysize = 200
           END
      output_info.noutputs EQ 5 OR output_info.noutputs EQ 9 OR output_info.noutputs EQ 10 : BEGIN
             num   = 5
             xsize = 210
             ysize = 200
           END
      ELSE: BEGIN
             num   = 6
             xsize = 170
             ysize = 150
           END
    ENDCASE

    IF output_info.format EQ 'ENERGY vs DETECTOR' OR output_info.format EQ 'SECTOR vs DETECTOR' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_detector) ) $
    ELSE IF output_info.format EQ 'DETECTOR vs SECTOR' OR output_info.format EQ 'ENERGY vs SECTOR' THEN $$
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_sector) ) $$
    ELSE $$
      dummy = WIDGET_LABEL ( col, VALUE='Full window, energy band '+STRTRIM(header.nbands-1,2))

    ; set up full draw widget
    ; the /FRAME makes the scroll windows and full window about the same size
    ; let draw window decide x size on its own (cant force it anyway)

    ids.draw_ids(0) = WIDGET_DRAW ( col, XSIZE=1100, YSIZE = 200, UVALUE='FULL', /FRAME, RETAIN=2 )

    IF NOT output_info.full_only THEN BEGIN

      ; set up time widget

      IF output_info.track THEN $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='MOVE CURSOR INTO FULL WINDOW ...', /DYNAMIC_RESIZE) $
      ELSE $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='CLICK MOUSE IN FULL WINDOW ...', /DYNAMIC_RESIZE )

      ; make row group

      row  = WIDGET_BASE ( col, /ROW)

      ; make column groups

      grp0 = WIDGET_BASE ( row, /COLUMN)
      grp1 = WIDGET_BASE ( row, /COLUMN)
      grp2 = WIDGET_BASE ( row, /COLUMN)
      grp3 = WIDGET_BASE ( row, /COLUMN)
      grp4 = WIDGET_BASE ( row, /COLUMN)
      grp5 = WIDGET_BASE ( row, /COLUMN)

      grp = grp0 ; set current group to be first column

      FOR k = 1, output_info.noutputs DO BEGIN

        CASE (k-1) MOD num OF
          0  : grp = grp0
          1  : grp = grp1
          2  : grp = grp2
          3  : grp = grp3
          4  : grp = grp4
          5  : grp = grp5
          ELSE :
        ENDCASE

        ; RETAIN=2 forces the widget to have backing store which means that the
        ; full window is mainrtained in store even though only part may be visible
        ; through the scroll window

        ids.draw_ids(k) = WIDGET_DRAW ( grp, XSIZE = xsize, YSIZE = ysize, UVALUE=STRTRIM(k,1), RETAIN=2)

      ENDFOR

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'OVERLAY' THEN BEGIN

    ; x/y draw view size

    xsize = 1100
    ysize = 800

    IF output_info.format EQ 'ENERGY vs DETECTOR' OR output_info.format EQ 'SECTOR vs DETECTOR' THEN $
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_detector) ) $
    ELSE IF output_info.format EQ 'DETECTOR vs SECTOR' OR output_info.format EQ 'ENERGY vs SECTOR' THEN $$
      dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_sector) ) $$
    ELSE $$
      dummy = WIDGET_LABEL ( col, VALUE='Full window, energy band '+STRTRIM(header.nbands-1,2))

    ; set up full draw widget
    ; the /FRAME makes the scroll windows and full window about the same size
    ; let draw window decide x size on its own (cant force it anyway)

    ids.draw_ids(0) = WIDGET_DRAW ( col, XSIZE=1100, YSIZE = 200, UVALUE='FULL', /FRAME, RETAIN=2 )

    IF NOT output_info.full_only THEN BEGIN

      ; set up time widget

      IF output_info.track THEN $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='MOVE CURSOR INTO FULL WINDOW ...', /DYNAMIC_RESIZE) $
      ELSE $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='CLICK MOUSE IN FULL WINDOW ...', /DYNAMIC_RESIZE )

      ; make row group

      grp  = WIDGET_BASE ( col, /ROW)

      ; RETAIN=2 forces the widget to have backing store which means that the
      ; full window is mainrtained in store even though only part may be visible
      ; through the scroll window

      ids.draw_ids[1] = WIDGET_DRAW ( grp, XSIZE = xsize, YSIZE = ysize, UVALUE=STRTRIM('1-?',1), RETAIN=2)

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN

    IF output_info.full_detector EQ header.ndetectors THEN $
      atitle = ' detector averaged' $
    ELSE $
      atitle = output_info.ptitle(output_info.full_detector)

    IF output_info.full_sector GT 0 THEN $
      atitle = atitle + ' Sc' + STRTRIM(output_info.full_sector-1,2) $
    ELSE $
      atitle = atitle + ' spin averaged'

    dummy = WIDGET_LABEL ( col, VALUE='Full window ' + atitle )

    ; set up full draw widget
    ; the /FRAME makes the scroll windows and full window about the same size
    ; let draw window decide x size on its own (cant force it anyway)

    ids.draw_ids(0) = WIDGET_DRAW ( col, XSIZE=1024, YSIZE=200, UVALUE='FULL', /FRAME, RETAIN=2 )

    IF NOT output_info.full_only THEN BEGIN

      ; set up time widget

      IF output_info.track THEN $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='MOVE CURSOR INTO FULL WINDOW ...', /DYNAMIC_RESIZE) $
      ELSE $
        ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='CLICK MOUSE IN FULL WINDOW ...', /DYNAMIC_RESIZE )

      ; set image size

      xsize = 128
      ysize = 126

      ; set up other widgets
      ; xsize, wsize specify the width and height of the drawing widget

      FOR k = 1, output_info.noutputs DO BEGIN

        ; make new row group

        IF (k MOD 8) EQ 1 THEN row  = WIDGET_BASE ( col, /ROW)

        ; make column group

        cl = WIDGET_BASE ( row, /COLUMN)

        ; change band labels if data resampled

        ids.title_ids[k] = WIDGET_LABEL ( cl, VALUE = band_labels(k-1), /DYNAMIC_RESIZE )

        ; RETAIN=2 forces the widget to have backing store

        ids.draw_ids[k] = WIDGET_DRAW ( cl, XSIZE = xsize, YSIZE = ysize, UVALUE=STRTRIM(k,1), RETAIN=2)

      ENDFOR

    ENDIF

  ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN

    dummy = WIDGET_LABEL ( col, VALUE='Full window ' + output_info.ptitle(output_info.full_detector) )

    ; set up full draw widget
    ; the /FRAME makes the scroll windows and full window about the same size
    ; let draw window decide x size on its own (cant force it anyway)

    ids.draw_ids(0) = WIDGET_DRAW ( col, XSIZE=1024, YSIZE=200, UVALUE='FULL', /FRAME, RETAIN=2 )

    IF NOT output_info.full_only THEN BEGIN

      ; print channels
      ; NB intended that energy went in here but band_labels contains sectors for TIMELINE
      ; needs rationalization

      ;ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='Detector vs Azimuth plots : ' + band_labels(header.nbands-1) )
      ids.title_ids[0] = WIDGET_LABEL ( col, VALUE='Detector vs Azimuth plots : ', /DYNAMIC_RESIZE)

      ; set image size

      xsize = 128
      ysize = 126

      ; set up other widgets
      ; xsize, wsize specify the width and height of the drawing widget

      FOR k = 1, output_info.noutputs DO BEGIN

        ; make new row group

        IF (k MOD 8) EQ 1 THEN row  = WIDGET_BASE ( col, /ROW)

        ; make column group

        cl = WIDGET_BASE ( row, /COLUMN)

        ids.title_ids[k] = WIDGET_LABEL ( cl, VALUE='00:00:00')

        ; RETAIN=2 forces the widget to have backing store

        ids.draw_ids(k) = WIDGET_DRAW ( cl, XSIZE = xsize, YSIZE = ysize, UVALUE=STRTRIM(k,1), RETAIN=2)

      ENDFOR

    ENDIF

  ENDIF

END

