;+
; Project     :	POLAR - CEPPAD
;
; Name        :	IES_DO_X_AXIS
;
; Purpose     :	Sets up X axis for display.
;
; Explanation :	Plots time along x axis. Picks suitable ticks for display.
;
; Use         : < ies_do_x_axis, xticks, xminor, xtickname, xtickv, ylo, XTITLE=xtitle, NO_XAXIS=no_xaxis >
;
; Inputs      : xticks     : INT no. of major tick intervals
;               xminor     : INT no. of minor tick intervals
;               xtickname  : STRARR tick labels
;               xtickv     : FLTARR tick positions in data coords.
;               ylo        : FLOAT minimum y axis value.
;
; Opt. Inputs : None.
;
; Outputs     : 
;
; Opt. Outputs:	None.
;
; Keywords    : XTITLE      : STRING Plot x title.
;               NO_XAXIS    : INT flag preventing x axis display.
;
; Calls       :	get_chsize, lshell_plot.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 31/1/96
;
; Modified    :	Version 0.1, MKC, 21/10/06
;                 Replaced charsize with !P.CHARSIZE.
;                 Adjusted position of x title.
;               Version 0.2, MKC, 13/2/97
;                 Added ylo to argument list to cope with non-zero y axis start.
;
; Version     :	Version 0.2, 13/2/97
;-
;**********************************************************


PRO ies_do_x_axis, xticks, xminor, xtickname, xtickv, ylo, XTITLE=xtitle, NO_XAXIS=no_xaxis

  IF NOT KEYWORD_SET(xtitle) THEN xtitle = ''

  ; get character sizes in normal coords

  x_chsize = get_chsize( /X )
  y_chsize = get_chsize( /Y ) 

 ; get normal coords of ticks

  vector = CONVERT_COORD ( xtickv, REPLICATE(ylo,xticks+1), /DATA, /TO_NORMAL )

  xtickv_normal = REFORM(vector(0,*)) 
  ytickv_normal = REFORM(vector(1,*)) 

  ; draw minor tick marks
  ; there are xminor minor tick spaces between the major ticks (xminor-1 minor tick marks)
  ; should always be one major tick interval
  ; add major tick at either end

  IF xticks GE 1 THEN BEGIN

    temp = [2*xtickv_normal(0)-xtickv_normal(1), xtickv_normal, 2*xtickv_normal(xticks)-xtickv_normal(xticks-1)]

    xtickv_minor = INTERPOLATE ( temp, findgen(1+(xticks+2)*xminor)/xminor )

    list = WHERE (xtickv_minor GE !P.POSITION(0) AND xtickv_minor LE !P.POSITION(2), count)

    IF count GT 0 THEN BEGIN

       xtickv_minor = xtickv_minor(list)

      FOR k = 0, N_ELEMENTS(xtickv_minor)-1 DO $
        PLOTS, [xtickv_minor(k), xtickv_minor(k)], [ytickv_normal(0),ytickv_normal(0)-y_chsize*0.2], /NORMAL

    ENDIF

  ENDIF

  ; draw major tick marks

  FOR k = 0, N_ELEMENTS(xtickv_normal)-1 DO $
    PLOTS, [xtickv_normal(k), xtickv_normal(k)], [ytickv_normal(0),ytickv_normal(0)-y_chsize*0.4], /NORMAL  

  IF NOT KEYWORD_SET(no_xaxis) THEN BEGIN

    ; puts the time axis below the plot window

    xtickv_normal = xtickv_normal - STRLEN(xtickname)*x_chsize/2.0
    ytickv_normal = ytickv_normal - 1.4*y_chsize
 
    ; explicitly write in tick values

    XYOUTS, xtickv_normal, ytickv_normal, XTICKNAME, /NORMAL, CHARSIZE=!P.CHARSIZE

    ; write x title

    XYOUTS, (!P.POSITION(0) < xtickv_normal(0)) - x_chsize*(STRLEN(xtitle) + 0.5), ytickv_normal(0), xtitle, /NORMAL, CHARSIZE=!P.CHARSIZE

  ENDIF

END
