;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_DRAW_LUTS
;
; Purpose     :	Does plot of pedestal shifts.
;
; Explanation :	Does not assume that data is evenly sampled in time.
;
; Use         : < ies_draw_luts, x, luts, xmargin, ymargin, datatype, /NO_XAXIS >
;
; Inputs      : x            : DBLARR(npoints) times for each sample in TAI format
;               luts         : BYTARR(npoints) LUT values at each sample point
;               xmargin      : INTARR(2) character size of left and right margins.
;               ymargin      : INTARR(2) character size of top and bottom margins.
;               datatype     : STRING containing instrument i.e IES, IPS, HISTe, HISTp.
;
; Opt. Inputs : None.
;
; Outputs     : Draws plot on current graphics device.
;
; Opt. Outputs:	None.
;
; Keywords    : /NO_XAXIS   = controls whether x axis displayed.
;               /SHRINK     = Allows time axis to be adjusted.
;               TSTART      = DOUBLE start time in seconds for display
;               TEND        = DOUBLE end time in seconds for display
;
; Calls       :	ies_do_axes.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 11/2/97
;
; Modified    :	Version 0.1, 9/5/97
;                 Modified axis labelling.
;               version 0.2, 18/8/97
;                 Added HIST LUT labels.
;               version 0.3, 9/10/98
;                 Added ncolors to argument list.
;                 Added SHRINK, TSTART and TEND keywords.
;               Version 0.4, 10/5/00
;                 Added RAPID_IES datatype.
;               Version 0.5, 18/8/00
;                 Adjusted to cope with -1=Unknown LUT
;               Version 0.6, 27/2/01
;                 Removed bug in removing time offset.
;               Version 0.7, 09/07/02
;                 Removed ncolors from argument list.
;               Version 0.8, 26/09/03
;                 Used new routine for instrument test.
;
; Version     :	Version 0.8, 26/09/03
;-
;**********************************************************

PRO ies_draw_luts, xt, y, xmargin, ymargin, datatype, NO_XAXIS=no_xaxis, $
                   SHRINK=shrink, TSTART=tstart, TEND=tend

  ; get dimensions

  npoints = (SIZE(y))(1)

  ; set up time_start and time_end

  IF KEYWORD_SET(tstart) THEN time_start = tstart ELSE time_start = xt(0)
  IF KEYWORD_SET(tend  ) THEN time_end   = tend   ELSE time_end   = xt(N_ELEMENTS(xt)-1)

  IF ies_instrument(datatype,/RAPID) AND ies_instrument(datatype,/IES) THEN BEGIN

    ; set up axes

    ies_do_axes, time_start, time_end, -1.5, 3.5, xmargin, ymargin, NO_XAXIS=no_xaxis, $
                 /NO_ERASE, XTITLE='UT:', YTITLE='Integration mode', /NO_YAXIS, SHRINK=shrink

    ; get character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    ; set up LUT no.s

    list = WHERE(y NE -1, count)
    IF count GT 0 THEN y[list] = y[list] MOD 4

    AXIS, YAXIS=0, YTICKLEN=-x_chsize, YTICKNAME=['X','2 us', '5 us', '15 us', '50 us'], YTICKS=4, YTICKV=[-1,0,1,2,3], CHARSIZE=!P.CHARSIZE*0.8

  ENDIF ELSE IF ies_instrument(datatype,/IES) THEN BEGIN

    ; set up axes

    ies_do_axes, time_start, time_end, -1.5, 2.5, xmargin, ymargin, NO_XAXIS=no_xaxis, $
                 /NO_ERASE, XTITLE='UT:', YTITLE='Integration mode', /NO_YAXIS, SHRINK=shrink

    ; get character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    ; set up LUT no.s
    ; -1 = 100 us L
    ;  0 = 10 us L
    ;  1 = 2 us L
    ;  2 = 2 us S

    list = WHERE(y EQ 0, count)
    IF count GT 0 THEN y(list) = -1

    list = WHERE(y EQ 12, count)
    IF count GT 0 THEN y(list) = -1

    y = y MOD 3

    AXIS, YAXIS=0, YTICKLEN=-x_chsize, YTICKNAME=['100 L', '10 L', '2 L', '2 S'], YTICKS=3, YTICKV=[-1,0,1,2], CHARSIZE=!P.CHARSIZE*0.8

  ENDIF ELSE IF ies_instrument(datatype,/HIST) THEN BEGIN

    ; set up axes

    ies_do_axes, time_start, time_end, -1.5, 2.5, xmargin, ymargin, NO_XAXIS=no_xaxis, $
                 /NO_ERASE, XTITLE='UT:', YTITLE='Integration mode', /NO_YAXIS

    ; get character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    ; set up LUT no.s

    y = ([0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 2,  0, 1, 2, 0,  0, 1, 2, 0,  0, 2, 2, 0] ) (y)

    AXIS, YAXIS=0, YTICKLEN=-x_chsize, YTICKS=3, YTICKV=[0,1,2], CHARSIZE=!P.CHARSIZE*0.8, $
          YTICKNAME=['ABC', 'BC', 'HBC' ]

  ENDIF ELSE BEGIN

    ; set up axes

    ies_do_axes, time_start, time_end, 0, MAX(y), xmargin, ymargin, NO_XAXIS=no_xaxis, $
                 /NO_ERASE, XTITLE='UT:', YTITLE='Integration mode', /NO_YAXIS

    ; get character sizes in normal coords

    x_chsize = get_chsize(/X)
    y_chsize = get_chsize(/Y)

    AXIS, YAXIS=0, YTICKLEN=-x_chsize, CHARSIZE=!P.CHARSIZE*0.8

  ENDELSE

  ; remove offset in internal time

  x = xt - time_start

  OPLOT, x, y

  ; overplot axes again to neaten up
  ; NB use dont use data coords because x axis extended in data coords
  ; NB dont want to use plot again because of multiple plots

  PLOTS, [ !X.WINDOW(0), !X.WINDOW(1), !X.WINDOW(1), !X.WINDOW(0), !X.WINDOW(0)], $
         [ !Y.WINDOW(0), !Y.WINDOW(0), !Y.WINDOW(1), !Y.WINDOW(1), !Y.WINDOW(0)], /NORMAL

  ; plot title

  title = 'LUTs'

  XYOUTS, (!P.POSITION(2)+!P.REGION(2) - STRLEN(title)*x_chsize)/2.0, !P.POSITION(3)-3*y_chsize, title, /NORMAL, CHARSIZE=1.5*!P.CHARSIZE

END
