;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	IES_DRAW_WIDGET
;
; Purpose     :	Sets up widgets used by IES_PLOT.
;
; Explanation : Sets up draw widgets and menu items for IES_PLOT.
;
; Use         : < base = ies_draw_widget() >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : base : Widget ID of topmost widget.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	ies_logo, ies_do_widget.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 5/6/95
;
; Modified    :	Version 0.1, MKC, 24/8/95
;                 Modified PS output so always 5 plots per page.
;               Version 0.2, MKC, 16/1/96
;                 Made change to new data format.
;               Version 0.3, MKC, 16/9/96
;                 Added txt print output option.
;                 Changed version number.
;                 Changed SHRINK to CLEAN.
;                 Extended ies_draw_widget_block.
;               Version 0.4, MKC, 7/11/96
;                 Removed SW version.
;               Version 0.5, 8/11/96
;                 Added PROFS button.
;                 Added RAT button.
;                 Added SUB BUTTON.
;               Version 0.6, 16/1/97
;                 Renamed clean to shift.
;               Version 0.7, 17/1/97
;                 Modified sensitivity test for SHIFT and SUB buttons to use presence of lut map tag.
;                 Removed PROFS button.
;                 Removed SHIFT button in ANGLE/ANGLE plot.
;                 Changed draw_ids array to noutputs+2
;               Version 0.8, 1/4/97
;                 Modified button placement.
;               Version 0.9, 15/4/97
;                 Made DIV, CAL and RAT buttons sensitive always.
;               Version 1.0, 22/8/97
;                 Removed LUT MAP test.
;               Version 1.1, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;               Version 1.2, 12/11/98
;                 Added title_ids.
;               Version 1.3, 22/2/99
;                 Added eloid and ehiid.
;               Version 1.4, 17/3/99
;                 removed zero elo, ehi, min, max.
;               Version 1.5, 15/9/99
;                 Set up so that logo widget generates event after 4 sec.
;           	Version 1.6, MKC, 28/5/00
;                 Added version no check for SENSITIVE keyword.
;               Version 1.7, 10/12/00
;                 Added PNG output
;               Version 1.8, 12/03/02
;                 Changed so that some buttons exclusive.
;               Version 1.9, 08/02/03
;                 Added OVERLAY plots.
;               Version 2.0, 26/09/03
;                 Desensitized SUBTRACT button if data has already been cleaned.
;                 Desensitized SUBTRACT and CLEAN buttons if data not POLAR/IES.
;                 Added NPITCHS input.
;                 Used new routine for instrument test.
;
; Version     :	Version 2.0, 26/09/03
;-
;**********************************************************

FUNCTION ies_draw_widget

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ;     .draw_ids      = LONARR(noutputs) widget IDs of each of the draw windows
   ;     .title_ids     = LONARR(noutputs) widget IDs of each of the draw window titles
   ;     .maxid         = ID of text widget for maximum input
   ;     .minid         = ID of text widget for minimum input
   ;     .ehiid         = ID of text widget for hi input
   ;     .eloid         = ID of text widget for lo input
   ;     .infoid        = ID of text widget for information display
   ;     .logoid        = ID of logo
   ;     .rngid         = ID of text widget for range input
   ;     .nptchid       = ID of text widget for npitchs input
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  ; explicitly kill multiple plots

  !P.MULTI = 0

  ; set up ids structure

  ids = { draw_ids:LONARR(output_info.noutputs+1), title_ids:LONARR(output_info.noutputs+1), $
          maxid:0L, minid:0L, ehiid:0L, eloid:0L, infoid:0L, logoid:0L, rngid:0L, nptchid:0L }

  ; set up display configuration
  ; should have windows as high as possible given constaints of maximum no. of windows
  ; and graphics device.
  ; Should avoid having y scroll bars

  ; set up main widget

  base = WIDGET_BASE ( TITLE = 'RAL DISPLAY TOOL', /COLUMN)

  ; set up 1st row

  row = WIDGET_BASE ( base, /ROW )

  ; set up logo

  logo   = ies_logo()
  dims   = SIZE(logo)
  ids.logoid = WIDGET_DRAW ( row, XSIZE=dims(1), YSIZE=dims(2), UVALUE='LOGO', RETAIN=2, /BUTTON_EVENTS)

  ; set up info display

  ids.infoid = WIDGET_TEXT ( row, VALUE=header.title, XSIZE=68, /FRAME,/scroll)

  ; set up buttons

  button_list = WHERE(output_info.button_labels NE '', button_count)

  IF button_count GT 0 THEN button_ids = LONARR(button_count)

  nonexclusive_list = WHERE(STRPOS(output_info.button_labels[button_list],'/') NE 0, $
                               nonexclusive_count)

  exclusive_list = WHERE(STRPOS(output_info.button_labels[button_list],'/') EQ 0, $
                               exclusive_count)

  ; set up nonexclusive buttons

  IF nonexclusive_count GT 4 THEN nrows = 3 ELSE nrows = 2

  tog  = WIDGET_BASE ( row, ROW=nrows, /NONEXCLUSIVE, /GRID_LAYOUT)

  ; set up fixed button labels

  logid = WIDGET_BUTTON ( tog, VALUE='LOG', $    ;The label for the button.
                                 UVALUE='LOG' )    ;The User Value for the button.

  trackid = WIDGET_BUTTON ( tog, VALUE='TRACK', $    ;The label for the button.
                                   UVALUE='TRACK' )       ;The User Value for the button.

  FOR b = 0, nonexclusive_count-1 DO button_ids(b) = WIDGET_BUTTON ( tog, $
    VALUE=output_info.button_labels(button_list[nonexclusive_list[b]]), $
    UVALUE=output_info.button_labels(button_list[nonexclusive_list[b]]) )

  ; set up program button_labels

  IF exclusive_count GT 0 THEN BEGIN

    ex_tog  = WIDGET_BASE ( row, ROW=nrows, /EXCLUSIVE, /GRID_LAYOUT)

    FOR b = 0, exclusive_count-1 DO button_ids(nonexclusive_count+b) = WIDGET_BUTTON ( ex_tog, $
      VALUE=output_info.button_labels(button_list[exclusive_list[b]]), $
      UVALUE=output_info.button_labels(button_list[exclusive_list[b]]) )

  ENDIF

  ; set up 2nd row

  row = WIDGET_BASE ( base, /ROW )

  ; set up pull down menu

  items = [  '1\FILE',       $
             '0\DISMISS',    $
             '0\SAVE',       $
             '0\RETURN',     $
             '0\COLOUR',     $
             '3\PRINT',      $
             '0\BWPS',       $
             '0\PS',         $
             '0\PNG',        $
             '0\GIF',        $
             '0\WIN',        $
             '2\TXT'         ]

  dummy = CW_PDMENU ( row, items, UVALUE='FILE', /RETURN_NAME)

  ; set up fixed button labels

  tog  = WIDGET_BASE ( row, /ROW, /NONEXCLUSIVE)

  calid = WIDGET_BUTTON ( tog, VALUE='CAL', $    ;The label for the button.
                               UVALUE='CAL' )    ;The User Value for the button.

  divid = WIDGET_BUTTON ( tog, VALUE='DIV', $    ;The label for the button.
                               UVALUE='DIV' )    ;The User Value for the button.

  ratid = WIDGET_BUTTON ( tog, VALUE='RAT', $    ;The label for the button.
                               UVALUE='RAT' )    ;The User Value for the button.

  subid = WIDGET_BUTTON ( tog, VALUE='SUB', $    ;The label for the button.
                                 UVALUE='SUB' )       ;The User Value for the button.

  shiftid = WIDGET_BUTTON ( tog, VALUE='SHIFT', $    ;The label for the button.
                                 UVALUE='SHIFT' )       ;The User Value for the button.

  ; set max and min values to be non-adjustable ie +ve

  dummy  = WIDGET_BUTTON ( row, UVALUE='MIN', VALUE='MIN')

  ids.minid = WIDGET_TEXT ( row, VALUE=STRTRIM(STRING(FORMAT='(G9.3)',output_info.minimum),2), $
                UVALUE='MINIMUM', /FRAME, XSIZE=9)

  dummy  = WIDGET_BUTTON ( row, UVALUE='MAX', VALUE='MAX')

  ids.maxid = WIDGET_TEXT ( row, VALUE=STRTRIM(STRING(FORMAT='(G9.3)',output_info.maximum),2), $
                UVALUE='MAXIMUM', /FRAME, XSIZE=9)

  ; set hi and lo values to be non-adjustable ie +ve

  dummy  = WIDGET_BUTTON ( row, UVALUE='ELO', VALUE='ELO')

  ids.eloid = WIDGET_TEXT ( row, VALUE=STRTRIM(STRING(FORMAT='(G9.3)',output_info.elow),2), $
                UVALUE='ELOW', /FRAME, XSIZE=9)

  dummy  = WIDGET_BUTTON ( row, UVALUE='EHI', VALUE='EHI')

  ids.ehiid = WIDGET_TEXT ( row, VALUE=STRTRIM(STRING(FORMAT='(G9.3)',output_info.ehigh),2), $
               UVALUE='EHIGH', /FRAME, XSIZE=9)

  ; set up range widget

  IF output_info.procname EQ 'PROFILES' OR output_info.procname EQ 'OVERLAY' THEN BEGIN

    dummy  = WIDGET_LABEL ( row, VALUE='RNG')

    ids.rngid = WIDGET_TEXT ( row, VALUE=STRTRIM(STRING(FORMAT='(I5)',output_info.rng),2), UVALUE='RANGE', /EDITABLE, XSIZE=5, /FRAME)

  ENDIF

  ; set up npitchs widget

  IF output_info.format EQ 'PITCH ANGLE vs ENERGY' THEN BEGIN

    dummy  = WIDGET_LABEL ( row, VALUE='NPA')

    ids.nptchid = WIDGET_TEXT ( row, VALUE=STRTRIM(STRING(FORMAT='(I3)',output_info.npitchs),2), UVALUE='NPITCHS', /EDITABLE, XSIZE=5, /FRAME)

  ENDIF

  ; make column group for full window

  col = WIDGET_BASE ( base, /COLUMN)

  ies_do_widget, col

  WIDGET_CONTROL, base, /REALIZE

  ; set program button_labels

  IF button_count GT 0 THEN BEGIN

    FOR b = 0, button_count-1 DO WIDGET_CONTROL, button_ids(b), SET_BUTTON=output_info.button_values(button_list(b))

  ENDIF

  ; set text widgets

  IF !VERSION.release GT '5.0.3' AND output_info.min EQ 0 THEN WIDGET_CONTROL, ids.minid, SENSITIVE=0

  IF output_info.min NE 0 THEN WIDGET_CONTROL, ids.minid, /EDITABLE

  IF !VERSION.release GT '5.0.3' AND output_info.max EQ 0 THEN WIDGET_CONTROL, ids.maxid, SENSITIVE=0

  IF output_info.max NE 0 THEN WIDGET_CONTROL, ids.maxid, /EDITABLE

  IF !VERSION.release GT '5.0.3' AND output_info.elo EQ 0 THEN WIDGET_CONTROL, ids.eloid, SENSITIVE=0

  IF output_info.elo NE 0 THEN WIDGET_CONTROL, ids.eloid, /EDITABLE

  IF !VERSION.release GT '5.0.3' AND output_info.ehi EQ 0 THEN WIDGET_CONTROL, ids.ehiid, SENSITIVE=0

  IF output_info.ehi NE 0 THEN WIDGET_CONTROL, ids.ehiid, /EDITABLE

  ; set fixed button_labels

  WIDGET_CONTROL, calid, SET_BUTTON=output_info.cal

  WIDGET_CONTROL, logid, SET_BUTTON=output_info.log

  WIDGET_CONTROL, divid, SET_BUTTON=output_info.div

  WIDGET_CONTROL, ratid, SET_BUTTON=output_info.rat

  WIDGET_CONTROL, shiftid, SET_BUTTON=output_info.shift

  ; check if POLAR IES data and cleanable and subtractable

  cleanable = ies_instrument(header.datatype,/IES) AND ies_instrument(header.datatype,/POLAR) AND header.clean EQ 0 AND header.subtract EQ 0

  ; desensitize CLEAN and SUBTRACT buttons if not applicable

  IF !VERSION.release GT '5.0.3' THEN $
    WIDGET_CONTROL, shiftid, SENSITIVE=cleanable AND output_info.procname NE 'LINEPLOT' AND $
                output_info.procname NE 'ANGLE/ANGLE'

  WIDGET_CONTROL, subid, SET_BUTTON=output_info.subtract

  IF !VERSION.release GT '5.0.3' THEN $
    WIDGET_CONTROL, subid, SENSITIVE=cleanable AND output_info.procname NE 'LINEPLOT'

  WIDGET_CONTROL, trackid, SET_BUTTON=output_info.track

  WIDGET_CONTROL, ids.draw_ids(0), DRAW_MOTION_EVENTS=output_info.track, DRAW_BUTTON_EVENTS=1-output_info.track

  IF exclusive_count GT 0 THEN BEGIN

    WIDGET_CONTROL, button_ids[nonexclusive_count], SET_BUTTON=1

    output_info.button_values[button_list[exclusive_list[0]]] = 1

  ENDIF

  ; set up logo

  WIDGET_CONTROL, ids.logoid, GET_VALUE=win

  WSET, win

  TV, logo

  ; make logo widget generate event

;  WIDGET_CONTROL, ids.logoid, TIMER=4.0

  ; return ID of widget

  RETURN, base

END
