;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_EFFICIENCY
;
; Purpose     :	Applies energy efficiency calibration.
;
; Explanation :	Applies energy efficiency calibration depending on LUT mode and reverses calibration.
;
; Use         : < ies_efficiency, lut_list, energy_efficiency, output_array >
;
; Inputs      : lut_list          : STRUCTURE ARRAY containing LUT run info.
;               energy_efficiency : FLTARR(nbands, nluts) efficiency factor for each energy in each LUT.
;               output_array      : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 18/8/97
;                 Adapted from ies_cal.
;
; Modified    :	Version 0.1, 19/8/00
;                 Explicitly allowed for LUT=-1
;              	Version 0.2, 10/09/01
;                 Extended loop variables to LONG.
;               Version 0.3, 24/10/03
;                 Changed to lut_list argument.
;
; Version     :	Version 0.3, 24/10/03
;-
;**********************************************************

PRO ies_efficiency, lut_list, energy_efficiency, output_array, REVERSE=reverse

  ; apply count calibration i.e convert from counts to counts/sr/cm2

  ; loop through runs

  FOR l = 0L, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut

    lut = lut_list[l].lut

    ; loop through samples

    FOR k = lut_list[l].st, lut_list[l].en DO FOR s = 0L, N_ELEMENTS(output_array(*,0,0,0))-1 DO FOR d = 0L, N_ELEMENTS(output_array(0,*,0,0))-1 DO BEGIN

      ; apply energy efficiency

      IF KEYWORD_SET(reverse) THEN BEGIN

        ; reverse calibration for each energy and each detector for this LUT

        output_array(s,d,*,k) = output_array(s,d,*,k) * energy_efficiency(*,lut)

      ENDIF ELSE BEGIN

        ; do calibration for each energy and each detector for this LUT

        output_array(s,d,*,k) = output_array(s,d,*,k)  / energy_efficiency(*,lut)

      ENDELSE

    ENDFOR

  ENDFOR

END
