;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	IES_EXTDATA
;
; Purpose     :	Extracts the requested data from an IES data array
;
; Explanation : Extracts the specified subset of data from a larger data array.
;               Allows averaging over energy bands.
;               If error then returns error message and 0 value.
;
; Use         : < ies_extdata, info_st, bands, input_header, input_data, full_detector, output_header, output_data, errmsg, $
;                              /FULL_DETECTOR_ONLY >
;
; Inputs      : info_st       : structure containing the parameters controlling the data extraction
;               bands         : structure containing energy band info
;               input_header  : structure containing data header info
;               input_data    : structure containing input data
;               full_detector : INT detector index for full display.
;
; Opt. Inputs : None.
;
; Outputs     : output_header : structure containing output data header info
;               output_data   : structure containing output data
;               full_detector : INT detector index for full display.
;               errmsg        : blank if no errors
;
; Opt. Outputs:	array : Applies time interval selection.
;
; Keywords    : A         : Secondary array to extract list from.
;               B         : Secondary array to extract list from.
;               C         : Secondary array to extract list from.
;               D         : Secondary array to extract list from.
;               E         : Secondary array to extract list from.
;               FULL_DETECTOR_ONLY : Flag indicating only full detector used.
;               EPHEM     : Structure array containing ephemeris data.
;               AT        : Structure array containing attitude data.
;               MF        : Structure array containing magnetic field in S/C coords.
;               MODES     : STRARR specifying HIST modes allowed.
;
; Calls       :	ies_header_structure, ies_gse_to_sc.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Command preparation.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 5/6/95
;
; Modified    : 0.1 31/7/95 Corrected input_axis manipulation.
;               0.2 8/8/95  Changed axis to keyword ARRAY.
;               0.3 7/12/95 Added sample averaging i.e spin averaging
;               0.4 16/1/96 Removed MAXCOUNTS option. Added ERRMSG.
;               0.5 29/2/96 Adapted to use internal time format.
;               0.6 1/11/96 Changed processing of full_detector
;               0.7 7/11/96 Set up clean and pedestal tags.
;               0.8 14/11/96 Set up div, rat and cal tags.
;               0.9 16/1/97 New pedestals and lut_map tags.
;                           Modified so that full detector and use_dets index into sorted detector list.
;                           Added full_detector_only to argument list.
;                           Shifted output_header.tstart and tstop to be data times rather than selected times.
;               1.0 14/4/97 Removed redundant code.
;                           Take into account channel size if dealing with cleaned data.
;               1.1 18/8/97 Added energy efficiency.
;                           Removed .pedestals and .lut_map header tags.
;               1.2 10/10/97
;                           Added energy_calibration offset.
;                           Put back .lut_map header tag.
;               1.3 30/10/97
;                           Added check for energy_cal and count_cal tags.
;               1.4 28/11/97
;                           Set time limits to requested rather than actual limits.
;               1.5 26/08/98
;                           Fixed bug with IPS detectors.
;               1.6 8/10/98
;                           Added in magnetic field and attitude.
;               1.7 28/10/98
;                           Allowed detector sum. Allowed only certain modes to be selected for HIST.
;               1.8 24/11/98
;                           Added explicit time overlap check for ephem and mf.
;               1.9 25/11/98
;                           Corrected IPS offset before 6/6/96.
;               2.0 01/02/99
;                           Modified to be compatible with non-CEPPAD data.
;               2.1 31/08/00
;                           Modified to allow lut=-1 in hist data for completeness,
;                           Excluded RAPID IES from sun sensor offset.
;               2.2 12/2/01
;                           Transferred endtime when present, else construct from next sample time.
;               2.3 26/10/01
;                           Fiddled with adding dummy MF, left as before.
;               2.4 21/02/02
;                           Added MVALS processing to MF.
;                           Corrected position of check for no data in time range.
;               2.5 20/06/02
;                           Found efficiency array wrong size for PEDESTAL IF DATA set
;                           -- issued warning
;               2.6 12/07/02
;                           Used ies_tag_exists.
;               2.7 31/01/03
;                           Used ies_sun_offsets.
;                           Allowed variation of offset within day.
;                           Included RAPID offset in calculation here.
;                           Removed dependence on ephem.r and mf.bt
;                           Added BT to output_data.mf
;               2.8 14/04/03
;                           Used ies_endtimes.
;               2.9 29/09/03
;                           Moved mvals calculation to ies_add_mf.
;                           Changed mvals to a flag rather than input_data[0].mvals
;               3.0 08/10/03
;                           Changed so that if MVALS present then sets up mf in ies restore file..
;               3.1 19/10/03
;                           Pass pitch angle tags .pa and pa_source through to header and data.
;                           Incorporated factor 2 increase in effective size used in ies pitch angle
;                           to eliminate sun and earth contamination of the pitch angle distributions.
;                           Passed offsets and midtimes though to ies add mf.
;                           Used routines ies add sun and ies add earth.
;               3.2 24/10/03
;                           Used input_data.mf tag for magnetic field if present.
;                           Changed argument list of ies div and used ies lut list.
;               3.3 31/10/03
;                           Changed name of ies magnetic field routine.
;                           Changed AT and EPHEM keywords in ies data structure to EARTH and SUN directly.
;                           Changed to use ies get sun, ies get earth and ies get mf.
;               3.4 20/11/03
;                           Fixed bug where ies div called with bin channel sizes rather than keV channel sizes.
;                           This is confusing rather than wrong since only relative channel sizes important in
;                           the instance in which it is used.
;
; Version     :	Version 3.4 20/11/03
;-
;**********************************************************

PRO ies_extdata, info_st, bands, input_header, input_data, full_detector, output_header, output_data, errmsg, $
                 A=a, B=b, C=c, D=d, E=e, FULL_DETECTOR_ONLY=full_detector_only, EPHEM=ephem, MF=mf, AT=at, $
                 MODES=modes

  ; initialize errmsg

  errmsg = ''

  ; get list of times within time interval

  time_list = WHERE ( input_data.time GE info_st.sttime AND input_data.time LE info_st.entime, npoints )

  ; check sufficient data points

  IF npoints LE 1 THEN BEGIN
    errmsg = 'Insufficient data for plotting, npoints = ' + STRTRIM(npoints,2)
    RETURN
  ENDIF

  ; check if data of only certain modes required (for HIST only)

  IF STRPOS(input_header.datatype,'HIST') GE 0 AND KEYWORD_SET(modes) THEN BEGIN

    ; set up decode of HIST LUTs

    hist_luts = (['ABC','BC','HBC'])([0, 1, 2, 0,  0, 1, 2, 0,  0, 1, 2, 2,  0, 1, 2, 0,  0, 1, 2, 0,  0, 2, 2, 0])

    IF N_ELEMENTS(modes) EQ 1 THEN $
      hist_list = WHERE ( hist_luts(input_data[time_list].lut>0) EQ modes[0], npoints) $
    ELSE IF N_ELEMENTS(modes) EQ 2 THEN $
      hist_list = WHERE ( hist_luts(input_data[time_list].lut>0) EQ modes[0] OR hist_luts(input_data[time_list].lut>0) EQ modes[1], npoints)

    IF npoints GT 0 THEN time_list = time_list[hist_list]

  ENDIF

  ; fix problem with IPS .detectors tag
  ; .detectors is primarliy used to give the ordering of items in .ptitle
  ; but is also used to locate the detectors in angle where it is assumed
  ; detectors go 1-10. IPS detectors go 0-9.

  IF STRPOS(input_header.datatype,'IPS') GE 0 THEN detectors = input_header.detectors + 1 $
    ELSE detectors = input_header.detectors

  ; get detectors to be used and detector for full display
  ; full_detector starts out as an index into sorted detector list
  ; then transformed into an index into detectors + 1
  ; full_detector = ndetectors implies detector sum
  ; checks that detector labelled as full detector has been selected

  detector_list = WHERE ( info_st.use_dets, ndetectors)

  IF full_detector LT input_header.ndetectors THEN BEGIN

    IF NOT KEYWORD_SET(full_detector_only) AND ndetectors GT 0 THEN BEGIN

      ; not full only and some detectors defined

      ; use_dets indexes into sorted detectors
      ; sort detectors

      detector_list = (SORT(detectors))(detector_list)

      ; check if full detector is used

      IF info_st.use_dets(full_detector) THEN $
        full_detector = LONG(TOTAL(info_st.use_dets(0:full_detector)))-1 $
      ELSE full_detector = 0

    ENDIF ELSE BEGIN

      ; full only or no detectors defined

      ; get full detector

      ndetectors = 1

      detector_list = [(SORT(detectors))(full_detector)]

      full_detector = 0

    ENDELSE

  ENDIF ELSE BEGIN

    ; detector sum

    IF ndetectors GT 0 THEN BEGIN

      ; some detectors defined, leave full_detector as detector sum

      ; use_dets indexes into sorted detectors
      ; sort detectors

      detector_list = (SORT(detectors))(detector_list)

      full_detector = ndetectors

    ENDIF ELSE BEGIN

      ; no detectors defined but require detector sum ?

      errmsg = 'No detectors selected'
      RETURN

    ENDELSE

  ENDELSE

  ; get no. of bands

  band_list = WHERE ( bands.use_bands, nbands )

  ; check whether to apply bottom and top channel selection
  ; only apply if bands.nbands=0

  IF bands.nbands EQ 0 THEN BEGIN

    start_band = bands.botchan
    end_band   = bands.topchan
    nbands     = end_band-start_band+1

  ENDIF ELSE BEGIN

    start_band = 0
    end_band   = nbands-1

  ENDELSE

  IF nbands LE 0 THEN BEGIN
    errmsg = 'No channel bands selected'
    RETURN
  ENDIF

  ; NB check whether new (vs 9) tag present or not

  IF ies_tag_exists(input_header,'ENERGY_CAL') THEN $
    energy_cal = input_header.energy_cal ELSE $
    energy_cal = 1

  IF ies_tag_exists(input_header,'COUNT_CAL') THEN $
    count_cal = input_header.count_cal ELSE $
    count_cal = 1

  IF ies_tag_exists(input_header,'ENERGY_EFFICIENCY') THEN BEGIN

    ; check for ccorrect no of energy bands

    IF (SIZE(input_header.energy_efficiency))[1] EQ input_header.nbands THEN BEGIN

      ; if summing over number of energy channels then easiest given program structure is to average over efficiencies

      efficiency = FLTARR(nbands,(SIZE(input_header.energy_efficiency))[2])

      FOR k = start_band, end_band DO BEGIN

        ; get band index

        b = band_list(k)

        efficiency(k-start_band,*) = TOTAL(input_header.energy_efficiency ( bands.lb(b):bands.ub(b), *), 1)/(bands.ub(b)-bands.lb(b)+1)

      ENDFOR

    ENDIF ELSE BEGIN

      MESSAGE, 'WARNING, IF file has invalid efficiency -- ignoring', /INFORMATIONAL

      efficiency=0

    ENDELSE

  ENDIF ELSE efficiency=0

  ; check if MVALS present
  ; set up mvals flag but not for EPAD2

  IF ies_tag_exists(input_data,'MVALS') AND STRPOS(input_header.datatype,'IES_EPAD2') LT 0 THEN mvals = 1

  ; NB check whether lut_map tag present or not
  ; set to 1 so that gets large pixel for IPS etc

  IF ies_tag_exists(input_header,'LUT_MAP') THEN lut_map = input_header.lut_map ELSE $
    lut_map = REPLICATE(1,input_header.nluts)

  ; set up pitch angle source

  IF ies_tag_exists(input_header, 'PA') THEN pa_source = input_header.pa

  ; set up output structure
  ; NB the brackets are necessary to preserve array character if ndetectors=1

  output_header = ies_header_structure ( input_header.datatype, $
                                         npoints, ndetectors, nbands, input_header.nsectors, input_header.nluts, $
                                         TIME_START = info_st.sttime, $
                                         TIME_STOP  = info_st.entime, $
                                         DETECTORS  = detectors(detector_list), $
                                         TITLE      = input_header.title,  $
                                         XTITLE     = input_header.xtitle, $
                                         YTITLE     = input_header.ytitle, $
                                         ZTITLE     = input_header.ztitle, $
                                         PTITLE     = (input_header.ptitle)(detector_list), $
                                         ENERGY_EFFICIENCY=efficiency, $
                                         ENERGY_CALIBRATION=energy_cal, $
                                         COUNT_CALIBRATION=count_cal, $
                                         LUT_MAP = lut_map, $
                                         MF=ies_magnetic_field_source(mf), PA=pa_source )

  ; add earth and sun tags to unnamed data structure if attitude and ephemeris available

  IF KEYWORD_SET(at) THEN sun = 1

  IF KEYWORD_SET(at) AND KEYWORD_SET(ephem) THEN earth = 1

  output_data   = ies_data_structure ( npoints, ndetectors, nbands, input_header.nsectors, $
    MF=KEYWORD_SET(mf), PA=KEYWORD_SET(pa_source), SUN=sun, EARTH=earth, MVALS=mvals)

  ; set up other header parameters

  ; NB check whether new (vs 6.2) tags present or not

  IF ies_tag_exists(input_header,'CAL') THEN output_header.cal = input_header.cal

  IF ies_tag_exists(input_header,'DIV') THEN output_header.div = input_header.div

  IF ies_tag_exists(input_header,'RAT') THEN output_header.rat = input_header.rat

  IF ies_tag_exists(input_header,'CLEAN') THEN output_header.clean = input_header.clean

  IF ies_tag_exists(input_header,'SUBTRACT') THEN output_header.subtract = input_header.subtract

  ; check if any summed channels

  dummy = WHERE ( bands.lb(band_list) NE bands.ub(band_list), count)

  IF count GT 0 THEN output_header.ztitle = 'Sum cnts'

  ; set up times

  output_data.time = input_data[time_list].time

  ; set up endtimes

  output_data.endtime = ies_endtimes(input_data,LIST=time_list)

  ; set up flags

  output_data.flag = input_data(time_list).flag

  ; set up luts

  output_data.lut = input_data(time_list).lut

  ; set up .mvals

  IF KEYWORD_SET(mvals) THEN output_data.mvals = input_data[time_list].mvals

  ; set up pa

  IF KEYWORD_SET(pa_source) THEN output_data.pa = input_data[time_list].pa

  ; extract time list and detector list from input data

  temp_data =  REFORM ( input_data(time_list).data ( *, detector_list, * ), input_header.nsectors, ndetectors, input_header.nbands, npoints)

  ; reverse channel size division if necessary

  IF output_header.div THEN BEGIN

    lut_list = ies_lut_list(output_data.lut, output_header.nluts)

    ies_div, lut_list, input_header.nbands, ndetectors, $
      ies_bins_to_kev(input_header.energy_cal,input_header.ch_positions[*,*,detector_list,*]), temp_data, /REVERSE

  ENDIF

  ; NB if bands overlap then normalization implied by channel positions will be wrong
  ;    i.e simply adding data in bands together but not adding no. of channels together

  FOR k = start_band, end_band DO BEGIN

    ; get band index

    b = band_list(k)

    ; NB IDL collapses the last dimension of array if one
    ; temp_data is sured up explicitly
    ; TOTAL(temp_data,3) is sured up because npoints > 1
    ; output_data.data(*,*,k) is sured up if ndetectors > 1 but not otherwise

    IF ndetectors GT 1 THEN output_data.data(*,*,k-start_band) = TOTAL(temp_data(*,*,bands.lb(b):bands.ub(b),*),3)  $
    ELSE output_data.data(*,*,k-start_band) = REFORM(TOTAL(temp_data(*,*,bands.lb(b):bands.ub(b),*),3))

    ; reset channel positions

    output_header.ch_positions(0,k-start_band,*,*) = input_header.ch_positions ( 0, bands.lb(b), detector_list,*)
    output_header.ch_positions(1,k-start_band,*,*) = input_header.ch_positions ( 1, bands.ub(b), detector_list,*)

  ENDFOR

  ; redo channel size division if necessary

  IF output_header.div THEN BEGIN

    ; temp_data is explicitly sured up

    temp_data = REFORM(output_data.data, output_header.nsectors, ndetectors, nbands, npoints)

    ies_div, lut_list, nbands, ndetectors, ies_bins_to_kev(output_header.energy_cal,output_header.ch_positions), temp_data

    IF nbands     GT 1 THEN output_data.data = temp_data ELSE $
    IF ndetectors GT 1 THEN output_data.data = REFORM(temp_data, output_header.nsectors, ndetectors, npoints) ELSE $
                            output_data.data = REFORM(temp_data)

  ENDIF

  ; get sample times and offsets

  midtimes = ies_midtimes(output_data)

  ; get sun offsets

  offsets = ies_sun_offsets(input_header.datatype, midtimes)

  ; add in sun position and size

  IF KEYWORD_SET(sun) THEN output_data.sun = ies_get_sun(at, midtimes, offsets)

  ; add in earth position and size

  IF KEYWORD_SET(earth) THEN output_data.earth = ies_get_earth(at, ephem, midtimes, offsets)

  ; add in magnetic field if present
  ; NB where mf contains magnetic field sampled at data times then need time_list

  IF KEYWORD_SET(mf) THEN output_data.mf = ies_get_mf(mf, midtimes, offsets, TIME_LIST=time_list)

  ; check if secondary arrays

  IF KEYWORD_SET(a) THEN a = a(*,time_list)
  IF KEYWORD_SET(b) THEN b = b(*,time_list)
  IF KEYWORD_SET(c) THEN c = c(*,time_list)
  IF KEYWORD_SET(d) THEN d = d(*,time_list)
  IF KEYWORD_SET(e) THEN e = e(*,time_list)

END