;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_FIT
;
; Purpose     :	CURVEFIT FCN FOR GAUSSIAN + EXPONENTIALS.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine interpolates the data to get samples
;               at specified bin positions. A pedestal shift for each column may optionally be applied.
;               A polynomial fit of degree 3 to log(y) against log(x) is used.
;
; Use         : < ies_fit,  x, a, f, fder >
;
; Inputs      : y            : FLTARR(nchannels,nsamples) counts per channel at each sample point for each channel
;               luts         : BYTARR(nsamples) LUT used for each sample point
;               ch_positions : FLTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               nchannels    : LONG no. of y channels
;               yp           : FLTARR channel or bin position for each new y sample
;
; Opt. Inputs : pedestals    : FLTARR(nsamples) bin positions of pedestals for each sample.
;
; Outputs     : y            : FLTARR(ysize,nsamples) interpolated image
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 10/10/96
;                 Adapted from ies rebin y.
;
; Modified    :	None.
;
; Version     :	Version 0.0, 10/10/96
;-
;**********************************************************
PRO ies_fit, x, a, f, fder

  ; Fitting function for Gaussian + 2 exponentials
  ; Used by CURVEFIT i.e
  ;  fit = CURVEFIT ( x, y, wgts, a, sigs, FUNCTION_NAME = ies_fit )

  ; x = list of npoint x values
  ; a = list of npar   parameter values
  ; fder = (npoints,npar) array of partial derivatives

  print, a

 ; 1 gauss curve + 2 exponentials

  IF a(2) NE 0.0 THEN gsz = (x-a(1))/a(2) ELSE gsz = 10

  gszz = ( gsz^2 ) < 100

  gsez = EXP(-gszz/2)

  gsy  = a(0)*gsez

  IF a(4) NE 0.0 THEN ex1z = x/a(4) ELSE ex1z = 10

  ex1ez = EXP ( -ex1z )

  ex1y  = a(3)*ex1ez

  IF a(6) NE 0.0 THEN ex2z = x/a(6) ELSE ex2z = 10

  ex2ez = EXP ( -ex2z )

  ex2y  = ABS(a(5))*ex2ez

  f = gsy + ex1y + ex2y

  IF ( N_PARAMS() GT 3 ) THEN BEGIN

    ; form partial derivatives

    fder = DBLARR(N_ELEMENTS(x),N_ELEMENTS(a))

    fder(*,0)  = gsez

    IF a(2) NE 0.0 THEN fder(*,1)  = gsy*gsz/a(2)

    fder(*,2)  = fder(*,1)*gsz

    fder(*,3) = ex1ez

    IF a(4) NE 0.0 THEN fder(*,4) = ex1y * ex1z / a(4)

    fder(*,5) = ex2ez

    IF a(6) NE 0.0 THEN fder(*,6) = ex2y * ex2z / a(6)

  ENDIF

END
