;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_GENERATE_MASKS
;
; Purpose     :	Generates SUN and EARTH masks for each valid sector/detector.
;
; Explanation :	Generates SUN and EARTH masks for each valid sector/detector.
;               Applies masks if keywords set : generates -ve pitch angles.
;
; Use         : < ies_generate_masks, header, data, array >
;
; Inputs      : header        : STRUCTURE info structure for data.
;               data          : STRARR for each sample point.
;
; Opt. Inputs : None.
;
; Outputs     : array         : FLTARR(nsectors, ndetectors, npoints)
;
; Opt. Outputs:	None.
;
; Keywords    : EARTH   : Apply earth mask.
;               SUN     : Apply sun mask.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	Separated from ies pitch angle.
;
; Written     :	Version 0.0, Martin Carter, RAL, 27/4/99
;
; Modified    :	Version 0.1, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.2, 20/07/01
;                 Used ies_runs.
;               Version 0.3 12/07/02
;                 Used ies_tag_exists.
;               Version 0.4 07/02/03
;                 Changed RAPID processing to include IIMS.
;                 Set invalid pitch angle to -1.
;                 Used duplicated pitch angle trick in calculation.
;               Version 0.5, 13/10/03
;                 Allow magnetic field direction for each sample to be passed by KEYWORD rather than through
;                 the data.mf tag.
;                 Fixed calculation for odd no of scetors (including one).
;               Version 0.6, 20/10/03
;                 Moved increase in effective size of sun and earth (factor 2) to ies extdata.
;                 Nb size also used directly in ies plot img.
;                 Separated masks from pitch angle calculation.
;               Version 0.7, 24/10/03
;                 Used ies lut list.
;
; Version     :	Version 0.7, 24/10/03
;-
;**********************************************************

PRO ies_generate_masks, header, data, array, EARTH=earth, SUN=sun

  ; set up masks array i.e contains 1 or -1 for each sector, detector, sample
  ; NB uses -1 for masked value and 0 for invalid detector
  ; (for POLAR IES small pixels only 6 detectors valid but .ndetectors=9
  ;  also IPS background detector set invalid)

  ; allow array to be passed through so can be masked off

  IF NOT KEYWORD_SET(array) THEN array = FLTARR(header.nsectors, header.ndetectors, header.npoints)

  ; check can apply masks

  IF KEYWORD_SET(earth) THEN earth = 1 ELSE earth = 0
  IF KEYWORD_SET(sun) THEN sun = 1 ELSE sun = 0

;  IF KEYWORD_SET(earth) THEN BEGIN
;    IF ies_tag_exists(data[0],'EARTH') THEN BEGIN
;      earth = 1
;    ENDIF ELSE BEGIN
;      earth = 0
;      PRINT, 'WARNING, NO EARTH MASK AVAILABLE'
;    ENDELSE
;  ENDIF

;  IF KEYWORD_SET(sun) THEN BEGIN
;    IF ies_tag_exists(data[0],'SUN') THEN BEGIN
;      sun = 1
;    ENDIF ELSE BEGIN
;      sun = 0
;      PRINT, 'WARNING, NO SUN MASK AVAILABLE'
;   ENDELSE
;  ENDIF

  ; loop through runs of fixed LUT

  ; get list of ends of runs of luts

  lut_list = ies_lut_list(data.lut,header.nluts)

  ; loop through runs of each lut

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = lut_list[l].lut

    ; dcount is no. of valid detectors
    ; dlist is index into data of valid detectors
    ; dpositions is angle where 0->180 is 0->dsize-1
    ; dpositions is also index into mask
    ; dposition 0 is antiparallel to spin

    ies_detector_polar_angle, header, lut, dlist, dcount, dsize, dpositions

    IF dcount GT 0 THEN BEGIN

      ; set valid sector/detectors
      ; except for IES small pixels and IPS background detector this wil be all detectors so is rather wasteful

      array[*,dlist,lut_list[l].st:lut_list[l].en] = 1

      ; loop through each sample point

      FOR p = lut_list[l].st, lut_list[l].en DO BEGIN

        ; generate IPS masks

        IF KEYWORD_SET(earth) OR KEYWORD_SET(sun) THEN BEGIN

          ; set up mask array
          ; NB mask[*,0] is anti-parallel to spin ie mask is in sector/detector order
          ; NB eliminate whole head if sun visible
          ; NB mask value is set to -1

          mask = REPLICATE(1,header.nsectors,3)

          ; do sun mask

          IF KEYWORD_SET(sun) THEN ies_mask, data[p].sun[0], data[p].sun[1], data[p].sun[2], mask, -1

          mask = REBIN(mask, header.nsectors, 9, /SAMPLE)

          ; do earth mask

          IF KEYWORD_SET(earth) THEN ies_mask, data[p].earth[0], data[p].earth[1], data[p].earth[2], mask, -1

          ; NB mask is indexed by dpositions  -- sector/detector order

          mask = mask[*,dpositions]

          ; apply mask

          array[*,dlist,p] = array[*,dlist,p]*mask

        ENDIF

      ENDFOR

    ENDIF

  ENDFOR

END
