;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	IES_GET_EARTH
;
; Purpose     :	Gets EARTH position and size in spacecraft coordinates for sample times
;
; Explanation : Gets EARTH position and size in spacecraft coordinates for sample times.
;
; Use         : < earth = ies_get_earth(at, ephem, times, offsets) >
;
; Inputs      : at       : STRUCTURE ARRAY giving S/C attitude
;               ephem    : STRUCTURE ARRAY giving S/C ephemeris
;               times    : DBLARR[nsamples] sample times
;               offsets  : FLTARR[nsamples] sun sensor offset for each sample
;
; Opt. Inputs : None.
;
; Outputs     : earth    : FLTARR[3,nsamples] earth azimuth, polar and size in S/C coords
;
; Opt. Outputs:
;
; Keywords    :
;
; Written     :	Version 0.0, Martin Carter, RAL, 20/10/03
;                 Derived from ies extdata
;
; Modified    : Version 0.1, 1/11/03
;                 Renamed from ies add earth.
;                 Changed to function and modified interface.
;                 Added check for AT time overlap.
;
; Version     :	Version 0.1, 1/11/03
;-
;**********************************************************

FUNCTION ies_get_earth, at, ephem, times, offsets

  nsamples = N_ELEMENTS(times)

  earth = FLTARR(3,nsamples)

  ; check that sample times overlap

  IF times[0] GT ephem[N_ELEMENTS(ephem)-1].time OR $
     times[nsamples-1] LT ephem[0].time OR $
     times[0] GT at[N_ELEMENTS(at)-1].time OR $
     times[nsamples-1] LT at[0].time THEN BEGIN

    MESSAGE,'No EPHEM/AT time overlap', /CONTINUE

  ENDIF ELSE BEGIN

    ; get earth vector
    ; NB previously interpolated ephem.r which is slightly different

    gse_earth = FLTARR(3,nsamples)
    gse_earth[0,*] = INTERPOL(ephem.gse_xyz[0,*],ephem.time, times)
    gse_earth[1,*] = INTERPOL(ephem.gse_xyz[1,*],ephem.time, times)
    gse_earth[2,*] = INTERPOL(ephem.gse_xyz[2,*],ephem.time, times)

    r = SQRT(gse_earth[0,*]^2+gse_earth[1,*]^2+gse_earth[2,*]^2)

    earth[0:1,*] = ies_gse_to_sc ( at.gse_ra, at.gse_dec, at.time, -gse_earth, times, offsets)

    ; calculate half cone angle subtended by earth

    earth[2,*] = ASIN(6378./ REFORM(r))

    ; increase size by factor 2 to avoid contamination in pitch angle plots

    earth[2,*] = 2*earth[2,*]

  ENDELSE

  RETURN, earth

END