;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_GET_FILE_NAME
;
; Purpose     :	Gets name of plot file.
;
; Explanation : Gets name of plot file.
;               Assumes string for ps, bwps, png  or gif is file name.
;               If file name defined by ps, bwps, png or gif, then if directory not present adds directory
;               defined by appropriate environment variable.
;               If file name is defined then leaves alone, regardless of whether directory is present.
;
; Use         : < result = ies_get_file_name(pageno, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, BWPS=bwps) >
;
; Inputs      : pageno   : INT page number of plot.
;
; Opt. Inputs : None.
;
; Outputs     : Returns 0 if successful, 1 if unsuccessful.
;
; Opt. Outputs:	unit   : INT logical unit number for file.
;               file   : STRING file name for output.
;
; Keywords    : /WINDOWS    = plots output in x window.
;               /POSTSCRIPT = plots output as postscript file
;                             Assumes string is file name.
;               /PNG        = plots output as PNG file
;                             Assumes string is file name.
;               /GIF        = plots output as GIF file
;                             Assumes string is file name.
;               /BWPS       = plots output as black and white postscript file
;                             Assumes string is file name.
;               /POLAR      = FLAG use POLAR output area
;               /RAPID      = FLAG use RAPID output area
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/03/03
;
; Modified    : Version 0.1, 30/09/03
;                 Added RAPID/POLAR keyword switch.
;                 Corrected problem if path is changed.
;                 Renamed from ies get file.
;                 Changed so that only adds directory if no filename defined.
;                 Used ies get dir.
;
; Version     :	Version 0.1, 30/09/03
;-
;**********************************************************

FUNCTION ies_get_file_name, pageno, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps, $
                               POLAR=polar, RAPID=rapid

  ; test output device

  IF KEYWORD_SET(windows) THEN BEGIN

    RETURN, 'WINDOW'

  ENDIF ELSE IF datatype(postscript) EQ 'STR' THEN BEGIN

    RETURN, postscript

  ENDIF ELSE IF datatype(png) EQ 'STR' THEN BEGIN

    RETURN, png

  ENDIF ELSE IF datatype(gif) EQ 'STR' THEN BEGIN

    RETURN, gif

  ENDIF ELSE IF datatype(bwps) EQ 'STR' THEN BEGIN

    RETURN, bwps

  ENDIF ELSE BEGIN

    ; file name undefined, need to set up

    ; determine type of output file

    IF KEYWORD_SET(png) THEN BEGIN

      ext = 'png'
      name = png

    ENDIF ELSE IF KEYWORD_SET(gif) THEN BEGIN

      ext = 'gif'
      name = gif

    ENDIF ELSE IF KEYWORD_SET(postscript) THEN BEGIN

      ext = 'ps'
      name = postscript

    ENDIF ELSE IF KEYWORD_SET(bwps) THEN BEGIN

      ext = 'ps'
      name = bwps

    ENDIF ELSE BEGIN

      RETURN, 'WINDOW'

    ENDELSE

    ; get appropriate environment variable

    path = ies_get_dir(STRUPCASE(ext),POLAR=polar, RAPID=rapid)

    ; ask for file name

    CASE pageno OF

       0 : IF datatype(name) NE 'STR' THEN BEGIN

             file=ies_pickfile(TITLE='Select file for '+STRUPCASE(ext)+' output', FILTER='*.'+ext, PATH=path, GET_PATH=new_path )

             path = new_path

           ENDIF ELSE file = name

       1 : IF NOT (KEYWORD_SET(postscript) OR KEYWORD_SET(bwps)) THEN BEGIN

             file=ies_pickfile(TITLE='Select second file for '+STRUPCASE(ext)+' output',FILTER='*.'+ext, PATH=path, GET_PATH=new_path )

             path = new_path

           ENDIF

       2 : IF NOT (KEYWORD_SET(postscript) OR KEYWORD_SET(bwps)) THEN BEGIN

             file=ies_pickfile(TITLE='Select third file for '+STRUPCASE(ext)+' output',FILTER='*.'+ext, PATH=path, GET_PATH=new_path )

             path = new_path

           ENDIF

    ENDCASE

    IF file EQ '' THEN MESSAGE, 'No data file given', /CONT, /TRACEBACK

    ; check if has directory defined

    IF STRLEN(strip_dirname(file)) EQ STRLEN(file) THEN file = concat_dir(path,file)

   ; test file ok

    ;OPENW, unit, file, /GET_LUN

    ;FREE_LUN, unit

    RETURN, file

  ENDELSE

END
