;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	IES_GET_MF
;
; Purpose     :	Gets MF in spacecraft coordinates for sample times
;
; Explanation : Gets MF in spacecraft coordinates for sample times.
;
; Use         : < mfield = ies_get_mf (mf, times, offsets) >
;
; Inputs      : mf       : STRUCTURE ARRAY giving magnetic field
;               times    : DBLARR[nsamples] sample times
;               offsets  : FLTARR[nsamples] sun sensor offset for each sample
;
; Opt. Inputs : None.
;
; Outputs     : mfield   : FLTARR[3,nsamples] magnetic field azimuth, polar and strength in S/C coords
;
; Opt. Outputs:
;
; Keywords    : TIME_LIST : LONARR[nsamples] indices into original IF array of selected samples.

;
; Written     :	Version 0.0, Martin Carter, RAL, 14/04/03
;                 Derived from ies extdata
;
; Modified    : Version 0.1, 29/09/03
;                 Added mval magnetic field calculation.
;                 Inverted magnetic field produced by mval calculation.
;               Version 0.2, 18/10/03
;                 Allowed magnetic field for each sample to be returned via a keyword argument.
;                 Only set up magnetic field in data structure if .mf tag present.
;                 Added MIDTIMES and OFFSETS keywords.
;                 Removed warning if mf not set.
;               Version 0.3, 31/10/03
;                 Added TIME_LIST keyword.
;                 Renamed from ies add mf.
;                 Changed to function and modified interface.
;
; Version     :	Version 0.3, 31/10/03
;-
;**********************************************************

FUNCTION ies_get_mf, mf, times, offsets, TIME_LIST=time_list

  nsamples = N_ELEMENTS(times)

  mfield = FLTARR(3,nsamples)

  ; check that sample times overlap

  IF times[0] GT mf[N_ELEMENTS(mf)-1].time OR $
     times[nsamples-1] LT mf[0].time THEN BEGIN

    MESSAGE,'No MF time overlap', /CONTINUE

  ENDIF ELSE BEGIN

    ; check if magnetic field already on time samples

    IF STRPOS(TAG_NAMES(mf,/STRUCTURE_NAME),'XYZ') GE 0 THEN BEGIN

      ; mf is in SC xyz coords

      ; get interpolated B field
      ; NB INTERPOL will extrapolate
      ; This could give really silly results if mf only defined for some parts of the time period.
      ; NB previously interpolated mf.bt which is slightly different

      bx = INTERPOL ( mf.mf[0], mf.time, times)
      by = INTERPOL ( mf.mf[1], mf.time, times)
      bz = INTERPOL ( mf.mf[2], mf.time, times)

      bt = SQRT(bx^2+by^2+bz^2)

      ; put B field azimuth and polar in data structure

      mfield[0,*] = (ATAN(by,bx) + 2*!PI + offsets) MOD (2*!PI) ; 0->2*pi
      mfield[1,*] = !PI/2 - ASIN(bz/bt) ; 0->pi
      mfield[2,*] = bt

    ENDIF ELSE BEGIN

      ; assume if mf structure name does not contain label XYZ then is already in correct reference frame in
      ; {theta,phi,strength} format

      ; these mf samples are taken from the original IF data set
      ; the data may be selected from within these by time range

      IF KEYWORD_SET(time_list) THEN mfield = mf[time_list].mf ELSE mfield = mf.mf

    ENDELSE

  ENDELSE

  RETURN, mfield

END