;+
; Project     :	CLUSTER - RAPID, POLAR _ CEPPAD
;
; Name        :	IES_GET_MINMAX
;
; Purpose     :	Finds minimum and maximum of data for all sectors/detectors used in display.
;
; Explanation :	Finds minimum and maximum values for data display.
;               Ignores bad quality data.
;               If display value is editable then uses display value else uses data values.
;               Avoids the zero problem for log plots by setting zeroes to minimum/10.
;               Enforces maximum greater than minimum.
;               Allows -ve values but not in log plots.
;
; Use         : < ies_get_minmax, y, flags, minid, maxid, output_info, LOG=log >
;
; Inputs      : y       : FLTARR(nsectors,ndetectors,nenergies,npoints) data values
;               flags   : FLTARR(npoints) data quality flag for each point.
;               minid   : LONG widget ID for minimum display.
;               maxid   : LONG widget ID for maximum display.
;               output_info : Output structure
;                                .min : flag.
;                                .max : flag.
;                                .minimum : FLOAT minimum to be used.
;                                .maximum : FLOAT maximum to be used.
;
; Opt. Inputs : None.
;
; Outputs     : minimum : FLOAT minimum data value.
;               maximum : FLOAT maximum data value.
;
; Opt. Outputs:	None.
;
; Keywords    : /LOG    : Log. plot for data.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, MKC, 19/9/96.
;                 Treat flag=1 as a signal rather than bad data indicator.
;                 Changed treatment of linear plots.
;               Version 0.2, MKC, 7/3/97
;                 Changed treatment if no good data.
;               Version 0.3, MKC, 27/10/98
;                 Fixed problem if data all zeroes and log plot.
;               Version 0.4, MKC, 22/2/99
;                 Changed use of -ve values.
;               Version 0.5, MKC, 17/3/99
;                 Used output_info.
;               Version 0.6, MKC, 10/5/00
;                 Extended good data range in flag from <=1 to AND 'F0'X
;               Version 0.7, MKC, 1/9/00
;                 Allowed use of -ve values for minimum and maximum.
;               Version 0.8, 07/02/02
;                 Used separate function for checking good data.
;
; Version     :	Version 0.8, 07/02/02
;-
;**********************************************************


PRO ies_get_minmax, y, flags, minid, maxid, output_info, LOG=log

  ; use whether text widget is editable to determine whether to use user supplied value or
  ; to calculate an automatic value -- signal whether automatic or not with min and max
  ; test maximum and minimum

  IF WIDGET_INFO(minid, /VALID_ID) THEN BEGIN
    IF WIDGET_INFO(minid, /TEXT_EDITABLE) THEN BEGIN
      WIDGET_CONTROL, minid, GET_VALUE=value
      output_info.minimum = FLOAT(value(0))
      output_info.min = 1
    ENDIF ELSE output_info.min = 0
  ENDIF

  IF WIDGET_INFO(maxid, /VALID_ID) THEN BEGIN
    IF WIDGET_INFO(maxid, /TEXT_EDITABLE) THEN BEGIN
      WIDGET_CONTROL, maxid, GET_VALUE=value
      output_info.maximum = FLOAT(value(0))
      output_info.max = 1
    ENDIF ELSE output_info.max = 0
  ENDIF

 ; check whether to use supplied minimum and maximum values

 IF output_info.min EQ 0 OR output_info.max EQ 0 THEN BEGIN

   ; get list of good data

   list = ies_good_data ( flags, COUNT=count )

   IF count EQ 0 THEN BEGIN

      ; print warning

      MESSAGE, /CONT, 'WARNING, NO GOOD DATA'

      ; use all data even if bad

      list = LINDGEN(N_ELEMENTS(flags))

   ENDIF

   dims = SIZE(y)

   IF dims(0) EQ 1 THEN temp_y = y(list) ELSE $
   IF dims(0) EQ 2 THEN temp_y = y(*,list) ELSE $
   IF dims(0) EQ 3 THEN temp_y = y(*,*,list) ELSE $
   IF dims(0) EQ 4 THEN temp_y = y(*,*,*,list)

   ; check if log

   IF KEYWORD_SET(log) THEN BEGIN

     non_zero_list = WHERE ( temp_y GT 0, count)

     ; do minimum

     IF output_info.min EQ 0 THEN BEGIN

       IF count EQ 0 THEN output_info.minimum = 1 ELSE output_info.minimum = MIN(temp_y(non_zero_list))

       ; check if any zeroes

       IF count NE N_ELEMENTS(temp_y) THEN output_info.minimum = output_info.minimum/10.0

     ENDIF

     ; do maximum

     IF output_info.max EQ 0 THEN BEGIN

       IF count EQ 0 THEN output_info.maximum = 1 ELSE output_info.maximum = MAX(temp_y)

     ENDIF

   ENDIF ELSE BEGIN

     ; do minimum

     IF output_info.min EQ 0 THEN output_info.minimum = MIN(temp_y)

     ; do maximum

     IF output_info.max EQ 0 THEN output_info.maximum = MAX(temp_y)

   ENDELSE

 ENDIF

 ; fix maximum greater than minimum

 IF output_info.maximum LT output_info.minimum THEN BEGIN
   temp = output_info.minimum
   output_info.minimum = output_info.maximum
   output_info.maximum = temp
 ENDIF

 ; if log then force positive

 IF KEYWORD_SET(log) THEN BEGIN

   IF output_info.minimum LE 0.0 AND output_info.maximum LE 0.0 THEN BEGIN
     output_info.minimum = 1
     output_info.maximum = 10
   ENDIF ELSE IF output_info.minimum LE 0.0 THEN BEGIN
     output_info.minimum = output_info.maximum/10.0
   ENDIF ELSE IF output_info.maximum LE 0.0 THEN BEGIN
     output_info.maximum = output_info.minimum*10.0
   ENDIF

 ENDIF

 ; IF min and max equal then alter

 IF output_info.minimum EQ output_info.maximum THEN BEGIN

   IF output_info.minimum EQ 0 THEN BEGIN
     output_info.minimum = 0
     output_info.maximum = 1
   ENDIF ELSE BEGIN
     output_info.minimum = output_info.minimum/2
     output_info.maximum = output_info.maximum*2
   ENDELSE

 ENDIF


 IF WIDGET_INFO(minid, /VALID_ID) THEN WIDGET_CONTROL, minid, SET_VALUE=STRTRIM(STRING(FORMAT='(G9.3)',output_info.minimum),2)
 IF WIDGET_INFO(maxid, /VALID_ID) THEN WIDGET_CONTROL, maxid, SET_VALUE=STRTRIM(STRING(FORMAT='(G9.3)',output_info.maximum),2)

END

