;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_GET_NORMALIZATION
;
; Purpose     : Normalizes pedestal shape to measured pedestal data.
;
; Explanation :	Normalizes pedestal shape to measured pedestal data.
;
; Use         : < normalization = ies_get_normalization(ped_array, ssamples) >
;
; Inputs      : ped_array        : FLTARR[npeds,nshifts] pedestal data repeated nshifts.
;               ssamples         : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data.
;                                  Whole spectrum is normalized to one.
;
; Opt. Inputs : None.
;
; Outputs     : normalization    : FLTARR[nshifts] normalization factor for ssamples for each shift.
;
; Opt. Outputs:	None.
;
; Keywords    : CHISQ  : FLAG indicating should use chisq for calculating normalization factor.
;
; Written     :	Version 0.0, 25/11/03
;                 Separated from cep pedestals.
;
; Modified    :
;
; Version     :	Version 0.0, 25/11/03
;-
;**********************************************************

FUNCTION ies_get_normalization, ped_array, ssamples, CHISQ=chisq

  ; use peak

  ;norm_ped_array = max_ped_array                                  ; FLOAT
  ;norm_ssamples = REFORM(ssamples[max_pos-istart,*])              ; FLTARR[max_count]

  ; use total

  ;norm_ped_array = tot_ped_array                                  ; FLOAT
  ;norm_ssamples = TOTAL(ssamples,1)                               ; FLTARR[max_count]

  IF KEYWORD_SET(chisq) THEN BEGIN

    ; minimize chisq for normalization

    norm_ped_array = TOTAL(ssamples,1)                            ; FLTARR[max_count]

    norm_ssamples  = TOTAL(ssamples*ssamples/(ped_array>1),1)   ; FLTARR[max_count]

  ENDIF ELSE BEGIN

    ; minimize mean squared error for normalization
    ; NB previously had ped2d_array*ped2d_array ??

    norm_ped_array = TOTAL(ssamples*ped_array,1)                ; FLTARR[max_count]

    norm_ssamples  = TOTAL(ssamples*ssamples,1)                   ; FLTARR[max_count]

  ENDELSE

  ; check if no data in pedestal channels for shifted histogram

  list = WHERE(norm_ssamples LE 0, count)

  ; avoid maths errors

  IF count GT 0 THEN norm_ssamples[list] = -1

  ; get normalization

  normalization = REFORM([norm_ped_array/norm_ssamples],N_ELEMENTS(norm_ssamples))

  RETURN, normalization

END