;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_GET_PITCH_ANGLE
;
; Purpose     :	Calculates pitch angle projection.
;
; Explanation :	Calculates pitch angle projection with sector/detector mask applied.
;               Have flux at a given pitch angle for a number of pitch angles.
;               These simply get binned into pitch angle bins,
;               Previously interpolated these to give flux at each pitch angle.
;
; Use         : < array = ies_get_pitch_angle (output_array, output_info, header, data) >
;
; Inputs      : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;               output_info   : STRUCTURE containing plotting and processing info
;               header        : STRUCTURE info structure for data.
;               data          : STRARR for each sample point.
;
; Opt. Inputs : None.
;
; Outputs     : array         : FLTARR(npitchs, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 28/10/98
;                 Adapted from ies_calibrate.
;
; Modified    :	Version 0.1, MKC, 11/11/98
;                 Dealt properly with HIST.
;                 Corrected IPS mask i.e masked out whole head rather than detector.
;               Version 0.2, MKC, 2/2/99
;                 Modified old pitch angle calculation which was rather obscure.
;                 Previously, divided each detector/sector bin into smaller bins
;                 to artificially increase the no. of samples. Divided pitch angle
;                 into a number of bins and accumulated histogram. If no pitch angle
;                 for a given bin then gave a zero which could give aliasing problems
;                 in ies_mk_image.
;                 Now, interpolate pitch angle samples to give distribution.
;               Version 0.3, MKC, 22/2/99
;                 Fixed bug if summing sectors over IPS background detector.
;                 Fixed bug averaging over small pixels for IES.
;               Version 0.4, MKC, 5/5/99
;                 Added routine ies pitch angle.
;               Version 0.5, 11/5/99
;                 Removed EARTH and SUN keywords.
;               Version 0.6, 20/8/00
;                 Dealt with RAPID data
;               Version 0.7, 12/07/02
;                 Corrected bug if no magnetic field data.
;               Version 0.8, 07/02/03
;                 Changed to 0-180 degrees pitch angle output.
;                 Set values for which no pitch angle measured equal to -1.
;               Version 0.9, 29/09/03
;                 Changed to selectable no. of pitch angle bins.
;                 Replaced NPITCHS keyword with PITCH_ANGLE keyword.
;                 Used new routine for instrument test.
;                 Fixed odd bug where pitch angle = 180 caused SW to crash.
;                 Can't believe that never encountered this before now ?
;               Version 1.0, 20/10/03
;                 Use routine ies generate masks.
;                 Updated pitch angle histogram calculation and prevented
;                 invalid/masked pitch angles appearing in bin zero.
;                 Checked for pitch angle tag in input.
;                 Created new routine ies get pitch angle.
;                 Moved pitch angle part of calculation from ies average.
;
; Version     :	Version 1.0, 20/10/03
;-
;**********************************************************

FUNCTION ies_get_pitch_angle, output_array, output_info, header, data

  ; get pitch angle histogram
  ; set value to -1 so that can tell if no pitch angle coverage
  ; used in ies_plot_profiles to tell if no coverage

  ; set up output array

  array = REPLICATE(-1.0,header.nbands, output_info.npitchs, header.npoints)

  ; get pitch angles

  IF ies_tag_exists(data, 'PA') THEN BEGIN

    ; use pitch angle tag

    pas = data.pa

  ENDIF ELSE BEGIN

    ; get pitch angles

    ies_pitch_angle, header, data, pas

  ENDELSE

  ; check magnetic field available

  IF N_ELEMENTS(pas) EQ 0 THEN RETURN, array

  ; generate index into pitch angle histogram

  angs = 0 > FIX(TEMPORARY(pas)/!PI*output_info.npitchs) < (output_info.npitchs-1)

  ; generate masks

  ies_generate_masks, header, data, masks, EARTH=ies_instrument(header.datatype,/IPS), SUN=ies_instrument(header.datatype,/IPS)

  ; loop through each sample point
  ; this could be speeded up

  FOR p = 0, header.npoints-1 DO BEGIN

    ; accumulate frequency and total in each pitch angle bin

    freq = LONARR(output_info.npitchs)

    FOR s = 0, header.nsectors-1 DO FOR d = 0, header.ndetectors-1 DO BEGIN

      ; check if masked off

      IF masks[s,d,p] EQ 1 THEN BEGIN

        ang = angs[s,d,p]

        IF freq[ang] EQ 0 THEN BEGIN

          array[*,ang,p] = output_array[s,d,*,p]

          freq[ang] = 1

        ENDIF ELSE BEGIN

          array[*,ang,p] = array[*,ang,p]+output_array[s,d,*,p]

          freq[ang] = freq[ang]+1

        ENDELSE

      ENDIF

    ENDFOR

    ; get average flux in each pitch angle bin if flux

    IF output_info.cal THEN array[*,*,p] = array[*,*,p] / (REPLICATE(1,header.nbands)#freq > 1)

if 0 then begin

      ; get pitch angles for each sector/detector

      pa = pas[*,*,p] ; 0-pi

      ; set up list of each sector/detector
      ; NB lis is set up so that lis = 0 is parallel to spin but elements are in sector/detector order
      ; NB pa < 0 copes with switch between small and large IES pixels else all entries set.

      lis = WHERE(pa GE 0, count)  ; each valid sector/detector

      IF count GT 0 THEN BEGIN

        map = (pa[lis]/!PI*output_info.npitchs) MOD output_info.npitchs ; pa for each element in lis

        ; order pitch angle bins

        l0 = SORT(map)

        lis0 = lis[l0]
        map0 = map[l0]

        ; do not extrapolate data

        pitch_min = FIX(MIN(map0)+0.999)
        pitch_max = FIX(MAX(map0))

        ; get output array for this sample point

        temp_array = REFORM(output_array[*,*,*,p])

        ; reform array

        temp_array = REFORM ( temp_array, header.nsectors*header.ndetectors, header.nbands)
        temp_array = TRANSPOSE(temp_array)

        ; loop through energy bands
        ; NB does not average if same pitch angle
        ; really needs some smoothing interpolation rather than linear interpolation

        FOR e = 0, header.nbands-1 DO BEGIN

          array[e,pitch_min:pitch_max,p] = $
           INTERPOL(REFORM(temp_array[e,lis0]),map0,pitch_min+FINDGEN(pitch_max-pitch_min+1))

        ENDFOR

        ;array[*,map0,p] = 10*temp_array[*,lis0] ; test of interpolation

      ENDIF
endif

  ENDFOR

  RETURN, array

END
