;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_GET_SAMPLES
;
; Purpose     : Returns pedestal calibration data for a given LUT sampling and time range sampled according to LUT.
;
; Explanation :	Associated with each time sample is an LUT number which defines the
;               bin positions of the energy samples. The pedestal shift is calculated to a
;               fraction of a channel by cross-correlating the known pedestal profile with the sampled data.
;               This routine calculates the pedestal samples corresponding to different shifts from the pedestal
;               histogram data. The result is normalized to give total counts in the whole shifted spectrum of unity.
;
; Use         : < samples = ies_get_samples ( pdata, npeds, ch_positions, nshifts, offset) >
;
; Inputs      : pdata         : FLTARR(nbins) pedestal calibration data
;               npeds         : INT no. of pedestal channels.
;               ch_positions  : INTARR(2,nbands) containing energy channel bin positions for given detector and LUT
;               nshifts       : INT no. of pedestal shifts (each shift is 0.1  bins)
;               offset        : INT pedestal shift offset (start shift is +offset/10 bins)
;
; Opt. Inputs : None.
;
; Outputs     : samples       : FLTARR(npeds,251) where 2nd dimension corresponds to different shifts of
;                               multiples of 0.1 bins ie shifts of -5 to 20 bins downwards.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Calls       :	ies_fit_pedestal.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 22/8/97
;                 Adapted from ies pedestals.
;
; Modified    :	Version 0.1, MKC, 17/06/98
;                 Added keyword to force selection of a particular pedestal data set.
;               Version 0.2, 20/08/00
;                 Dealt with lut=-1
;               Version 0.3, 20/09/00
;                 Moved set up of pedestal calibration data.
;                 Added calibration file to COMMON block.
;                 Changed npeds in argument list to ped_chans.
;                 Changed so that normalized to no of counts in histogram.
;                 Also added KEYWORD NCOUNTS
;               Version 0.4, 11/10/00
;                 Changed so that now puts the pedestal counts on bin centres rather than on bin start.
;                 This shifts results by half a bin compared to previously.
;                 Moved code into ies fit pedestal, ies get pedestal.
;                 Removed CENTROID keyword.
;                 Changed argument list.
;                 Remove KEYWORD NCOUNTS. Normalize sampled arrays to 1.
;               Version 0.5, 21/11/03
;                 Changed ped_chans argument to npeds.
;               Version 0.6, 11/12/03
;                 Added nshifts argument.
;                 Removed fit to pedestal and pedestal_posn and pedestal_width arguments.
;
; Version     :	Version 0.6, 11/12/03
;-
;**********************************************************

FUNCTION ies_get_samples, pdata, npeds, ch_positions, nshifts, offset

  ; get no. of pedestal samples

  nbins = N_ELEMENTS(pdata)

  ; form sampled pedestal arrays
  ; interpolate the data samples at one tenth bin intervals
  ; shift pedestals moving start position in interpolated array
  ; use every tenth sample to give bin samples
  ; this gives bin samples of shifted pedestals
  ; sum using luts to give sampled shifted pedestals
  ; only keep bottom npeds energy channels

  ; get temporary samples array

  samples = FLTARR(npeds,nshifts)

  ; interpolate pedestal histogram data

;          ; use linear or cubic fit to interpolate data

;          idata = INTERPOLATE ( pdata, FINDGEN((10*(nbins-1)+1))/10.0)
;          idata = INTERPOLATE ( pdata, FINDGEN((10*(nbins-1)+1))/10.0, /CUBIC)

  ; substitute Guassian fit to pedestal data

;          pdata(*)=0
;          pdata((maxx-30)>0:(maxx+30)<(nbins-1)) = f

  ; use linear or cubic fit to interpolate LOG(data)

  zero_list = WHERE (pdata EQ 0, zero_count)

  IF zero_count GT 0 THEN pdata(zero_list) = 1.e-5

  ; idata is interpolated by 10 pedestal histogram
  ; it is decimated by 10 to give pedestal histogram corresponding to different pedestal shifts

  idata = INTERPOLATE ( ALOG(pdata), FINDGEN((10*(nbins-1)+1))/10.0)
;          idata = INTERPOLATE ( ALOG(pdata), FINDGEN((10*(nbins-1)+1))/10.0, /CUBIC)

  idata = EXP(idata)

  zero_list = WHERE(idata LT 0.1, zero_count)

  IF zero_count GT 0 THEN idata[zero_list] = 0

  ; get normalization factors
  ; ignore last point

  norm_idata = FLTARR(10)

  FOR k = 0, 9 DO norm_idata[k] = TOTAL(idata[10*INDGEN(nbins)+k])

  norm_factor = 1.0/norm_idata

  ; loop over low energy channels

  FOR c = 0, npeds-1 DO BEGIN

    ; get original sample points in idata for channel
    ; physically, on right shift all data off bottom will appear in bin 0.

    IF ch_positions(0,c) NE 0 THEN $
      inds = 10*(ch_positions(0,c)+INDGEN(ch_positions(1,c)-ch_positions(0,c))) $
;            ; make pedestal bins inclusive of end points -- for no better reason than fits the data better
;              inds = 10*(ch_positions(0,c,d,m)+INDGEN(1+ch_positions(1,c,d,m)-ch_positions(0,c,d,m))) $
    ELSE $
      inds = 10*INDGEN(ch_positions[1,c]+20)-200
;            ; make pedestal bins inclusive of end points -- for no better reason than fits the data better
;              inds = 10*INDGEN(1+ch_positions(1,c,d,m)+20)-200

    ; shift pedestal by b bins

    FOR b = 0, nshifts-1 DO BEGIN

      ; NB does (N_ELEMENTS(idata)-1) < (inds+b-offset) > 0
      ; this is OK for -ve values assumes that when shift pedestal to the right
      ; the value at bin 0 is extended to the left.
      ; dont need to worry about -ve values because size of idata is larger than any shift
      ; encountered but the same holds true i.e last value is extended to the right.

      samples[c,b] = TOTAL(idata[inds+b-offset])*norm_factor[b MOD 10]

    ENDFOR ; loop over b

  ENDFOR ; loop over c

  RETURN, samples

END