;+
; Project     :	CLUSTER - POLAR - PULSAUR
;
; Name        :	IES_GET_SUN
;
; Purpose     :	Gets SUN position and size in spacecraft coordinates for sample times
;
; Explanation : Gets SUN position and size in spacecraft coordinates for sample times.
;
; Use         : < sun = ies_get_sun(at, times, offsets) >
;
; Inputs      : at       : STRUCTURE ARRAY[n] giving S/C attitude
;               times    : DBLARR[nsamples] sample times
;               offsets  : FLTARR[nsamples] sun sensor offset for each sample
;
; Opt. Inputs : None.
;
; Outputs     : sun      : FLTARR[3,nsamples] sun azimuth, polar and size in S/C coords
;
; Opt. Outputs:
;
; Keywords    :
;
; Written     :	Version 0.0, Martin Carter, RAL, 20/10/03
;                 Derived from ies extdata
;
; Modified    : Version 0.1, 1/11/03
;                 Renamed from ies add sun.
;                 Changed to function and modified interface.
;
; Version     :	Version 0.1, 1/11/03
;-
;**********************************************************

FUNCTION ies_get_sun, at, times, offsets

  nsamples = N_ELEMENTS(times)

  sun = FLTARR(3,nsamples)

  ; check that sample times overlap

  IF times[0] GT at[N_ELEMENTS(at)-1].time OR $
     times[nsamples-1] LT at[0].time THEN BEGIN

    MESSAGE,'No AT time overlap', /CONTINUE

  ENDIF ELSE BEGIN

    ; get sun vector

    gse_sun = [1.,0,0]#REPLICATE(1,nsamples)

    sun[0:1,*] = ies_gse_to_sc ( at.gse_ra, at.gse_dec, at.time, gse_sun, times, offsets)

    ; calculate half cone angle subtended by sun
    ; true size of sun is 16 arcmin but effective size is much bigger because of scattering from instrument
    ; sun contamination effects pitch angle plots most

    sun[2,*] = 20./180*!PI

  ENDELSE

  RETURN, sun

END