;+
; Project     :	CLUSTER - RAPID - PULSAUR
;
; Name        :	IES_GETSET
;
; Purpose     :	Widget tool for specifying data selection.
;
; Explanation : Allows the selection of energy bands, time interval and detectors.
;               At present uses modal KEYWORD to xmanager which is obsolete.
;               This also seems to give a floating point error for some reason.
;               ies_do_prelims calls ies_getset and continues when it finishes.
;               ies_plot calls ies_do_prelims during setup and then continues.
;               On return it kills itself then calls itself again.
;               NB ies_do_prelims not relevant to above structure. ie effectively ies_getset.
;
; Use         : <st = ies_getset ( nbands, input_files, GROUP=group) >
;
; Inputs      : nbands : INT no. of bands to be selected from input data
;                        If nbands equals nchannels then bands are set to channels
;                        i.e as for spectral plot.
;
; Opt. Inputs : None.
;
; Outputs     : st : structure containing data selection information.
;                     lb        : INTARR(nbands) : Lower channel bound on each band.
;                     ub        : INTARR(nbands) : Upper channel bound on each band.
;                     use_bands : INTARR(nbands) : Indicates whether each band to be used.
;                     sttime    : start time of period of interest in internal format
;                     entime    : end time of period of interest in internal format
;                     use_dets  : INTARR(ndetectors) indicating whether each detector to be used
;
; Opt. Outputs:	None.
;
; Keywords    : GROUP : Widget ID of parent.
;
; Common      :	ies_getset_block, ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Command preparation.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 5/6/95
;
; Modified    :	Version 0.1, 16/1/96
;                 Updated for use by CEPPAD.
;               Version 0.2, 29/2/96
;                 Modifed to use internal time.
;               Version 0.3, 19/9/96
;                 Added SECTOR vs DETECTOR plot option.
;                 Updated tool version number.
;                 Changed initial size of some labels to adapt to new font.
;                 Extended ies_draw_widget_block.
;                 Used DYNAMIC_RESIZE keyword for label widget.
;                 Added TIMELINE plot.
;                 Added clean tag.
;               Version 0.4, 7/11/96
;                 Checked whether .clean tag present for compatability with old data structure.
;                 Removed SW version.
;               Version 0.5, 11/11/96
;                 Add full sector selection.
;                 Added droplists for full detector and full sector selection.
;               Version 0.6, 20/1/97
;                 Added PROFILES routine.
;                 Added AXES button.
;                 Modified sector list to be 8 OR LESS sectors.
;                 Modified so that full detector and use_dets index into sorted detector list.
;                 Added DETECTOR vs SECTOR format.
;                 Added full_only tag to output_info.
;                 Changed definition of output_info.full_sector and added Sum option.
;                 Updated no. of energy bands in display when read new data set.
;               Version 0.7, 15/4/97
;                 Set up .cal, .div and .rat tags for new file.
;               Version 0.8, 15/5/97
;                 Added header structure version number to display.
;                 Protect against full detector or sector being out of range in new data set.
;                 Added .gz decompression to restore file with ies_gunzip.
;                 Display all energy bands if HIST data.
;                 Changed XPDMENU to CW_PDMENU as the latter is now obsolete.
;               Version 0.9, 7/4/98
;                 Added spin average tp DETECTOR vs SECTOR format.
;                 Added instrument type to get file to eliminate problem with
;                 PICKFILE not dealing properly with directories other than
;                 initial directory.
;               Version 1.0, 1/9/98
;                 Fixed bug with initialization of full sector number.
;               Version 1.1, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Added PITCH angle plot.
;                 Added MF file type.
;               Version 1.2, 11/11/98
;                 Added MFE/MF distinction. and ies_concatenate.
;               Version 1.3, 13/12/99
;                 Replaced PICKFILE with ies_pickfile.
;               Version 1.4, 11/2/00
;                 Added RAPID data to list of data without integral channel
;                 Changed to allow struct to be zeroed outside for bands.
;               Version 1.5, 10/5/00
;                 Added check for POLAR before retrieving ephemeris data etc
;           	Version 1.6, MKC, 28/5/00
;                 Added version no check for SENSITIVE keyword.
;           	Version 1.7, MKC, 20/8/00
;                 Modified to cope with large nos of detectors
;           	Version 1.8, MKC, 31/8/00
;                 Modified to prevent more bands than channels being selected for lineplot.
;                 Added RAPID option in restore files.
;                 Corrected bug in DN. Used bands.nbands instead of header.nbands.
;           	Version 1.9, MKC, 17/9/00
;                 Modified to default to using all bands in LINEPLOT option if
;                 struct not set  since this causes confusion.
;           	Version 2.0, MKC, 26/10/00
;                 Modified to allow STRING verision number.
;           	Version 2.1, MKC, 15/01/02
;                 Corrected bug in retrieving mf and at for new file.
;               Version 2.2, 13/02/02
;                 Changed FILTER for DAT files.
;               Version 2.3, 27/02/02
;                 Dealt with MF contained in structure through mvals.
;               Version 2.4, 12/03/02
;                 Fixed bug with sensitivity of ENERGY BAND button.
;                 Changed band selected for integral channel.
;               Version 2.5, 08/05/02
;                 Moved ies_restore_file to separate file.
;                 Set up output_info and restore file within ies_getset.
;               Version 2.6 12/07/02
;                 Used ies_tag_exists.
;               Version 2.7 15/11/02
;                 Changed so that looks for RAPID data first instead of POLAR.
;               Version 2.8 08/02/03
;                 Added GYRORADIUS vs ENERGY format.
;                 Added OVERLAY plot.
;               Version 2.9, 29/09/03
;                 Added separate version box.
;                 Added routine ies_data_version.
;                 Changed names of OR, AT and MFE boxes.
;                 Disable GYRORADIUS if magnetic field direction only.
;                 Changed ies_do_format argument list.
;                 Added npitchs tag to output_info.
;                 Used new routine for instrument test.
;                 Changed ies get dir.
;
; Version     :	Version 2.9, 29/09/03
;-
;**********************************************************

;**********************************************************
; blok_ies_getset : Sets up common block.
;                   Stores widget IDs for accessing and current values of various parameters.
;**********************************************************

PRO blok_ies_getset, ndetectors, nsectors, start_time, end_time

  ; assumed start_time, end_time passed by value

  COMMON ies_getset_block, struct

  ; set up array to use all detectors

  use_dets  = REPLICATE(1,ndetectors)

  ; set up full button value

  full = 1

  ; set up full detector string

  fulld = 0

  ; set up full sector

  fulls = 0

  ; set up axes button

  axes = 0

  ; check if structure already set up
  ; so that previous settings may be saved
  ; but protect against previous settings messing up new plot

  IF N_TAGS(struct) GT 0 THEN BEGIN

    ; check no. of detectors

    IF ndetectors EQ N_ELEMENTS(struct.use_dets) THEN BEGIN

      use_dets(*) = struct.use_dets

    ENDIF

    ; check time interval

    IF ( struct.sttime GE start_time AND struct.entime LE end_time ) THEN BEGIN

      start_time = struct.sttime
      end_time   = struct.entime

    ENDIF

    ; set up full button value

    full  = struct.full

    fulld = struct.fulld < ndetectors

    fulls = struct.fulls < nsectors  ; .fulls = 0 is sum

    ; set up axes

    axes = struct.axes

    ; propogate some ids

    id_sttime    = struct.id_sttime
    id_entime    = struct.id_entime
    IF ndetectors EQ N_ELEMENTS(struct.id_detectors) THEN id_detectors = struct.id_detectors $
                                                     ELSE id_detectors = LONARR(ndetectors)
    id_files_parent = struct.id_files_parent
    id_times_parent = struct.id_times_parent
    id_detectors_parent = struct.id_detectors_parent
    id_files_child = struct.id_files_child
    id_times_child = struct.id_times_child
    id_detectors_child = struct.id_detectors_child
    id_type   = struct.id_type
    id_version   = struct.id_version
    id_info   = struct.id_info
    id_proc = struct.id_proc
    id_fulld = struct.id_fulld
    id_fulls = struct.id_fulls
    id_full = struct.id_full
    id_infoid = struct.id_infoid
    id_energy = struct.id_energy
    id_axes   = struct.id_axes
    id_format_parent = struct.id_format_parent
    id_format_child = struct.id_format_child
    id_top = struct.id_top
    id_or = struct.id_or
    id_at = struct.id_at
    id_mf = struct.id_mf

  ENDIF ELSE BEGIN

    id_sttime    = 0L
    id_entime    = 0L
    id_detectors = LONARR(ndetectors)
    id_files_parent = 0L
    id_times_parent = 0L
    id_detectors_parent = 0L
    id_files_child = 0L
    id_times_child = 0L
    id_detectors_child = 0L
    id_type   = 0L
    id_version = 0L
    id_info   = 0L
    id_proc = 0L
    id_fulld = 0L
    id_fulls = 0L
    id_full = 0L
    id_infoid = 0L
    id_energy = 0L
    id_axes = 0L
    id_format_parent = 0L
    id_format_child = 0L
    id_top = 0L
    id_or = 0L
    id_at = 0L
    id_mf = 0L

  ENDELSE

  ; set up structure, use anonymous structure so can be extended

  struct = { sttime:start_time,  entime:end_time, use_dets:use_dets, $
             full:full, fulld:fulld, fulls:fulls, axes:axes, $
             id_sttime:id_sttime, id_entime:id_entime, id_detectors:LONARR(ndetectors), $
             id_files_parent:id_files_parent, id_files_child:id_files_child, $
             id_times_parent:id_times_parent, id_times_child:id_times_child, $
             id_detectors_parent:id_detectors_parent, id_detectors_child:id_detectors_child, $
             id_type:id_type, id_version:id_version, id_info:id_info, id_proc:id_proc, id_infoid:id_infoid, $
             id_fulld:id_fulld, id_fulls:id_fulls, id_full:id_full, id_energy:id_energy, id_axes:id_axes, $
             id_format_parent:id_format_parent, id_format_child:id_format_child, id_top:id_top, $
             id_or:id_or, id_at:id_at, id_mf:id_mf, quit:0 }

  ; NB this must be set up separately to avoid collapsing if ndetectors =1

  struct.id_detectors = id_detectors

END

;**********************************************************
; ies_data_version : gets version string
;**********************************************************

FUNCTION ies_data_version, full_header

  ; set data version

  IF ies_tag_exists(full_header,'VERSION') THEN BEGIN

    IF datatype(full_header.version) EQ 'STR' THEN vs = full_header.version $
      ELSE vs = STRTRIM(STRING(full_header.version,FORMAT='(F3.1)'),2)

  ENDIF ELSE vs = '***'

  RETURN, vs

END

;**********************************************************
; ies_check_full : check full detector allowed
;                  When detector is deselected then need to check if full
;                  detector is still allowed.
;**********************************************************

PRO ies_check_full, struct, format

  IF (NOT struct.full) AND (struct.fulld LT N_ELEMENTS(struct.use_dets)) AND format NE 'ENERGY vs SECTOR' THEN BEGIN

    IF NOT struct.use_dets(struct.fulld) THEN BEGIN

      detector_list = WHERE(struct.use_dets,ndetectors)
      IF ndetectors GT 0 THEN struct.fulld = detector_list((LONG(TOTAL(struct.use_dets(0:struct.fulld)))-1) > 0) $
                          ELSE struct.fulld = 0
      WIDGET_CONTROL, struct.id_fulld, SET_DROPLIST_SELECT=struct.fulld

    ENDIF

  ENDIF

END

;**********************************************************
; ies_do_files : widgets for files output
;**********************************************************

PRO ies_do_files, struct, files

  row = WIDGET_BASE  ( struct.id_files_parent, /ROW )

  ; must set up child widget first because still want to destroy it later

  struct.id_files_child = row

  label  = WIDGET_LABEL ( row, VALUE='DATA FILES : ' )

  text = WIDGET_TEXT  ( row, VALUE=files(1<(N_ELEMENTS(files)-1):0>(N_ELEMENTS(files)-1)), $
                                          XSIZE=MAX(STRLEN(files)), $
                                          YSIZE=(N_ELEMENTS(files)-1)>1 )

END

;**********************************************************
; ies_do_format : widget for format
;**********************************************************

FUNCTION ies_do_format, struct, mf, format

  ; check whether magnetic field strength and direction are available

  mf_strength = 0
  mf_direction = 0
  IF KEYWORD_SET(mf) THEN BEGIN
    mf_direction = 1
    IF STRPOS(TAG_NAMES(mf,/STRUCTURE_NAME),'MFE') GE 0 OR STRPOS(TAG_NAMES(mf,/STRUCTURE_NAME),'FGM') GE 0 THEN mf_strength=1
  ENDIF

  IF mf_strength AND mf_direction THEN BEGIN

    items = [ {CW_PDMENU_S, 1, 'CHANGE DISPLAY'}, $
              {CW_PDMENU_S, 1, 'SPECTRAL'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'GYRORADIUS vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'LINEPLOT'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'GYRORADIUS vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'PROFILES'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'GYRORADIUS vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'OVERLAY'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'GYRORADIUS vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'ANGLE/ANGLE'}, $
              {CW_PDMENU_S, 2, 'TIMELINE'} ]

  ENDIF ELSE IF mf_direction THEN BEGIN

    items = [ {CW_PDMENU_S, 1, 'CHANGE DISPLAY'}, $
              {CW_PDMENU_S, 1, 'SPECTRAL'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'LINEPLOT'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'PROFILES'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'OVERLAY'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'PITCH ANGLE vs ENERGY'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'ANGLE/ANGLE'}, $
              {CW_PDMENU_S, 2, 'TIMELINE'} ]

    IF format EQ 'GYRORADIUS vs ENERGY' THEN format = 'ENERGY vs DETECTOR'

  ENDIF ELSE BEGIN

    items = [ {CW_PDMENU_S, 1, 'CHANGE DISPLAY'}, $
              {CW_PDMENU_S, 1, 'SPECTRAL'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'LINEPLOT'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'PROFILES'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 1, 'OVERLAY'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs DETECTOR'}, $
              {CW_PDMENU_S, 0, 'ENERGY vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'SECTOR vs DETECTOR'}, $
              {CW_PDMENU_S, 2, 'DETECTOR vs SECTOR'}, $
              {CW_PDMENU_S, 0, 'ANGLE/ANGLE'}, $
              {CW_PDMENU_S, 2, 'TIMELINE'} ]

    IF format EQ 'PITCH ANGLE vs ENERGY' OR format EQ 'GYRORADIUS vs ENERGY' THEN format = 'ENERGY vs DETECTOR'

  ENDELSE

  struct.id_format_child = WIDGET_BASE  ( struct.id_format_parent, /COLUMN )

  dummy = CW_PDMENU ( struct.id_format_child, items, UVALUE='CD', /RETURN_FULL_NAME)

  RETURN, format

END


;**********************************************************
; ies_do_times : widgets for times input
;**********************************************************

PRO ies_do_times, struct, full_header

  ; time interval selection

  col    = WIDGET_BASE  ( struct.id_times_parent, /COLUMN )

  struct.id_times_child = col

  ; check time interval exists
  ; must set up child widget first because still want to destroy it later

  IF ( full_header.tstop LE full_header.tstart ) THEN RETURN

  ; show time interval

  label  = WIDGET_LABEL ( col, VALUE = 'TIME INTERVAL' )

  row = WIDGET_BASE ( col, /ROW)

  ; set time sliders

  col = WIDGET_BASE ( row, /COLUMN)

  label  = WIDGET_LABEL ( col, VALUE = 'START TIME : ' +  anytim2cal ( full_header.tstart, form=9)  )

  struct.id_sttime = ies_tslider  ( col, $
                                    MINIMUM = full_header.tstart, $
                                    MAXIMUM = full_header.tstop, $
                                    TITLE   = 'Start Time', $
                                    UVALUE  = 'ST',$
                                    XSIZE   = 248, $
                                    /EDIT)

  col = WIDGET_BASE ( row, /COLUMN)

  label  = WIDGET_LABEL ( col, VALUE = '    END   TIME : ' +  anytim2cal ( full_header.tstop, form=9 )  )

  struct.id_entime = ies_tslider  ( col, $
                                    MINIMUM = full_header.tstart, $
                                    MAXIMUM = full_header.tstop, $
                                    TITLE   = 'End Time', $
                                    UVALUE  = 'EN',$
                                    XSIZE   = 248, $
                                    /EDIT)

END

;**********************************************************
; ies_do_detectors : widgets for detector input
;**********************************************************

PRO ies_do_detectors, struct, full_header

  ; detector selection

  col    = WIDGET_BASE  ( struct.id_detectors_parent, /COLUMN )

  struct.id_detectors_child = col

  row    = WIDGET_BASE  ( col, /ROW)

  label  = WIDGET_LABEL ( row, VALUE = 'FULL WINDOW : DETECTOR' )

  ; sort detectors

  det_sort = SORT(full_header.detectors)

  ; set up widget ID of full window detector
  ; NB full_detector may have been mucked around with a bit

  struct.id_fulld = WIDGET_DROPLIST  ( row, VALUE=[full_header.ptitle(det_sort),'Sum'], UVALUE='FD', /FRAME)

  label  = WIDGET_LABEL ( row, VALUE = 'SECTOR' )

  ; set up widget ID of full window sector

  struct.id_fulls = WIDGET_DROPLIST  ( row, VALUE=['Sum',STRTRIM(INDGEN(full_header.nsectors),2)], UVALUE='FS', /FRAME)

  ; set up button for no detectors

  label  = WIDGET_LABEL ( row, VALUE = '  ONLY' )

  struct.id_full = WIDGET_BUTTON ( row, VALUE='ON', UVALUE='FU' )

  ; set up button for no extra axes

  label  = WIDGET_LABEL ( row, VALUE = '  XTRA' )

  struct.id_axes = WIDGET_DROPLIST ( row, VALUE=['NONE','AXES','LUTS'], UVALUE='AX', /FRAME )

  FOR k = 0, full_header.ndetectors-1 DO BEGIN

    IF (k MOD 16) EQ 0 THEN BEGIN

      row    = WIDGET_BASE  ( col, /ROW)

      label  = WIDGET_LABEL ( row, VALUE = 'DETECTORS ' )

      buttons = WIDGET_BASE  ( row, /ROW, /NONEXCLUSIVE)

    ENDIF

    d = det_sort(k)

    struct.id_detectors(k) = WIDGET_BUTTON ( buttons, $
                                VALUE=full_header.ptitle(d), $
                                UVALUE= 'DT'+STRTRIM(k,1))

  ENDFOR

END

;**********************************************************
; ies_set_values : set values in widgets
;**********************************************************

PRO ies_set_values, full_header, struct, files, format, procname, eph, at, mf

  ; set procname

  WIDGET_CONTROL, struct.id_proc, SET_VALUE=procname+' : '+format

  ; check full detector

  ies_check_full, struct, format

  ; test if ENERGY vs SECTOR format
  ; or if DETECTOR vs SECTOR format

  IF format EQ 'ENERGY vs SECTOR' THEN BEGIN
    IF full_header.nsectors GT 8 THEN sector_list = (INDGEN(8)*full_header.nsectors)/8 $
                                 ELSE sector_list = INDGEN(full_header.nsectors)
    WIDGET_CONTROL, struct.id_fulls, SET_VALUE=STRTRIM(sector_list,2)
    struct.fulls = struct.fulls < (N_ELEMENTS(sector_list)-1)
  ENDIF ELSE IF format EQ 'DETECTOR vs SECTOR' THEN BEGIN
    WIDGET_CONTROL, struct.id_fulls, SET_VALUE=['Sum',STRTRIM(INDGEN(full_header.nsectors),2)]
  ENDIF ELSE BEGIN
    WIDGET_CONTROL, struct.id_fulls, SET_VALUE=['Sum',STRTRIM(INDGEN(full_header.nsectors),2)]
  ENDELSE

  ; set full detector

  WIDGET_CONTROL, struct.id_fulld, SET_DROPLIST_SELECT=struct.fulld

  IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_fulld, $
    SENSITIVE=(format NE 'DETECTOR vs SECTOR') AND (format NE 'PITCH ANGLE vs ENERGY') AND (format NE 'GYRORADIUS vs ENERGY')

  ; set full sector

  WIDGET_CONTROL, struct.id_fulls, SET_DROPLIST_SELECT=struct.fulls

  IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_fulls, $
    SENSITIVE=(format NE 'SECTOR vs DETECTOR') AND (format NE 'PITCH ANGLE vs ENERGY') AND (format NE 'GYRORADIUS vs ENERGY')

  ; set full display button

  WIDGET_CONTROL, struct.id_full, SET_BUTTON=0
  IF struct.full THEN BEGIN
    WIDGET_CONTROL, struct.id_full, /SET_BUTTON
    WIDGET_CONTROL, struct.id_full, SET_VALUE='On'
  ENDIF ELSE WIDGET_CONTROL, struct.id_full, SET_VALUE='Off'

  ; set axes button

  WIDGET_CONTROL, struct.id_axes, SET_DROPLIST_SELECT=struct.axes

  ; set detector buttons

  FOR k = 0, full_header.ndetectors-1 DO WIDGET_CONTROL, struct.id_detectors(k), SET_BUTTON=0

  IF !VERSION.release GT '5.0.3' THEN $
    FOR k = 0, full_header.ndetectors-1 DO WIDGET_CONTROL, struct.id_detectors(k), $
      SENSITIVE=(struct.full EQ 0 OR format EQ 'DETECTOR vs SECTOR' OR format EQ 'PITCH ANGLE vs ENERGY' OR (format NE 'GYRORADIUS vs ENERGY'))

  list = WHERE ( struct.use_dets , nlist)

  FOR k = 0, nlist-1 DO WIDGET_CONTROL, struct.id_detectors(list(k)), /SET_BUTTON

  IF !VERSION.release GT '5.0.3' THEN $
    FOR k = 0, nlist-1 DO WIDGET_CONTROL, struct.id_detectors(list(k)),  $
      SENSITIVE=(struct.full EQ 0 OR format EQ 'DETECTOR vs SECTOR' OR format EQ 'PITCH ANGLE vs ENERGY' OR (format NE 'GYRORADIUS vs ENERGY'))

  ; set time boundaries

  IF ( full_header.tstop GT full_header.tstart ) THEN BEGIN

    WIDGET_CONTROL, struct.id_sttime, SET_VALUE=struct.sttime

    WIDGET_CONTROL, struct.id_entime, SET_VALUE=struct.entime

  ENDIF

  ; set datatype

  WIDGET_CONTROL, struct.id_type, SET_VALUE=full_header.datatype

  ; set data version

  WIDGET_CONTROL, struct.id_version, SET_VALUE=ies_data_version(full_header)

  ; set data info

  WIDGET_CONTROL, struct.id_info, SET_VALUE='NO. OF POINTS=' + STRTRIM(full_header.npoints,2) + $
                                            '  NO. OF ENERGIES=' + STRTRIM(full_header.nbands,2) + $
                                            '  NO. OF DETECTORS=' + STRTRIM(full_header.ndetectors,2) + $
                                            '  NO. OF SECTORS=' + STRTRIM(full_header.nsectors,2)
  ; set or, at and mf buttons

  WIDGET_CONTROL, struct.id_or, SET_BUTTON=0

  IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_or, SENSITIVE=0

  WIDGET_CONTROL, struct.id_at, SET_BUTTON=0

  IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_at, SENSITIVE=0

  WIDGET_CONTROL, struct.id_mf, SET_BUTTON=0

  IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_mf, SENSITIVE=0

  IF KEYWORD_SET(eph) THEN BEGIN

    WIDGET_CONTROL, struct.id_or, /SET_BUTTON

    IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_or, SENSITIVE=0

  ENDIF

  IF KEYWORD_SET(at) THEN BEGIN

    WIDGET_CONTROL, struct.id_at, /SET_BUTTON

    IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_at, SENSITIVE=0

  ENDIF

  IF KEYWORD_SET(mf) THEN BEGIN

    WIDGET_CONTROL, struct.id_mf, /SET_BUTTON

    IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_mf, SENSITIVE=0

    IF STRMID(TAG_NAMES(mf,/STRUCTURE_NAME),0,3) EQ 'CMF' THEN WIDGET_CONTROL, struct.id_mf, SET_VALUE='MF'

    IF STRMID(TAG_NAMES(mf,/STRUCTURE_NAME),0,3) EQ 'MFE' THEN WIDGET_CONTROL, struct.id_mf, SET_VALUE='MFE'

  ENDIF

END

;**********************************************************
; ies_getset_event : event handler for main routine
;**********************************************************

PRO ies_getset_event, event

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  COMMON ies_getset_block, struct

  WIDGET_CONTROL, event.id, GET_UVALUE = uvalue

  CASE STRMID(uvalue,0,2) OF

   'DN'  : BEGIN

             CASE event.value OF

               'DONE' : BEGIN

                  ; set detector for full display

                  full_detector = struct.fulld

                  ; get band info

                  bands = ies_bands(/GET_CURRENT)

                  ; get copy of counts for relevant time interval summed over required channels
                  ; NB may update full_detector

                  WIDGET_CONTROL, struct.id_infoid, SET_VALUE='Extracting data, please wait'

                  ies_extdata, struct, bands, full_header, full_data, full_detector, header, data, errmsg, $
                    FULL_DETECTOR_ONLY=(struct.full OR output_info.format EQ 'ENERGY vs SECTOR') $
                    AND output_info.format NE 'DETECTOR vs SECTOR' $
                    AND output_info.format NE 'GYRORADIUS vs ENERGY' $
                    AND output_info.format NE 'PITCH ANGLE vs ENERGY', MF=mf, AT=at, EPHEM=ephem

                  IF errmsg NE '' THEN BEGIN
                    WIDGET_CONTROL, struct.id_infoid, SET_VALUE=errmsg
                  ENDIF ELSE BEGIN
                    WIDGET_CONTROL, event.top, /DESTROY
                  ENDELSE

                  ; check if full detector only used

                  output_info.full_only = struct.full

                  ; set up full detector

                  output_info.full_detector = full_detector

                  ; set up full sector

                  output_info.full_sector = struct.fulls

                  ; set up full energy band

                  output_info.full_band = 0

                  ; set up axes button

                  output_info.axes = struct.axes

                  ; set up band_labels
                  ; defined here because after this point lose connection with original bands
                  ; check whether to apply bottom and top channel selection
                  ; only apply if bands.nbands=0

                  IF bands.nbands EQ 0 THEN BEGIN

                    FOR k = 0, header.nbands-1 DO BEGIN
                      output_info.band_labels(k) = STRTRIM(bands.lb(bands.botchan+k),1) + ':' + STRTRIM(bands.ub(bands.botchan+k),1)
                    ENDFOR

                  ENDIF ELSE BEGIN

                    band_list  =  WHERE ( bands.use_bands )          ; get bands
                    FOR k = 0, bands.nbands-1 DO BEGIN
                      output_info.band_labels(k) = STRTRIM(bands.lb(band_list(k)),1) + ':' + STRTRIM(bands.ub(band_list(k)),1)
                    ENDFOR

                  ENDELSE

                END

               'QUIT'  : BEGIN
                  struct.quit = 1
                  WIDGET_CONTROL, event.top, /DESTROY
                END

               'ADD FILE.RAPID': res = ies_restore_file( ies_get_dir('IF',/RAPID), /ADD)
               'ADD FILE.IES'  : res = ies_restore_file( ies_get_dir('IES',/POLAR), /ADD)
               'ADD FILE.IPS'  : res = ies_restore_file( ies_get_dir('IPS',/POLAR), /ADD)
               'ADD FILE.HISTe': res = ies_restore_file( ies_get_dir('HISTe',/POLAR), /ADD)
               'ADD FILE.HISTp': res = ies_restore_file( ies_get_dir('HISTp',/POLAR), /ADD)
               'ADD FILE.HK'   : res = ies_restore_file( ies_get_dir('HK',/POLAR), /ADD)
               'ADD FILE.MF'   : res = ies_restore_file( ies_get_dir('MF',/POLAR), /ADD)

               'NEW FILE.RAPID': res = ies_restore_file( ies_get_dir('IF',/RAPID), /NEW)
               'NEW FILE.IES'  : res = ies_restore_file( ies_get_dir('IES',/POLAR), /NEW)
               'NEW FILE.IPS'  : res = ies_restore_file( ies_get_dir('IPS',/POLAR), /NEW)
               'NEW FILE.HISTe': res = ies_restore_file( ies_get_dir('HISTe',/POLAR), /NEW)
               'NEW FILE.HISTp': res = ies_restore_file( ies_get_dir('HISTp',/POLAR), /NEW)
               'NEW FILE.HK'   : res = ies_restore_file( ies_get_dir('HK',/POLAR), /NEW)
               'NEW FILE.MF'   : res = ies_restore_file( ies_get_dir('MF',/POLAR), /NEW)

                ELSE  : MESSAGE, 'IMPLEMENTATION ERROR', /TRACEBACK

             ENDCASE

           END

   'ST'  : BEGIN

             ; set slide start time

             struct.sttime = event.value

             ; set slide end time if changed

             WIDGET_CONTROL, struct.id_entime, GET_VALUE = entime

             IF ( event.value GT entime ) THEN BEGIN
               WIDGET_CONTROL, struct.id_entime, SET_VALUE = event.value
               struct.entime = event.value
             ENDIF

           END

   'EN'  : BEGIN

             ; set slide end time

             struct.entime = event.value

             ; set slide start time if changed

             WIDGET_CONTROL, struct.id_sttime, GET_VALUE = sttime

             IF ( event.value LT sttime ) THEN BEGIN
               WIDGET_CONTROL, struct.id_sttime, SET_VALUE = event.value
               struct.sttime = event.value
             ENDIF

           END

   'DT' : BEGIN

            ch = FIX(STRMID(uvalue,2,1))

            struct.use_dets(ch) = 1 - struct.use_dets(ch)

            ies_check_full, struct, output_info.format

          END

   'LG' : IF event.type EQ 1 THEN dsp_strarr, ies_logo_info()

   'FD' : BEGIN

            struct.fulld = event.index

            ies_check_full, struct, output_info.format

          END

   'FS' : struct.fulls = event.index

   'FU' : BEGIN

            struct.full = 1 - struct.full

            IF struct.full THEN $
                WIDGET_CONTROL, struct.id_full, SET_VALUE='On' $
            ELSE $
                WIDGET_CONTROL, struct.id_full, SET_VALUE='Off'

            ies_check_full, struct, output_info.format

            ies_set_values, full_header, struct, files, output_info.format, output_info.procname, ephem, at, mf

          END

   'AX' : struct.axes = event.index

   'SC' : BEGIN

            IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_top, SENSITIVE=0

            bands=ies_bands(GROUP=struct.id_top)

            IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, struct.id_top, SENSITIVE=1

            IF bands.nbands EQ 0 THEN dummy = '   BOTTOM  CHANNEL : ' + STRTRIM(bands.botchan,2) + $
                                              '   TOP CHANNEL : ' + STRTRIM(bands.topchan,2)  $
                                 ELSE dummy = '   NUMBER OF ENERGY BANDS : ' + STRTRIM(bands.nbands,2)

            ; get no. of bands

            band_list = WHERE ( bands.use_bands, nbands )

            IF bands.nbands EQ 0 THEN end_band = bands.topchan ELSE end_band = nbands-1

            IF nbands LE 0 THEN BEGIN
              WIDGET_CONTROL, struct.id_infoid, SET_VALUE='WARNING, No energy bands selected'
            ENDIF ELSE BEGIN
              dummy = dummy + '   INTEGRAL CHANNEL : '+$
                STRTRIM(bands.lb(band_list[0]),2)+'-'+STRTRIM(bands.ub(band_list[0]),2)
            ENDELSE

            WIDGET_CONTROL, struct.id_energy, SET_VALUE=dummy

          END

   'CD'  : BEGIN

            output_info.procname = STRMID(event.value,15,STRLEN(event.value)-15) ; chop off 'CHANGE DISPLAY.'

             IF STRPOS(output_info.procname,'.') GT 0 THEN BEGIN
               output_info.format = STRMID(output_info.procname,STRPOS(output_info.procname,'.')+1, $
                                           STRLEN(output_info.procname)-STRPOS(output_info.procname,'.')-1)
               output_info.procname = STRMID(output_info.procname,0,STRPOS(output_info.procname,'.'))
             ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN
               output_info.format = 'ENERGY vs DETECTOR'
             ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN
               output_info.format = 'SECTOR vs DETECTOR'
             ENDIF ELSE MESSAGE, 'IMPLEMENTATION ERROR', /TRACEBACK

             ies_set_values, full_header, struct, files, output_info.format, output_info.procname, ephem, at, mf

           END

    ELSE  : RETURN

  ENDCASE

END

;**********************************************************
; main routine for widget
;**********************************************************

FUNCTION ies_getset, GROUP=group

   ; define common blocks used internally for draw widgets
   ; full_header   = structure containing original data header information
   ; full_data     = structure array containing original data
   ; header        = structure containing extracted data header information
   ; data          = structure array containing extracted data
   ; output_info   = structure containing plotting and processing info
   ; output_chans  = array containing processed channel positions
   ; output_array  = array containing processed data for plotting
   ; ids           = structure giving widget IDs
   ; dnfull        = co-ordinate conversion array for full window
   ; dnscroll      = co-ordinate conversion array for magnified windows
   ; files         = STRARR giving data file names
   ; ephem         = structure containing ephemeris data
   ; mf            = structure containing magnetic field data
   ; at            = structure containing attitude data

   COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                 ids, dnfull, dnscroll, files, ephem, mf, at

  COMMON ies_getset_block, struct

  ; check if already running

  IF XREGISTERED ("ies_getset") THEN  RETURN, 1 ; quit

  ; set up files array

  IF N_ELEMENTS(files) EQ 0 THEN files = ''

  ; restore file

  IF NOT KEYWORD_SET(full_header) OR NOT KEYWORD_SET(full_data) THEN BEGIN

    path = ies_get_dir('IF',/RAPID)

    IF path EQ '' THEN path = ies_get_dir('IES',/POLAR)

    IF NOT ies_restore_file(path, /NEW) THEN RETURN, 1 ; quit

  ENDIF

  ; set up output_info structure
  ; if min or max undefined then force calculation from data
  ; initialize fixed button labels if undefined

  IF N_ELEMENTS(output_info) EQ 0 THEN $
    output_info = { log:1, cal:full_header.cal, div:full_header.div, rat:full_header.rat, $
                    shift:0, subtract:0, track:1, rng:0.0, npitchs:180, $
                    button_labels:STRARR(12), button_values:INTARR(12), procname:'SPECTRAL', $
                    full_only:1, full_detector:0, full_sector:0, full_band:0, axes:0, $
                    band_labels:STRARR(256), $
                    noutputs:0, format:'ENERGY vs DETECTOR', ptitle:STRARR(64), time:0.0D0, ysamples:0, $
                    minimum:0.0, maximum:0.0, elow:0.0, ehigh:0.0, min:0, max:0, elo:0, ehi:0 }

  ; fix default no. of bands if first time into getset
  ; NB check whether clean tag present or not first

  IF ies_tag_exists(full_header,'CLEAN') THEN clean = full_header.clean ELSE clean = 0

  ; check if integral channel present

  IF clean EQ 0 AND (NOT ies_instrument(full_header.datatype,/HIST)) AND (NOT ies_instrument(full_header.datatype,/RAPID)) THEN def_bands = 0 ELSE def_bands = -1

  IF NOT KEYWORD_SET(struct) THEN $
    bands=ies_bands(SET_CURRENT=[full_header.nbands,def_bands]) $
  ELSE $
    bands = ies_bands(/GET_CURRENT)

  ; set up common block

  blok_ies_getset, full_header.ndetectors, full_header.nsectors, full_header.tstart, full_header.tstop

  ; ***********************************************************

  ; set up main widget

  IF KEYWORD_SET(group) THEN $
    base = WIDGET_BASE ( TITLE = 'RAL SELECT DATA TOOL', /COLUMN, GROUP_LEADER=group, /MODAL) $
  ELSE $
    base = WIDGET_BASE ( TITLE = 'RAL SELECT DATA TOOL', /COLUMN)

  struct.id_top = base

  ; ***********************************************************

  ; set up logo

  row    = WIDGET_BASE  ( base, /ROW )

  logo   = ies_logo()
  dims   = SIZE(logo)

  logoid = WIDGET_DRAW ( row, XSIZE=dims(1), YSIZE=dims(2), UVALUE='LG', RETAIN=2, /BUTTON_EVENTS)

  ; set up info text widget

  struct.id_infoid = WIDGET_TEXT ( row, VALUE='Ready for input', XSIZE=68, /FRAME)

  ; ***********************************************************

  ; set up pull down menu

  junk = { CW_PDMENU_S,  flags:0, name:'' }

  items = [ {CW_PDMENU_S, 0, 'DONE'}, $
            {CW_PDMENU_S, 1, 'ADD FILE'}, $
            {CW_PDMENU_S, 0, 'RAPID'}, $
            {CW_PDMENU_S, 0, 'IES'}, $
            {CW_PDMENU_S, 0, 'IPS'}, $
            {CW_PDMENU_S, 0, 'HISTe'}, $
            {CW_PDMENU_S, 0, 'HISTp'}, $
            {CW_PDMENU_S, 0, 'HK'}, $
            {CW_PDMENU_S, 2, 'MF'}, $
            {CW_PDMENU_S, 1, 'NEW FILE'}, $
            {CW_PDMENU_S, 0, 'RAPID'}, $
            {CW_PDMENU_S, 0, 'IES'}, $
            {CW_PDMENU_S, 0, 'IPS'}, $
            {CW_PDMENU_S, 0, 'HISTe'}, $
            {CW_PDMENU_S, 0, 'HISTp'}, $
            {CW_PDMENU_S, 0, 'HK'}, $
            {CW_PDMENU_S, 2, 'MF'}, $
            {CW_PDMENU_S, 2, 'QUIT'} ]

  dummy = CW_PDMENU ( base, items, UVALUE='DN', /RETURN_FULL_NAME)

  ; ***********************************************************

  ; data file selection

  struct.id_files_parent   = WIDGET_BASE  ( base, /ROW )

  ; set up files widget

  ies_do_files, struct, files

  ; ***********************************************************

  ; datatype display

  row    = WIDGET_BASE  ( base, /ROW )

  label = WIDGET_LABEL ( row, VALUE='DATA TYPE : ')

  struct.id_type =  WIDGET_LABEL  ( row, VALUE = full_header.datatype, /DYNAMIC_RESIZE, /FRAME )

  ; ***********************************************************

  ; version display

  label = WIDGET_LABEL ( row, VALUE='   VERSION : ')

  struct.id_version =  WIDGET_LABEL  ( row, VALUE=ies_data_version(full_header), /DYNAMIC_RESIZE, /FRAME )

  ; ***********************************************************

  ; program selection

  ; set up text widgets

  row    = WIDGET_BASE  ( base, /ROW )

  label  = WIDGET_LABEL ( row, VALUE='DISPLAY PROCEDURE : ')

  ; allow enough space for any output

  struct.id_proc = WIDGET_TEXT ( row, UVALUE='PR', VALUE='', XSIZE=35 )

  ; set up format menu

  struct.id_format_parent = WIDGET_BASE  ( row, /COLUMN )

  output_info.format = ies_do_format( struct, mf, output_info.format)

  ; ***********************************************************

  ; set no. of channels and points
  ; NB  IDL on shaper doesnt like the size of the string to change
  ; cant get it to work at all on alpha

  row    = WIDGET_BASE  ( base, /ROW )

  struct.id_info = WIDGET_LABEL ( row, VALUE='', /DYNAMIC_RESIZE)

  ; set up or, at, mf indicator button

  buttons = WIDGET_BASE  ( row, /ROW, /NONEXCLUSIVE)

  struct.id_or = WIDGET_BUTTON ( buttons, VALUE='EPHEMERIS', UVALUE= 'OR')
  struct.id_at = WIDGET_BUTTON ( buttons, VALUE='ATTITUDE', UVALUE= 'AT')
  struct.id_mf = WIDGET_BUTTON ( buttons, VALUE='MAGN. FIELD', UVALUE= 'MF')

  ; ***********************************************************

  ; channel selection

  row    = WIDGET_BASE  ( base, /ROW )

  ; set up button

  energyid = WIDGET_BUTTON( row, UVALUE='SC', VALUE='SELECT ENERGY BANDS')

  IF bands.nbands EQ 0 THEN dummy = '   BOTTOM  CHANNEL : ' + STRTRIM(bands.botchan,2) + $
                                           '   TOP CHANNEL : ' + STRTRIM(bands.topchan,2)  $
                       ELSE dummy = '   NUMBER OF ENERGY BANDS : ' + STRTRIM(bands.nbands,2)

  ; get no. of bands

  band_list = WHERE ( bands.use_bands, nbands )

  ;IF bands.nbands EQ 0 THEN end_band = bands.topchan ELSE end_band = nbands-1

  IF nbands NE 0 THEN BEGIN
    dummy = dummy + '   INTEGRAL CHANNEL : '+$
                STRTRIM(bands.lb(band_list[0]),2)+'-'+STRTRIM(bands.ub(band_list[0]),2)
  ENDIF

  struct.id_energy = WIDGET_LABEL ( row, VALUE=dummy, /DYNAMIC_RESIZE )

  ; ***********************************************************

  ; set up widget ID of time displays

  struct.id_times_parent  = WIDGET_BASE  ( base, /COLUMN, /FRAME)

  ; set up times widget

  ies_do_times, struct, full_header

  ; ***********************************************************

  ; set up widget ID of detector displays

  struct.id_detectors_parent  = WIDGET_BASE  ( base, /COLUMN, /FRAME)

  ; set up detector widget

  ies_do_detectors, struct, full_header

  ; ***********************************************************

  WIDGET_CONTROL, base, /REALIZE

  ; ***********************************************************

  ; set up logo

  WIDGET_CONTROL, logoid, GET_VALUE=win

  WSET, win

  TV, logo

  ; set up values

  ies_set_values, full_header, struct, files, output_info.format, output_info.procname, ephem, at, mf

  ; ***********************************************************

  XMANAGER, "ies_getset", base, /MODAL

  ; return

  RETURN, struct.quit

  ; ***********************************************************

END
