;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_GSE_TO_SC
;
; Purpose     :	Converts vector in GSE coords to vector in SC coords.
;
; Explanation :	Defines coord system relative to sun vector.
;               Rotates coord system by right ascension and declination.
;               This is not uniquely defined.
;               Rotates so that sun vector is on original meridian to define uniquely.
;
; Use         : < v = ies_gse_to_sc( ra, dec, tim, u, t, offset) >
;
; Inputs      : ra    : FLTARR(npoints) GSE right ascension over period of interest
;               dec   : FLTARR(npoints) GSE declination over period of interest
;               tim   : DBLARR(npoints) TAI times
;               u     : FLTARR(3,nvals) GSE coords
;               t     : DBLARR(nvals) TAI times
;               offset: FLOAT or FLTARR(npoints) sun sensor offset
;
; Opt. Inputs : None.
;
; Outputs     : v     : FLTARR(2,nvals) SC coords : 0 = azimuth angle 0->2*pi (with 0 along sun direction)
;                                                   1 = polar angle   0->pi (with 0 along spin axis)
;
; Opt. Outputs:	None.
;
; Keywords    : XYZ : Returns in S/C XYZ coords.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/10/98
;
; Modified    :
;
; Version     :	Version 0.0, 13/10/98
;-
;**********************************************************

FUNCTION ies_gse_to_sc, ra, dec, tim, u, t, offset, XYZ=xyz

  ; check that sample times overlap

  IF t[0] GT tim[N_ELEMENTS(tim)-1] OR t[N_ELEMENTS(t)-1] LT tim[0] THEN BEGIN
    MESSAGE,'No time overlap', /CONTINUE
    RETURN, FLTARR(2,N_ELEMENTS(t))
  ENDIF

  ; interpolate to get ra and dec at sample times

  r = INTERPOL( ra, tim, t)
  d = INTERPOL( dec, tim, t)

  ; get sines and cosines

  cosr = COS(r)
  sinr = SIN(r)
  cosd = SIN(d) ; COS(90-dec)
  sind = COS(d) ; SIN(90-dec)

  ; get azimuthal rotation
  ; -e is azimuthal position of sun vector after ra and decl rotations in new coord system
  ; want then to fix azimuth relative to this for each vector so rotate by -e

  e = ATAN ( sinr, cosr*cosd )

  ; do ra rotation of axes

  v = FLOAT(u)
  v(0,*) =  cosr*u(0,*) + sinr*u(1,*)
  v(1,*) = -sinr*u(0,*) + cosr*u(1,*)

  ; do dec rotation of axes

  w = v
  w(0,*) = cosd*v(0,*) - sind*v(2,*)
  w(2,*) = sind*v(0,*) + cosd*v(2,*)

  IF KEYWORD_SET(xyz) THEN BEGIN

    ; do az rotation

    x = w
    x(0,*) =  COS(e)*w(0,*) - SIN(e)*w(1,*)
    x(1,*) =  SIN(e)*w(0,*) + COS(e)*w(1,*)

    RETURN, x

  ENDIF ELSE BEGIN

    ; get azimuth and polar angles

    az = REFORM(ATAN(w(1,*), w(0,*)) + e + offset)
    po = REFORM(ASIN(w(2,*)/SQRT(w(0,*)^2+w(1,*)^2+w(2,*)^2)))

    az = (az + 2*!PI) MOD (2*!PI) ; 0->2*pi
    po = !PI/2 - po ; 0->pi

    RETURN, TRANSPOSE(REFORM([az,po],N_ELEMENTS(az),2))

  ENDELSE

END
