;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_GYRO_RADIUS
;
; Purpose     :	Returns gyroradius of particle in km.
;
; Explanation : Returns gyroradius of particle in km.
;
; Use         : < radius = rcs_gyro_radius(kinetic_energy, field_strength, pitch_angle, Re=re, E=e, P=p) >
;
; Inputs      : kinetic_energy : FLOAT energy of particle in keV
;               field_strength : FLOAT field strength in nT
;               pitch_angle    : FLOAT pitch angle in radians.
;
; Opt. Inputs : None.
;
; Outputs     : radius : FLOAT gyroradius of particle in km
;
; Opt. Outputs:	None.
;
; Keywords    : Re : caluculate gyroradius in units of earth radius
;               e  : do for electron
;               p  : do for proton
;
; Written     :	Version 0.0, 23/05/02
;
; Modified    : Version 0.1, 08/02/03
;                 Added pitch angle.
;
; Version     :	Version 0.1, 08/02/03
;-
;**********************************************************

FUNCTION ies_gyro_radius, kinetic_energy, field_strength, pitch_angle, Re=re, E=e, P=p

  ; set up pitch angle

  IF NOT KEYWORD_SET(pitch_angle) THEN pitch_angle = !PI/2

  ; get particle mass in units of MeV/c2

  IF KEYWORD_SET(p) THEN mass = 938.27231 ELSE mass = 0.51099906

  ; get speed of light in m/s

  c = 2.99792458E8

  ; convert energy to MeV

  ke = kinetic_energy/1000.0*SIN(pitch_angle)^2

  ; convert field_strength to Tesla

  b = field_strength/1e9

  ; calculate gyroradius in km

  radius = SQRT(ke*(ke+2*mass))*1e3/c/b

  ; convert to units of earth radii

  IF KEYWORD_SET(re) THEN radius = radius/63781400

  RETURN, radius

END
