;+
; Project     :	POLAR - CEPPAD
;
; Name        :	ies_header_structure
;
; Purpose     :	Returns structure used for data.
;
; Explanation : Returns a structure containing information about the nature and format of
;               data stored in the data structure. IF KEYWORD <experiment> is set then
;               returns values for no_of_points, no_of_detectors, no_of_energies, no_of_sectors, no_of_luts
;               else these must be input. If KEYWORD <experiment> is set then structure is named
;               because the relevant array sizes are determined within the routine. If not set then
;               the structure is anonymous.
;               Originally expected to pass all relevant information through with data structure so that
;               stands alone. However, IES pedestal calibration information is too large to do this
;               sensibly. Moving to idea that some info is stored in calibration files accessed only while
;               processing data. Sensible dividing line might be where information is only relevant to a
;               particular instrument such as IES pedestals (or, perhaps later, HIST efficiencies).
;
; Use         : <struct=ies_header_structure( datatype, mjulday, no_of_points, no_of_detectors, no_of_energies, no_of_sectors, no_of_luts, $
;                                             TIME_START=time_start, TIME_STOP=time_stop, DETECTORS=detectors, $
;                                             TITLE=title, XTITLE=xtitle, YTITLE=ytitle, ZTITLE=ztitle, PTITLE=ptitle, $
;                                             EXPERIMENT=experiment, MODE=mode, ENERGY_EFFICIENCY=energy_efficiency )>
;
; Inputs      : datatype        : STRING indicating type of data.
;               mjulday         : LONG giving modified Julian day number for data.
;               no_of_points    : LONG number of spins.
;               no_of_detectors : LONG number of detectors at each sample
;               no_of_energies  : LONG number of energies at each sample
;               no_of_sectors   : LONG number of sectors at each sample
;               no_of_luts      : LONG number of look up tables
;
; Opt. Inputs : None.
;
; Outputs     : no_of_points     : LONG number of samples.
;               no_of_energies   : LONG number of energies at each sample
;               no_of_sectors    : LONG number of sectors at each sample
;               struct : Structure for header information
;                  .version      : FLOAT giving version number of software
;                  .datatype     : STRING describing data type i.e POLAR/CEPPAD/IES
;                                         normal mode SPIN_AVERAGED
;                  .version      : FLOAT version no.
;                  .npoints      : LONG no. of time samples
;                  .ndetectors   : LONG no.of detectors
;                  .nbands       : LONG no. of energy channels
;                  .nsectors     : LONG no. of azimuthal sectors
;                  .nluts        : LONG no. of LUTs for energy channels to energy bins
;                  .detectors    : LONARR(ndetectors) detector numbers (will reorder
;                                         detectors for display)
;                  .tstart       : DOUBLE data TAI start time
;                  .tstop        : DOUBLE data TAI stop time
;                  .ch_positions : FLTARR (2, nenergies, ndetectors > 1, nluts) tables
;                                         for energy channels to energy bins
;                  .count_cal    : FLTARR(nluts) calibration factor for counts -> full
;                                         counts per cm2 per sr
;                                         NB includes integration time dependent
;                                         mark/space compensation as well as geometric
;                                         factor
;                  .energy_cal   : FLOAT calibration factor for bins to energy [keV] conversion.
;                                        may have energy offset also as element 1.
;                  .energy_efficiency : FLTARR(nbands,nluts) energy efficiency factor for each energy bin for each LUT
;                  .cal          : INT flag indicating data calibrated to /cm2/sr
;                  .rat          : INT flag indicating data calibrated to /sec
;                  .div          : INT flag indicating data calibrated to /keV
;                  .clean        : INT version no. of cleaning procedure (0=raw)
;                  .subtract     : INT flag indicating pedestal subtracted.
;                  .title        : STRING default data set title.
;                  .xtitle       : STRING default time axis title
;                  .ytitle       : STRING default energy axis title
;                  .ztitle       : STRING default counts axis title
;                  .ptitle       : STRARR default title for each detector plot
;
; Opt. Outputs:	None.
;
; Keywords    : TIME_START : earliest sample time in internal format, zero if unspecified.
;               TIME_STOP  : latest sample time in internal format, zero if unspecified.
;               DETECTORS  : Integer giving detector number, 1->no_of_detectors if unspecified.
;               TITLE      : Overall plot title, blank if unspecified.
;               XTITLE     : X axis title, blank if unspecified i.e time.
;               YTITLE     : Y axis title, blank if unspecified i.e energy
;               ZTITLE     : Z axis title, blank if unspecified i.e counts
;               PTITLE     : Individual plot title, detector number if unspecified.
;               EXPERIMENT : STRING indicates should return particular named structure.
;               MODE       : STRING indicates should return particular named structure.
;               ENERGY_EFFICIENCY : FLTARR(nbins,nluts) efficiency for each bin for each LUT
;               ENERGY_CALIBRATION : FLOAT 0 = bin to energy [keV] conversion factor
;                                          1 = energy [keV] offset (optional)
;               COUNT_CALIBRATION : FLOAT counts to counts/cm2/sr factor.
;               LUT_MAP : INTARR(nluts) integration mode for each LUT.
;               SPINS_PER_SAMPLE : INT no of spin summed per sample.
;               VERSION    : STRING giving SW version number.
;               MF         : STRING flag indicating that should create a .mf tag.
;               PA         : STRING flag indicating that should create a .pa tag.
;
; Calls       :	get_utc.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display software.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 12/1/96
;
; Modified    :	Version 0.1, 29/2/96
;                 Converted to use CDS internal time format.
;               Version 0.2, 16/9/96
;                 Changed no_of_luts to 36 for IES.
;                 Changed ch_positions to a float array.
;                 Changed so that returns no. of sectors in data set rather than for expt.
;                 Added CLEAN_n modes.
;                 Added SPIN_n modes.
;                 Got rid of ndetectors, nluts, nenergies.
;                 Added detector reordering to IPS.
;                 no_of_detectors, no_of_luts, no_of_energies now set up externally.
;               Version 0.3, 11/11/96
;                 Added pedestal bin widths, changed tag to pedestals, changed pedestals to FLTARR.
;                 Added subtract, rat, div and cal tags.
;               Version 0.4, 16/1/97
;                 Modified pedestals tag. Added PEDESTALS and LUT_MAP keywords.
;                 Added version number to structure names to allow compatability with earlier data sets.
;               Version 0.5, 14/5/97
;                 Added version no to structure to specify exactly what has been done
;                 Have also changed data structure (structure vs 2)
;               Version 0.6, 4/7/97
;                 Added ENERGY_EFFICIENCY keyword.
;                 Removed PEDESTALS tag and keyword which passed the pedestal histogram data for each detector and mode.
;                 Removed LUT_MAP tag and keyword which gave the mode for each LUT.
;                 Structure vs 3.
;               Version 0.7, 10/10/97
;                 Added energy_calibration and count_calibration keywords.
;                 Put back LUT_MAP tag and keyword which gave the mode for each LUT.
;                 Structure vs 4.
;               Version 0.8, 4/3/98
;                 Changed version no. of SW.
;               Version 0.9, 30/7/98
;                 Added KEYWORD SPINS_PER_SAMPLE.
;               Version 1.0, 7/9/98
;                 Added terms to HISTp structure.
;                 Structure vs 5.
;               Version 1.1, 5/2/99
;                 Added energy_efficiency to IPS structure.
;                 Added separate structure version nos for instruments.
;                 Structure vs 6.
;               Version 1.2, 26/6/00
;                 Added LUT_TIMES keyword.
;                 Structure vs 7.
;               Version 1.3, 3/9/00
;                 Added RAP version no.
;               Version 1.4, 18/07/01
;                 Added VERSION keyword.
;               Version 1.5, 23/07/01
;                 Changed version to string (previously float).
;               Version 1.6, 29/01/02
;                 Corrected bug in definition of INTEGRAL_ENERGY.
;                 Now 32 sectors by no of spins by integral energy.
;               Version 1.7, 03/10/03
;                 Allowed MF tag to be added to named structure.
;                 Added MF keyword.
;                 Added PA keyword.
;                 Changed version to 9_2.
;               Version 1.8, 28/11/03
;                 Changed how structure name set.
;               Version 1.9, 15/12/03
;                 Changed RAPID structure name to VRAP3 to allow for LUT changes in lut_map.
;               Version 2.0, 08/01/04
;                 Used ies sw version.
;
; Version     : Version 2.0, 08/01/04
;-
;**********************************************************

FUNCTION ies_header_structure, datatype, no_of_points, no_of_detectors, no_of_energies, no_of_sectors, no_of_luts, $
         TIME_START=time_start, TIME_STOP=time_stop, DETECTORS=detectors, $
         TITLE=title, XTITLE=xtitle, YTITLE=ytitle, ZTITLE=ztitle, PTITLE=ptitle, $
         EXPERIMENT=experiment, MODE=mode, ENERGY_EFFICIENCY=energy_efficiency, $
         ENERGY_CALIBRATION=energy_calibration, COUNT_CALIBRATION=count_calibration, LUT_MAP=lut_map, LUT_TIMES=lut_times, $
         SPINS_PER_SAMPLE=spins_per_sample, VERSION=version, MF=mf, PA=pa

  ; set structure name variable

  structure_name = ''

  ; deal with experiment and mode keywords

  IF KEYWORD_SET(experiment) THEN structure_name = 'header_' + experiment

  IF KEYWORD_SET(mode) THEN BEGIN

    structure_name = structure_name + '_' + mode

    IF mode EQ 'SPIN_AVERAGED' THEN BEGIN
      no_of_sectors = 1
      spins_per_sample = 1
    ENDIF ELSE IF mode EQ 'INTEGRAL_ENERGY' THEN BEGIN
      no_of_energies = 1
      spins_per_sample = 1./no_of_sectors
    ENDIF ELSE IF mode EQ 'FULL_RESOLUTION' THEN BEGIN
      no_of_points  = no_of_points*no_of_sectors
      no_of_sectors = 1
      spins_per_sample = 1./no_of_sectors
    ENDIF ELSE IF mode EQ 'LOW_RESOLUTION' THEN BEGIN
      no_of_points  = no_of_points/16
      no_of_sectors = 1
      spins_per_sample = 16
    ENDIF ELSE IF mode EQ 'CLEAN_SURVEY' THEN BEGIN
      no_of_points   = no_of_points/16
      spins_per_sample = 16
      no_of_sectors  = 16
      no_of_energies = 10
    ENDIF ELSE IF mode EQ 'SURVEY' THEN BEGIN
      no_of_points   = no_of_points/16
      spins_per_sample = 16
      no_of_sectors  = 16
    ENDIF ELSE IF mode EQ 'MAJOR_SPIN_AVERAGED' THEN BEGIN
      no_of_points  = no_of_points/16
      spins_per_sample = 16
    ENDIF ELSE IF STRMID(mode,0,5) EQ 'SPIN_' THEN BEGIN

      ; get no of sectors

      no_of_sectors = FIX(STRMID(mode,5,2))

      ; check for sample averaging

      md = STRMID(mode,5,20)

      pos = STRPOS(md,'Z')

      spins_per_sample = FIX(STRMID(md,pos+1,2))

      IF pos GE 0 THEN no_of_points = no_of_points/spins_per_sample

    ENDIF ELSE IF STRMID(mode,0,6) EQ 'CLEAN_' THEN BEGIN

      no_of_energies = 10

      ; get no of sectors

      no_of_sectors = FIX(STRMID(mode,6,2))

      ; check for sample averaging

      md = STRMID(mode,6,20)

      pos = STRPOS(md,'Z')

      spins_per_sample = FIX(STRMID(md,pos+1,2))

      IF pos GE 0 THEN no_of_points = no_of_points/spins_per_sample

    ENDIF ELSE MESSAGE, 'MODE NOT RECOGNIZED',/TRACEBACK

  ENDIF

  IF structure_name NE '' AND KEYWORD_SET(mf) THEN structure_name = structure_name + '_mf'

  IF structure_name NE '' AND KEYWORD_SET(pa) THEN structure_name = structure_name + '_pa'

  ; check if keywords set

  IF NOT KEYWORD_SET(time_start) THEN time_start = 0.0D0
  IF NOT KEYWORD_SET(time_stop ) THEN time_stop  = 0.0D0
  IF NOT KEYWORD_SET(detectors)  THEN detectors  = 1+INDGEN(no_of_detectors)
  IF NOT KEYWORD_SET( title)     THEN title  = ''
  IF NOT KEYWORD_SET(xtitle)     THEN xtitle = ''
  IF NOT KEYWORD_SET(ytitle)     THEN ytitle = ''
  IF NOT KEYWORD_SET(ztitle)     THEN ztitle = ''
  IF NOT KEYWORD_SET(ptitle)     THEN ptitle = STRTRIM(detectors,2)

  IF KEYWORD_SET(energy_efficiency) THEN str_energy_efficiency  = 'energy_efficiency:energy_efficiency,' ELSE str_energy_efficiency = ''

  IF KEYWORD_SET(energy_calibration) THEN str_energy_calibration = 'energy_cal:energy_calibration,' ELSE str_energy_calibration = ''

  IF KEYWORD_SET(count_calibration) THEN str_count_calibration = 'count_cal:count_calibration,' ELSE str_count_calibration = ''

  IF KEYWORD_SET(lut_map) THEN str_lut_map = 'lut_map:lut_map,' ELSE str_lut_map = ''

  IF KEYWORD_SET(lut_times) THEN str_lut_times = 'lut_times:lut_times,' ELSE str_lut_times = ''

  IF KEYWORD_SET(mf) THEN str_mf = 'mf:mf,' ELSE str_mf = ''

  IF KEYWORD_SET(pa) THEN str_pa = 'pa:pa,' ELSE str_pa = ''

  IF structure_name NE '' THEN BEGIN
    IF STRPOS(structure_name, 'RAPID') GE 0 THEN structure_name = structure_name + '_VRAP3,' ELSE $
    IF STRPOS(structure_name, 'IES') GE 0 THEN structure_name = structure_name + '_VIES7,' ELSE $
    IF STRPOS(structure_name, 'IPS') GE 0 THEN structure_name = structure_name + '_VIPS7,' ELSE $
    IF STRPOS(structure_name, 'HISTe') GE 0 THEN structure_name = structure_name + '_VHISTE7,' ELSE $
    IF STRPOS(structure_name, 'HISTp') GE 0 THEN structure_name = structure_name + '_VHISTP7,' ELSE $
    structure_name = structure_name + '_V8,'
  ENDIF

  IF NOT KEYWORD_SET(version) THEN version = ies_sw_version()

  structure = 0

  ; NB structure name contains a ','

  IF EXECUTE ( 'structure={' + structure_name + $
            'version:STRING(version),'+ $
            'datatype:datatype,'+ $
            'npoints:LONG(no_of_points),'+ $
            'ndetectors:LONG(no_of_detectors),'+ $
            'nbands:LONG(no_of_energies),'+ $
            'nsectors:LONG(no_of_sectors),'+ $
            'nluts:LONG(no_of_luts),' + $
            'detectors:LONG(detectors),'+ $
            'tstart:DOUBLE(time_start),'+ $
            'tstop:DOUBLE(time_stop),'+ $
            'ch_positions:FLTARR (2, no_of_energies, no_of_detectors > 1, no_of_luts),'+ $
             str_count_calibration +$
             str_energy_calibration +$
             str_energy_efficiency +$
             str_lut_map +$
             str_lut_times +$
             str_mf +$
             str_pa +$
            'cal:0,'+$
            'rat:0,'+$
            'div:0,'+$
            'clean:0,'+$
            'subtract:0,'+$
            'title:STRING(title),'+ $
            'xtitle:STRING(xtitle),'+ $
            'ytitle:STRING(ytitle),'+ $
            'ztitle:STRING(ztitle),'+ $
            'ptitle:STRING(ptitle) }' ) THEN RETURN, structure

  MESSAGE, 'Execute did not work', /TRACEBACK

END