;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_INTERP
;
; Purpose     :	Applies a pedestal shift and interpolation.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine interpolates the data to get samples
;               at specified bin positions. A pedestal shift for each column may optionally be applied.
;               Assumes data has raw data format i.e has continuous coverage of energy and last
;               channel integrates everything above high energy threshold of IES.
;               Assumes this even when last channel is not integral channel for want of good
;               alternative.
;               Assumes input data has not been divided by channel size.
;               A linear interpolation of log(cumulative counts) against log(energy) is used.
;               The interpolation may be performed either on the original distribution or the
;               cumulative distribution. The latter may have unphysical discontinuities in the
;               resulting non-cumulative distribution because of the log interpolation.
;               However, interpolating the cumulative distribution does preserve the total counts
;               whereas interpolating the original distribution does not.
;               In the interpolation consider the original data values to be positioned at the
;               channel start boundaries. Then interpolate to get the values at the new channel
;               start boundaries incorporating the pedestal shift if required. This makes sense
;               only in the cumulative mode. Pedestals gives the bin positions of the pedestals
;               where the counts are interpreted as being on bin centres.
;
; Use         : < ies_interp, y, lut_list, ch_positions, nchannels, yp, minimum, pedestals, SHFTCH=shftch  >
;
; Inputs      : y            : FLTARR(nchannels+...,nsamples) counts per channel at each
;                              sample point for each channel
;                              NB passed by reference.
;                              Can be extra channels present ie summed channel.
;               lut_list     : STRUCTURE ARRAY containing LUT run info.
;               ch_positions : FLTARR(2,nchannels+...,1,nluts) channel bin positions for each LUT.
;               nchannels    : LONG no. of y channels to be used
;               yp           : FLTARR(ysize) start channel or bin boundaries for each new y sample
;                              NB last bin is integral above start so does not need boundary.
;               minimum      : FLOAT minimum non-zero value in data
;               pedestals    : FLTARR(nsamples) bin positions of pedestals for each sample.
;
; Opt. Inputs : None.
;
; Outputs     : y            : FLTARR(ysize,nsamples) interpolated image
;
; Opt. Outputs:	None.
;
; Keywords    : /SHFTCH      : indicates whether to shift data to match pedestal positions.
;               /CUMULATIVE  : Flag to indicate that should interpolate cumulative distribution
;               FLAGS        : LONARR[nsamples] flags for extrapolation warning.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 10/10/96
;                 Adapted from ies rebin y.
;
; Modified    :	Version 0.1, 5/2/97
;                 Modified treatment of minimum values.
;           	Version 0.2, 4/3/97
;                 Adjusted so that can cope with negative pedestal positions.
;           	Version 0.3, 2/4/97
;                 Adjusted so that yp is now bin boundaries rather than bin centres.
;           	Version 0.4, 27/2/98
;                 Uses bin boundaries rather than bin centres for everything.
;           	Version 0.5, 3/2/99
;                 Removed IES specific stuff.
;                 Changed so that only input start boundary of each new bin in yp.
;                 Added CUMULATIVE switch to ies_interp.
;                 Always uses cumulative at present.
;               Version 0.6, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.7, 11/10/00
;                 Added more comments.
;                 NB changed ies get samples so uses bin centres.
;               Version 0.8, 15/08/01
;                 Added extrapolation FLAG.
;               Version 0.9, 02/12/02
;                 Changed so that allows extrapolation outside of valid data range.
;                 Used N_ELEMENTS check rather than KEYWORD_SET check on flag to ensure flag set when
;                 only one element.
;               Version 1.0, 24/10/03
;                 Changed to lut_list argument.
;
; Version     :	Version 1.0, 24/10/03
;-
;**********************************************************

PRO ies_interp, y, lut_list, ch_positions, nchannels, yp, minimum, pedestals, SHFTCH=shftch, CUMULATIVE=cumulative, FLAGS=flags

 ; get no. of samples

 no_of_samples = lut_list[N_ELEMENTS(lut_list)-1].en+1

 ; get new no. of y samples
 ; NB last channel is integral channel so does not need bin boundary

 ysize = N_ELEMENTS(yp)

 ; if no pedestal shift then take logs, remove any zeroes

 IF NOT KEYWORD_SET(shftch) THEN typ = ALOG(yp(0:ysize-1) > 0.1)

 IF KEYWORD_SET(cumulative) THEN BEGIN

   ; get channel start boundaries for all luts for channels 0:14

   channel_starts = REFORM(ch_positions(0,0:nchannels-1,0,*))

 ENDIF ELSE BEGIN

   ; not coded yet -- see old version

 ENDELSE

 ; get logs of channel boundaries, ensure any invalid LUT 0's do not cause a problem.

 log_starts = ALOG(channel_starts > 0.1)

 ; form blank array

 yy = FLTARR(ysize,no_of_samples)

 ; get channel data

 yc = y(0:nchannels-1,*)

 ; get cumulative counts from channel 14

 IF KEYWORD_SET(cumulative) THEN $
   FOR k = 1, nchannels-1 DO yc(nchannels-1-k,*) = yc(nchannels-1-k,*) + yc(nchannels-k,*)

 ; take log (may have fractional counts per spin)
 ; summed channel should not be present
 ; set any zero samples to 0.1*minimum

 yc = ALOG(yc > (0.1*minimum))

 ; loop through runs

 FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

   ; get LUT for sample

   lut = lut_list[l].lut

   xtemp = log_starts(*,lut)

   ; loop through samples

   FOR k = lut_list[l].st, lut_list[l].en DO BEGIN

     ; get copy of counts for sample

     ytemp = yc(*,k)

     ; check if pedestal shift required
     ; get LOG bin positions for each sample corrected for pedestal shift
     ; remove any zeroes

     IF KEYWORD_SET(shftch) THEN typ = ALOG((yp(0:ysize-1) + pedestals(k)) > 0.1)

     ; set warning flag if new bins outside of valid data range
     ; NB xtemp(nchannels-1) is the start boundary of the last channel

     IF typ[0] LT xtemp[0] THEN BEGIN
       IF N_ELEMENTS(flags) NE 0 THEN flags[k] = flags[k] AND '00000100'XL
     ENDIF

     IF typ[ysize-1] GT xtemp[nchannels-1] THEN BEGIN
       IF N_ELEMENTS(flags) NE 0 THEN flags[k] = flags[k] AND '00000200'XL
     ENDIF

     ; do not extrapolate outside of data

     ; typ = xtemp(nchannels-1) < typ > xtemp(0)

     ; interpolate data using linear form in LOG(e) vs LOG(x)

     IF N_ELEMENTS(xtemp) GT 1 THEN $
       yy(*,k) = EXP ( INTERPOL ( ytemp, xtemp, typ ) ) $
     ELSE $
       yy(*,k) = EXP(ytemp(0))

   ENDFOR

 ENDFOR

 ; get counts rather than cumulative counts (except last channel)

 IF KEYWORD_SET(cumulative) THEN FOR k = 0, ysize-2 DO yy(k,*) = yy(k,*) - yy(k+1,*)

 y = yy

 ; set any resampled value less than 0.5*minimum to zero

 list = WHERE ( y LT (0.5*minimum), count)

 IF count GT 0 THEN y(list)= 0

END