;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_MASK
;
; Purpose     :	Generates a mask in POLAR/AZIMUTH coords. 
;
; Explanation :	Calculates polar/azimuth pixels corresponding to a given solid angle about a given vector.  
;               Produces a mask of polar/azimuth pixels corresponding to solid angle.
;               Takes into account finite size of sector bins at high polar angles.
;               Pixel masked if any part of solid angle appears in pixel.            
;
; Use         : < ies_mask, az, po, delta, mask, val
;
; Inputs      : az     : FLOAT azimuth angle of vector in S/C coords 
;                           0->2*pi with 0 along sun direction           
;               po     : FLOAT polar angle of vector in S/C coords
;                           0->pi with 0 along spin axis            
;               delta  : FLOAT half cone angle in radians
;               mask   : ARRAY(az_dim,po_dim) polar/azimuth array
;               val    : value to fill mask with where intersects solid angle
;
; Opt. Inputs : None.
;
; Outputs     : mask   : ARRAY(az_dim,po_dim) contains val if any part of solid angle in pixel
;                          Assumes row 0 is spin anti-parallel
;
; Opt. Outputs:	None.
;
; Keywords    : FACTOR : factor to multiply azimuthal extent
;
; Calls       :	None.
;                
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/10/98
;
; Modified    :	Version 0.1, 1/12/98
;                  Modified azimuthal correction.
;
; Version     :	Version 0.1, 1/12/98
;-
;**********************************************************

PRO ies_mask, az, po, delta, mask, val, FACTOR=factor

  ; set calculational error 

  err = 1e-3

  ; get dimensions of mask array

  az_dim = (SIZE(mask))(1)
  po_dim = (SIZE(mask))(2)

  ; work out polar angles corresponding to pixel boundaries (0->pi)

  po_s = FINDGEN(po_dim+1)/po_dim*!PI

  ; get polar boundaries which cross circle defined by solid angle around vector 
  ; if touching boundary then do not include adjacent row

  po_list = WHERE(po_s GT (ABS(po-delta)+err) AND po_s LT ((!PI-ABS(po+delta-!PI))-err), po_count)

  ; process polar angles which cut circle

  IF po_count GT 0 THEN BEGIN

    ; work out azimuthal extent of segment of circle -- half azimuthal size

    cos_po_s = COS(po_s[po_list]) ; SIN(90-po_s) 

    ; insist >0 to account for calculation inaccuracies

    num = SQRT((1 - COS(po)^2 - cos_po_s^2 - COS(delta)^2 + 2*COS(po)*COS(delta)*cos_po_s)>0)
 
    den = COS(delta) - COS(po)*cos_po_s

    az_size = ABS(ATAN(num,den)) ; 0 -> !PI

    ; increase azimuthal size due to finite sector size i.e 32 sectors

    theta = ATAN(!PI/32/SIN(po))

    beta  = ASIN(!PI/32*COS(theta))

    az_size = az_size + theta - beta

    ; multiply azimuthal size by fudge factor

    IF KEYWORD_SET(factor) THEN az_size = factor*az_size

    ; get start and end columns corresponding to azimuthal extent
    ; if on column boundary then do not include adjacent column

    st_az = FIX((az - az_size)/2.0/!PI*az_dim + az_dim + err)
    en_az = FIX((az + az_size)/2.0/!PI*az_dim + az_dim + 1 - err)

    ; fill in mask for each row of pixels touching the boundaries

    ; do first and last rows
    ; (NB first and last boundaries cannot be included because can only touch boundary)

    mask[(st_az[0]+INDGEN(en_az[0]-st_az[0])+az_dim) MOD az_dim,po_dim-1-(po_list[0]-1)] = val

    mask[(st_az[po_count-1]+INDGEN(en_az[po_count-1]-st_az[po_count-1])+az_dim) MOD az_dim,po_dim-1-po_list[po_count-1]] = val

    ; check if more than on boundary cut

    IF po_count GT 1 THEN BEGIN

      ; rest of rows are have upper and lower azimuthal extents 
      ; use max azimuthal extent for each row

      list = WHERE(az_size[0:po_count-2] LT az_size[1:*],count)

      IF count GT 0 THEN BEGIN
        st_az[list] = st_az[list+1]
        en_az[list] = en_az[list+1]
      ENDIF

      ; loop through rest of rows

      FOR p = 0, po_count-2 DO mask[(st_az[p]+INDGEN(en_az[p]-st_az[p])) MOD az_dim,po_dim-1-po_list[p]] = val

    ENDIF

  ENDIF ELSE IF po_count EQ 0 AND (po-delta) GE err AND (po+delta) LE (!PI-err) THEN BEGIN

    ; no boundary cuts so solid angle lies inside a polar bin or is nearly concentric with spin axis 
    ; nearly concentric case is handled below, small delta case is handled here

    ; work out azimuthal extent of segment of circle -- half azimuthal size

    cos_po_s = COS(po) ; SIN(90-po_s) 

    ; insist >0 to account for calculation inaccuracies

    num = SQRT((1 - COS(po)^2 - cos_po_s^2 - COS(delta)^2 + 2*COS(po)*COS(delta)*cos_po_s)>0)
 
    den = COS(delta) - COS(po)*cos_po_s

    az_size = ABS(ATAN(num,den)) ; 0 -> !PI

    ; increase azimuthal size due to finite sector size i.e 32 sectors

    az_size = az_size + ASIN(!PI/32/SIN(po)) - ASIN(!PI/32)

    ; multiply azimuthal size by fudge factor

    IF KEYWORD_SET(factor) THEN az_size = factor*az_size

    ; get start and end columns corresponding to azimuthal extent

    st_az = FIX((az - az_size)/2.0/!PI*az_dim + az_dim + err)
    en_az = FIX((az + az_size)/2.0/!PI*az_dim + az_dim + 1 - err)

    ; fill in mask for each row of pixels touching the boundaries

    ; do first row
    ; (NB first boundary cannot be included because can only touch boundary)

    mask[(st_az[0]+INDGEN(en_az[0]-st_az[0])+az_dim) MOD az_dim,po_dim-1-FIX(po/!PI*po_dim)] = val

  ENDIF

  ; process polar angles which lie inside north circle
  ; if touching boundary then do not include adjacent row

  po_list = WHERE(po_s LT delta-po-err, po_count)

  ; process polar angles which lie inside north circle

  IF po_count GT 0 THEN mask[*,po_dim-1-po_list] = val

  ; process polar angles which lie inside south circle
  ; if touching boundary then do not include adjacent row

  po_list = WHERE(po_s GT (2*!PI-delta-po)+err, po_count)

  ; process polar angles which lie inside south circle

  IF po_count GT 0 THEN mask[*,po_dim-1-(po_list-1)] = val

END
