;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_MK_IMAGE
;
; Purpose     :	Produces rebinned 2-D image of counts for y vs x.
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine maps the input image into a new image
;               size in both x and y axes optionally taking into account the different column to
;               column y mapping i.e the different y channel sizes.
;               It also allows an option to shift each column to match pedestal positions.
;
; Use         : < ies_mk_image, y, x, luts, ch_positions, nchannels, ylo, yhi, xsize, ysize, $
;                               SHOWCH=showch, SHFTCH=shftch, LOG=log >
;
; Inputs      : y            : FLTARR(nchannels,npoints) data values at each sample point for each channel
;               x            : DBLARR(npoints) times for each sample in TAI format
;               luts         : BYTARR(npoints) LUT used for each sample point
;               ch_positions : FLTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               nchannels    : LONG no. of y channels
;               ylo          : LONG low y axis value
;               yhi          : LONG high y axis value
;               xsize        : LONG x dimension of output array
;               ysize        : LONG y dimension of output array
;
; Opt. Inputs : None.
;
; Outputs     : y            : FLTARR(ysize,xsize) image
;
; Opt. Outputs:	None.
;
; Keywords    :/SHOWCH   : indicates whether to show real channel sizes or not
;              /SHFTCH   : indicates whether to shift data to match pedestal positions.
;                          Only available if showch is also set.
;              /LOG      : indicates logarithmic y axis display.
;               ENDTIME     : DBLARR(npoints) end times for each sample in TAI format if present
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 2/6/95
;
; Modified    :	Version 0.1, MKC, 19/9/96
;                 Improved pedestal shift algorithm.
;                 Added DIV keyword.
;                 Added INTERP keyword.
;                 Shift pedestal to zero.
;                 Added ies rebin y.
;               Version 0.2, 11/11/96
;                 Removed DIV, new argument list for rebin_y
;               Version 0.3, 1/4/97
;                 Added LOG keyword.
;                 Changed yp to channel boundaries rather than channel centres.
;               Version 0.4, 28/4/97
;                 Corrected routine for case where more than 32767 data points.
;               Version 0.5, 2/3/98
;                 Removed interpolation option. Removed minimum argument.
;               Version 0.6, 14/2/01
;                 Added ENDTIME keyword.
;
; Version     :	Version 0.6, 14/2/01
;-
;**********************************************************

PRO ies_mk_image, y, x, luts, ch_positions, nchannels, ylo, yhi, xsize, ysize, $
                  SHOWCH=showch, SHFTCH=shftch, LOG=log, ENDTIME=endtime

  ; fiddle x range of data to fit into plot box

  ; calculate x positions of input data in device pixels

  device_x = LONG ( ((x-!X.CRANGE[0]) * xsize)/ (!X.CRANGE[1]-!X.CRANGE[0]) )

  ; create array containing for each device pixel the corresponding index into the data
  ; NB the plot size is greater than the data region
  ; device_x is the device pixel corresponding to each data pixel

  ; fudge for now
  ; endtime was previously not set even though existed as a tag

  IF KEYWORD_SET(endtime) THEN BEGIN

    IF endtime[0] EQ endtime[N_ELEMENTS(endtime)-1] THEN endtime = 0

  ENDIF

  IF KEYWORD_SET(endtime) THEN BEGIN

    ; find device pixel corresponding to endtime

    end_device_x = LONG ( ((endtime-!X.CRANGE[0]) * xsize)/ (!X.CRANGE[1]-!X.CRANGE[0]) )

    ; can be outside of device pixels

    device_x = device_x < (xsize-1)

    end_device_x = end_device_x < (xsize-1)

    ; calculate first and last device pixels and number of pixels containing data

    device_xstart = device_x[0] > 0
    device_xstop  = end_device_x(N_ELEMENTS(end_device_x)-1) < (xsize-1)
    device_xsize  = device_xstop - device_xstart + 1

    ; xp is the data index for device pixels from device_xstart to device_xstop

    ; set device pixel to default value

    xp = REPLICATE(N_ELEMENTS(x),device_xsize)

    ; set device pixel to last sample

    FOR k = 0L, N_ELEMENTS(x)-1 DO xp[device_x[k]-device_xstart:end_device_x[k]-device_xstart] = k

    ; add dummy y sample

    y = [[[y]],[FLTARR((SIZE(y))[1])]]

  ENDIF ELSE BEGIN

    ; calculate first and last device pixels and number of pixels containing data

    device_xstart = device_x[0] > 0
    device_xstop  = device_x(N_ELEMENTS(device_x)-1) < (xsize-1)
    device_xsize  = device_xstop - device_xstart + 1

    ; find for each device pixel the first data pixel corresponding to it
    ; if one exists then the this data is used for the device pixel
    ; if not then the last data available is used, this will correspond to the
    ; data pixel before the next first data pixel.

    ; xp is the data index for device pixels from device_xstart to device_xstop

    xp = LONARR(device_xsize)
    data_pos = 0L
    device_pos = device_x[data_pos]
    max_pos  = N_ELEMENTS(device_x)-1

    FOR xpi = device_xstart, device_xstop DO BEGIN

      ; set device pixel to first data value encountered
      ; set subsequent pixels to last data value encountered
      ; NB This ensures that display sees any end of data flags and
      ;    if all data falls in one pixel then sees the data rather
      ;    than the end of data flag and allows the last pixel in the display
      ;    to be seen.

      IF device_pos GT xpi THEN xp(xpi-device_xstart) = data_pos-1 ELSE BEGIN
        xp(xpi-device_xstart) = data_pos
        IF data_pos LT max_pos THEN BEGIN
          data_pos = data_pos + 1
          device_pos = device_x(data_pos)
        ENDIF
        WHILE device_pos LE xpi AND data_pos LT max_pos DO BEGIN
          data_pos = data_pos + 1
          device_pos = device_x(data_pos)
        ENDWHILE
      ENDELSE

    ENDFOR

  ENDELSE

      ; do y axis

      ; yp is an array that gives the channel or bin positions corresponding to each device pixel boundary
      ; form linear or logarithmic map of (ylo,yhi) into ysize pixels in plot box

      IF KEYWORD_SET(log) THEN BEGIN

        ; design so that device pixels divided into LOG10(yhi)-LOG10(ylo) channels

        yp = 10.0^(ALOG10(ylo) + (ALOG10(yhi)-ALOG10(ylo))*FINDGEN ( FIX(ysize+1) ) / FLOAT(ysize))

      ENDIF ELSE BEGIN

        ; design so that device pixels divided into yhi-ylo channels
        ; i.e if ylo=0, yhi=nchannels, ysize=nchannels then yp=FINDGEN(nchannels+1)

        yp = ylo + (yhi-ylo)*FINDGEN ( FIX(ysize+1) ) / FLOAT(ysize)

      ENDELSE

      ; check which is quickest : to apply y fix to data samples or to device pixels

      IF N_ELEMENTS(x) GE N_ELEMENTS(xp) THEN BEGIN

        ; use device pixels
        ; get samples for each device pixel

        ; get y values for each device pixel

        y = y[*,xp]

        ; rebin y
        ; get lut for each device pixel

        ies_rebin_y, y, luts[xp], ch_positions, nchannels, yp, FLTARR(N_ELEMENTS(xp)), $
                     SHOWCH=showch, SHFTCH=shftch

      ENDIF ELSE BEGIN

        ; rebin y

        ies_rebin_y, y, luts, ch_positions, nchannels, yp, FLTARR(N_ELEMENTS(x)), $
                     SHOWCH=showch, SHFTCH=shftch

        ; get y values for each device pixel

        y = y[*,xp]

      ENDELSE

      ; rearrange to get points along x axis

      y = TRANSPOSE(y)

      ; do ends of spectrum

      ; add blank array for unused pixels before data

      rsize = device_xstart

      IF rsize GT 0 THEN y = [ FLTARR( rsize, FIX(ysize) ), y]

      ; add blank array for unused pixels before data

      rsize = xsize - 1 - device_xstop

      IF  rsize GT 0 THEN y = [ y, FLTARR( rsize, FIX(ysize) )]

END