;+
; Project     :	POLAR - CEPPAD
;
; Name        :	IES_NCOLORS
;
; Purpose     :	Returns no. of colors for display.
;
; Explanation :	Returns no. of colours for screen display, deals with 24 bit displays.
;               SW has two requirements :
;                 SURVEY plot generation using PS or Z buffer display where want to use all
;                        256 colours and get results independent of shared colour map.
;                 IES_PLOT tool where want to be able to alter colour table and preserve results in PS and
;                        PNG/GIF (Z buffer) plots ie use shared colour map.
;               Routine is called with SET when in X windows mode or Z buffer mode so proper current no. of colours is
;               stored. Subsequent calls return set no. of colours.
;               No. of colours in shared colour map is set up after call to LOADCT or WINDOW.
;               Can set up private colour map with 256 colours but this must be done when first
;               in IDL so not flexible enough. Need to reload colour table in PS or PNG/GIF mode.
;               Although 256 colours are used in PS and PNG/GIF, colour table still corresponds to
;               shared colour map.
;               With true colour system may have !D.N_COLORS > 256 so need to limit to 256.
;
;               Shared color table allocates a range of colors from 256 bin color table to IDL ie colors
;               from 56 to 228. Then have !d.n_colors=173. IDL sets up a translation table which maps 0:172
;               into 56:228. This is used for any display of color in windows ie color 0 gets mapped to
;               color 56 in color table.
;               When using all 256 colors on Z device need to set color table after SET_PLOT.
;               When using colors available in shared color table need to restrict use of colors to
;               number available. This is done using ies_ncolors().
;               Still have problem that  default color for foreground and background will be 0 and 255.
;               The latter will not be defined in the color table. So have to explicitly set !P.COLOR
;               when change device.
;               An alternative would be to modify the translation table to map 0:255 to 56:228.
;               Would then need to explicitly apply map when running with wysiwyg and PNG output.
;               Sometimes will not want to the foreground color in the plot. In this case
;               the foreground keyword should be set.
;
; Use         : < ncolors=ies_ncolors() >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : ncolors : INT no. of colors.
;
; Opt. Outputs:	None.
;
; Keywords    : /SET        = sets stored no. of colors.
;               PRIVATE     = INT private color table index
;               SHARED      = INT shared color table index
;               FOREGROUND  = FLAG for no foreground color reported in ncolors.
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 9/10/98
;
; Modified    :	Version 0.1, 13/12/99
;                 Added switch off of decomposed colors.
;               Version 0.2, 10/12/00
;                 Added PNG output
;               Version 0.3, 13/03/01
;                 Check that WIN or X device before trying to set decomposed=0
;                 If device Z or PS then does not allow decomposed=0.
;               Version 0.4, 09/07/02
;                 Removed wysiwyg keyword.
;                 Added CT and SHARED keywords.
;                 Added /SILENT to LOADCT.
;               Version 0.5, 03/09/02
;                 Ran into problems again with setting decomposed = 0 in batch mode.
;                 Cannot use in Z or PS or from remote session (needs an X window).
;                 Previously called ies_ncolors from Z mode ie dont need to set decomposed=0
;                 from remote session because not going to use the host 24 bit color tables,
;                 going to do everything in z or ps mode.
;               Version 0.6, 15/05/03
;                 Added FOREGROUND keywords.
;
; Version     :	Version 0.6, 15/05/03
;-
;**********************************************************

FUNCTION ies_ncolors, SET=set, PRIVATE=private, SHARED=shared, FOREGROUND=foreground

  COMMON color_store, ncolors

  IF KEYWORD_SET(set) THEN BEGIN

    ; set up ncolors

    IF KEYWORD_SET(private) THEN BEGIN

      ; load 256 bin color table independent of shared memory map

      devname = !D.NAME

      SET_PLOT, 'Z'

      LOADCT, private, /SILENT

      ncolors = !D.N_COLORS < 256

      SET_PLOT, devname

    ENDIF ELSE IF KEYWORD_SET(shared) THEN BEGIN

      ; Until WINDOW or LOADCT called !d.n_colors is not set up properly

      LOADCT, shared, /SILENT

      ncolors = !D.N_COLORS < 256

      ; turn off 24 bit colors for X or WIN
      ; if running with shared color tables then must be calling from win or X

      IF !D.NAME EQ 'WIN' OR !D.NAME EQ 'X' THEN DEVICE, DECOMPOSED=0

    ENDIF ELSE BEGIN

      ncolors = !D.N_COLORS < 256

      ; turn off 24 bit colors for X or WIN

      IF !D.NAME EQ 'WIN' OR !D.NAME EQ 'X' THEN DEVICE, DECOMPOSED=0

    ENDELSE

  ENDIF

  ; remove background and foreground colors if not available for plot

  IF KEYWORD_SET(foreground) THEN ncolors = ncolors - 1

  RETURN, ncolors

END
