;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_PEDESTAL_SAMPLES
;
; Purpose     : Gets pedestal sample info for given detector and LUT.
;
; Explanation :	Gets pedestal sample info for given detector and LUT.
;
; Use         : < ies_pedestal_samples, npeds, nshifts, offset, detector, lut_no, temperature_index, ch_positions, $
;                    samples, max_sample_lists, pedestal_posn, pedestal_width >
;
; Inputs      : npeds             : INT no. of pedestal channels
;               nshifts           : INT no. of pedestal shifts (each shift is 0.1  bins)
;               offset            : INT pedestal shift offset (start shift is +offset/10 bins)
;               detector          : INT detector no. 1:9
;               lut_no            : INT LUT no 0:7
;               temperature_index : INT index into temperature
;               ch_positions      : FLTARR(2, no_of_energies) LUT definitions for detector and LUT
;
; Opt. Inputs : None.
;
; Outputs     : samples          : FLTARR[npeds,nshifts] is an array containing shifted pedestal calibration
;                                  data sampled as for the real data. Only the first npeds energy channels are kept.
;               max_sample_list  : FLTARR(1+nshifts, npeds) indices of shifts with peak in channel 1+channel.
;                                  Element 0 contains no of indices.max_samples contains the index of the peak
;                                  for each shift. This is used to restrict the search for max cross-correlation
;                                  to samples with the same peak position as the data
;               pedestal_posn    : FLOAT result of fit to pedestal.
;               pedestal_width   : FLOAT result of fit to pedestal.
;               pedestal_centre  : FLOAT position of center of pedestal channels.
;               channel_centres  : FLTARR(npeds) position of centers of pedestal channels.
;
; Opt. Outputs:	None.
;
; Keywords    : None.
;
; Written     :	Version 0.0, 11/12/03
;                 Separated from cep pedestals.
;
; Modified    :
;
; Version     :	Version 0.0, 11/12/03
;-
;**********************************************************

PRO ies_pedestal_samples, npeds, nshifts, offset, detector, lut_no, temperature_index, ch_positions, samples, max_sample_lists, $
                          pedestal_posn, pedestal_width, pedestal_centre, channel_centres

  ; get pedestal calibration data

  pdata = ies_get_pedestal_histogram( lut_no, detector, temperature_index)

  ; get pedestal position and FWHM

  ies_fit_pedestal, pdata, pedestal_posn, pedestal_width

  ; samples is an array containing shifted pedestal calibration data sampled as for the real data
  ; only the first npeds samples are kept

  samples = ies_get_samples ( pdata, npeds, ch_positions, nshifts, offset )

  ; set up max_sample_lists array
  ; list of shifts for each pedestal channel maxima position

  max_sample_lists = ies_get_sample_lists(samples)

  ; get pedestal centre

  pedestal_centre = ch_positions[0,npeds/2]

  ; get channel centres

  channel_centres = TOTAL(ch_positions[*,0:npeds-1],1)/2.0

END