;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_PITCH_ANGLE
;
; Purpose     :	Calculates pitch angle for each detector/sector : 0->PI.
;
; Explanation :	Calculates pitch angle for each valid sector/detector.
;               Uses -1.0 for non-valid detectors.
;               NB provided there are an even no. of sectors then
;               pitch angles of one half are PI - pitch angles of other half.
;
; Use         : < ies_pitch_angle, header, data, array, /PA >
;
; Inputs      : header        : STRUCTURE info structure for data.
;               data          : STRARR for each sample point.
;
; Opt. Inputs : None.
;
; Outputs     : array         : FLTARR(nsectors, ndetectors, npoints)
;
; Opt. Outputs:	None.
;
; Keywords    : PA      : Calculate pitch angles.
;               EARTH   : Apply earth mask.
;               SUN     : Apply sun mask.
;               MF      : FLTARR(3,npoints) magnetic field azimuth, polar and magnitude for each sample.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 27/4/99
;
; Modified    :	Version 0.1, 19/8/00
;                 Explicitly allowed for LUT=-1
;               Version 0.2, 20/07/01
;                 Used ies_runs.
;               Version 0.3 12/07/02
;                 Used ies_tag_exists.
;               Version 0.4 07/02/03
;                 Changed RAPID processing to include IIMS.
;                 Set invalid pitch angle to -1.
;                 Used duplicated pitch angle trick in calculation.
;               Version 0.5, 13/10/03
;                 Allow magnetic field direction for each sample to be passed by KEYWORD rather than through
;                 the data.mf tag.
;                 Fixed calculation for odd no of scetors (including one).
;               Version 0.6, 20/10/03
;                 Moved increase in effective size of sun and earth (factor 2) to ies extdata.
;                 Nb size also used directly in ies plot img.
;                 Separated sun and earth masks into routien ies generate masks
;               Version 0.7, 24/10/03
;                 Used ies lut list.
;
; Version     :	Version 0.7, 24/10/03
;-
;**********************************************************

PRO ies_pitch_angle, header, data, array, MF=mf

  ; check magnetic field available

  IF KEYWORD_SET(pa) AND (NOT ies_tag_exists(data[0],'MF')) AND (NOT KEYWORD_SET(mf)) THEN RETURN

  ; get pitch angle array i.e contains pitch angle for each sector, detector, sample
  ; NB valid pitch angle set to range 0->PI
  ; NB uses -ve values for invalid pitch angle or masked values
  ; NB important that this is a floating point array

  array = REPLICATE(-1.0, header.nsectors, header.ndetectors, header.npoints)

  ; loop through runs of fixed LUT

  ; get list of ends of runs of luts

  lut_list = ies_lut_list(data.lut, header.nluts)

  ; get sin and cos b field

  IF KEYWORD_SET(mf) THEN BEGIN
    cosb_az = COS(mf[0,*])
    sinb_az = SIN(mf[0,*])
    cosb_po = COS(mf[1,*])
    sinb_po = SIN(mf[1,*])
  ENDIF ELSE BEGIN
    cosb_az = COS(data.mf[0])
    sinb_az = SIN(data.mf[0])
    cosb_po = COS(data.mf[1])
    sinb_po = SIN(data.mf[1])
  ENDELSE

  ; loop through runs of each lut

  FOR l = 0, N_ELEMENTS(lut_list)-1 DO BEGIN

    ; get lut and no. of points in run

    lut = lut_list[l].lut

    ; dcount is no. of valid detectors
    ; dlist is index into data of valid detectors
    ; dpositions is angle where 0->180 is 0->dsize-1
    ; dpositions is also index into mask
    ; dposition 0 is antiparallel to spin

    ies_detector_polar_angle, header, lut, dlist, dcount, dsize, dpositions

    IF dcount GT 0 THEN BEGIN

      ; change so that dpos 0 is parallel to spin
      ; ensure array

      dpos = [dsize-1-dpositions]

      ; set polar and azimuth angles
      ; ensure arrays even if only one sector or detector by using [...]
      ; use fact that pitch angle of sector s is Pi - pitch angle of sector s+Pi

      d_po = [(0.5+dpos)/FLOAT(dsize)*!PI]

      ; check if even no. of sectors

      IF (header.nsectors MOD 2) EQ 0 THEN do_sectors = header.nsectors/2 ELSE do_sectors = header.nsectors

      d_az = [(0.5+INDGEN(do_sectors))/FLOAT(header.nsectors)*2*!PI]

      ; set xyz values for each sector/detectors
      ; these are xyz positions in sector/detector order

      d_x = REFORM(COS(d_az) # SIN(d_po),do_sectors*dcount)
      d_y = REFORM(SIN(d_az) # SIN(d_po),do_sectors*dcount)
      d_z = REFORM(REPLICATE(1,do_sectors) # COS(d_po),do_sectors*dcount)

      ; loop through each sample point

      FOR p = lut_list[l].st, (lut_list[l].en < n_elements(cosb_az)-1 )DO BEGIN

        ; set B field xyz

        b_x = cosb_az[p]*sinb_po[p]
        b_y = sinb_az[p]*sinb_po[p]
        b_z = cosb_po[p]

        ; get pitch angles for each sector/detector

        array[0:do_sectors-1,dlist,p] = ACOS( d_x*b_x + d_y*b_y + d_z*b_z) ; 0-pi

        ; check if even no of sectors

        IF do_sectors NE header.nsectors THEN array[do_sectors:*,dlist,p] = !PI - array[0:do_sectors-1,dlist,p]

      ENDFOR

    ENDIF

  ENDFOR

END
