;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_PLOT
;
; Purpose     :	Invokes widget tools for CEPPAD data display.
;
; Explanation : Displays data in different draw widgets.
;               Displays detector data in scrolling windows.
;               Displays angle/angle data.
;               Displays full range of data from selected detector.
;               Scrolls each detector plot in synchronization with cursor position in
;               full window plot.
;               Updates ANGLE/ANGLE plot in synchronization with cursor position in
;               full window plot.
;               Allows selection of different plot types.
;
; Use         : < ies_plot, input_header, input_data, RESTORE=restore_file >
;
; Inputs      : None.
;
; Opt. Inputs : input_header : header structure overwriting current header.
;               input_data   : data structure overwriting current data.
;
; Outputs     : Displays plots in window.
;
; Opt. Outputs:	PNG, GIF or PS files. May SAVE processed data.
;
; Keywords    : RESTORE  : Intermediate format file name for restoring header and data.
;
; Common      :	ies_draw_widget_block.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    : Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 5/6/95
;
; Modified    :	Version 0.1, 24/8/95
;                  Added PNG file output option.
;                  Modified PS output so always 5 plots per page.
;                  Modified so that always produces color bar.
;                  Modified so that ysize same on alpha and ultrix and copes with 9 detectors
;               Version 0.2, 17/1/96
;                  Modified to use ies_draw_widget
;               Version 0.3, 16/9/96
;                  Added text plot option.
;                  Added detector sort.
;                  Changed SHRINK to CLEAN.
;                  Extended ies_draw_widget_block.
;                  Added TIMELINE plot.
;                  Added ephemeris data.
;                  Added TRACK.
;                  Added clean tag to output info.
;               Version 0.4, 8/11/96
;                 Added PROFS button.
;               Version 0.5, 16/1/97
;                 Renamed output_info.clean tag to .shift.
;                 Added PROFILES plot.
;                 Removed PROFS button.
;                 Moved detector sort to ies extdata.
;                 Added DETECTOR vs SECTOR format.
;                 Added full_only to output_info.
;                 Removed TRACEBACK if no data file selected.
;                 Added time box to SPECTRAL, LINEPLOT and PROFILES.
;                 Modified time sampling in ANGLE/ANGLE, PROFILES.
;               Version 0.6, 15/4/97
;                 Changed so that DIV, CAL and RAT may be applied to clean data.
;                 Set up .cal, .div and .rat tags for new file.
;                 Added .gz file decompression with ies_gzip.
;               Version 0.7, MKC, 10/9/97.
;                 Added error return to top level.
;               Version 0.8, MKC, 23/11/97.
;                 Added correct small pixel sizes.
;               Version 0.9, MKC, 8/4/98.
;                 Added ies_get_dir.
;               Version 1.0, MKC, 31/7/98.
;                 Corrected a small bug in TIMELINE.
;               Version 1.1, MKC, 14/9/98.
;                 Changed initial default directory to current directory.
;               Version 1.2, 7/10/98
;                 Added Magnetic field and attitude arrays to common block.
;                 Rationalized use of colors.
;                 IES large pixels reversed in expected direction.
;                 Added sun, earth and magnetic field and projection to ANGLE/ANGEL and TIMELINE.
;               Version 1.3, 12/11/98
;                 Modified button assignments.
;                 Rationalized use of noutputs.
;               Version 1.4, 22/2/99
;                 Added elo and ehi uvalues.
;                 Changed to use of CW_PDMENU.
;                 Added min and max uvalues.
;                 Removed input_minimum and input_maximum.
;               Version 1.5, 17/3/99
;                 Removed setting of elo and ehi, min and max values.
;               Version 1.6, 25/3/99
;                 Added ies_ctitle.
;               Version 1.7, 15/9/99
;                 Deal with TIMER event from logoid.
;               Version 1.8, 14/12/99
;                 Added ies_pickfile.
;               Version 1.9, 11/2/00
;                 Removed /SAVE_CC from call to ies_widget_plot.
;               Version 2.0, 10/5/00
;                 Added check for POLAR before retrieving ephemeris data etc
;                 Added xregistered check to ensure only one instance running.
;           	Version 2.1, MKC, 28/5/00
;                 Added version no check for SENSITIVE keyword.
;           	Version 2.2, MKC, 30/8/00
;                 Fixed bug where if no. of energy bands = 1 then use of REFORM
;                 in calculation of average energy gives error.
;                 Cope with lut=-1.
;                 Changed so that uses output_info.minimum and maxcimum without ABS.
;                 Changed so that if no input data and no file given then returns with message.
;               Version 2.3, 10/12/00
;                 Added PNG output
;               Version 2.4, 15/01/02
;                 Corrected bug in retrieving MF and AT data.
;               Version 2.5, 13/02/02
;                 Changed FILTER for DAT files.
;               Version 2.6, 12/03/02
;                 Changed decimate to average in TIMELINE plots.
;               Version 2.7, 08/05/02
;                 Removed input_header, input_data and RESTORE_FILE arguments.
;                 Moved restore file to ies_getset.
;                 Used ies_plot_profiles, ies_plot_angleangle and ies_plot_timeline.
;               Version 2.8, 09/07/02
;                 Removed WYSIWYG keyword.
;                 Set ncolors.
;               Version 2.9, 08/02/03
;                 Added PROFILES plot.
;               Version 3.0, 29/09/03
;                 Added variable NPITCHS.
;                 Changed ies get dir.
;
; Version     :	Version 3.0, 29/09/03
;-
;**********************************************************

;**********************************************************
; ies_plot_event : event handler for main routine
;**********************************************************

PRO ies_plot_event, event

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information
  ; data          = structure array containing extracted data
  ; output_info   = structure containing plotting and processing info
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem, mf, at

  IF TAG_NAMES(event, /STRUCTURE_NAME) EQ 'WIDGET_TIMER' THEN BEGIN

    ; get more data

    ; replot data

    IF NOT ies_calibrate() THEN ies_widget_plot

    ; make logo widget generate event

    WIDGET_CONTROL, ids.logoid, TIMER=4.0

    RETURN

  END

  WIDGET_CONTROL, event.id, GET_UVALUE = uvalue

  CASE uvalue OF

   'FULL' : BEGIN

              ; don't respond to button releases

              IF event.release THEN RETURN

              IF NOT output_info.full_only THEN BEGIN

                ; event.x gives the cursor position in the fullw window in device co-ords
                ; want to translate this into a position in the scroll window device co-ords

                WIDGET_CONTROL, ids.draw_ids(0), GET_VALUE=win
                WSET, win

                ; reset co-ord conversion

                !X.S = dnfull

                ; kill any log scaling

                !X.TYPE=0

                r = CONVERT_COORD( [ event.x, event.y], /DEVICE, /TO_DATA)

                ; set up current time

                output_info.time = header.tstart + r(0)

                IF output_info.procname EQ 'SPECTRAL' OR output_info.procname EQ 'LINEPLOT' THEN BEGIN

                  WIDGET_CONTROL, ids.title_ids[0], SET_VALUE = 'Time = '+anytim2cal(output_info.time)

                  WIDGET_CONTROL, ids.draw_ids[1], GET_VALUE=win
                  WSET, win

                  ; reset co-ord conversion

                  !X.S = dnscroll

                  r = CONVERT_COORD( r, /DATA, /TO_DEVICE)

                  FOR k = 1, output_info.noutputs DO $
                    WIDGET_CONTROL, ids.draw_ids(k), SET_DRAW_VIEW=[r(0),0]

                ENDIF ELSE IF output_info.procname EQ 'PROFILES' THEN BEGIN

                  ies_plot_profiles, /WIDGET, EVENTX=event.x

                ENDIF ELSE IF output_info.procname EQ 'OVERLAY' THEN BEGIN

                  ies_plot_profiles, /WIDGET, EVENTX=event.x, /OVERLAY

                ENDIF ELSE IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN

                  ies_plot_angleangle, /WIDGET

                ENDIF ELSE IF output_info.procname EQ 'TIMELINE' THEN BEGIN

                  ies_plot_timeline, /WIDGET

                ENDIF

              ENDIF

            END

   'FILE'    : BEGIN

                CASE event.value OF

                  'DISMISS' : WIDGET_CONTROL, event.top, /DESTROY

                  'SAVE'    : BEGIN   ; allow current header and data to be saved

                               save_file = ies_pickfile ( PATH=ies_get_dir('IES',/POLAR), FILTER='*.DAT', $
                                 /WRITE, /FIX_FILTER, TITLE='Select output data file' )

                               IF save_file EQ '' THEN RETURN

                               ; change variable names

                               input_header = header
                               input_data   = data

                               SAVE, FILE=save_file, input_header, input_data, /XDR

                               input_header = 0
                               input_data   = 0

                              END

                  'RETURN'  : BEGIN
                                WIDGET_CONTROL, event.top, /DESTROY
                                ies_plot
                              END

                  'COLOUR'  : xloadct, GROUP = event.id

                  'BWPS' : err = ies_window_plot ( !D.NAME, file, /BWPS)

                  'PS'   : err = ies_window_plot ( !D.NAME, file, /POSTSCRIPT)

                  'PNG'  : err = ies_window_plot ( !D.NAME, file, /PNG)

                  'GIF'  : err = ies_window_plot ( !D.NAME, file, /GIF)

                  'WIN'  : err = ies_window_plot ( !D.NAME, file, /WINDOWS)

                  'TXT'  : ies_text_plot, POLAR=ies_instrument(header.datatype,/POLAR), RAPID=ies_instrument(header.datatype,/RAPID)

                ENDCASE

               END

   'MIN'     : BEGIN

                 IF WIDGET_INFO(ids.minid, /TEXT_EDITABLE) THEN BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.minid, SENSITIVE=0

                   WIDGET_CONTROL, ids.minid, EDITABLE=0

                   IF NOT ies_calibrate() THEN ies_widget_plot

                 ENDIF ELSE BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.minid, /SENSITIVE

                   WIDGET_CONTROL, ids.minid, /EDITABLE

                 ENDELSE

               END

   'MAX'     : BEGIN

                 IF WIDGET_INFO(ids.maxid, /TEXT_EDITABLE) THEN BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.maxid, SENSITIVE=0

                   WIDGET_CONTROL, ids.maxid, EDITABLE=0

                   IF NOT ies_calibrate() THEN ies_widget_plot

                 ENDIF ELSE BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.maxid, /SENSITIVE

                   WIDGET_CONTROL, ids.maxid, /EDITABLE

                 ENDELSE

               END

   'ELO'     : BEGIN

                 IF WIDGET_INFO(ids.eloid, /TEXT_EDITABLE) THEN BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.eloid, SENSITIVE=0

                   WIDGET_CONTROL, ids.eloid, EDITABLE=0

                   IF NOT ies_calibrate() THEN ies_widget_plot

                 ENDIF ELSE BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.eloid, /SENSITIVE

                   WIDGET_CONTROL, ids.eloid, /EDITABLE

                 ENDELSE

               END

   'EHI'     : BEGIN

                 IF WIDGET_INFO(ids.ehiid, /TEXT_EDITABLE) THEN BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.ehiid, SENSITIVE=0

                   WIDGET_CONTROL, ids.ehiid, EDITABLE=0

                   IF NOT ies_calibrate() THEN ies_widget_plot

                 ENDIF ELSE BEGIN

                   IF !VERSION.release GT '5.0.3' THEN WIDGET_CONTROL, ids.ehiid, /SENSITIVE

                   WIDGET_CONTROL, ids.ehiid, /EDITABLE

                 ENDELSE

               END

   'MINIMUM' : IF NOT ies_calibrate() THEN ies_widget_plot

   'MAXIMUM' : IF NOT ies_calibrate() THEN ies_widget_plot

   'ELOW'    : IF NOT ies_calibrate() THEN ies_widget_plot

   'EHIGH'   : IF NOT ies_calibrate() THEN ies_widget_plot

   'RANGE'   : BEGIN
                 WIDGET_CONTROL, ids.rngid, GET_VALUE=range
                 output_info.rng = range(0)
               END

   'NPITCHS' : BEGIN
                 WIDGET_CONTROL, ids.nptchid, GET_VALUE=npitchs
                 output_info.npitchs = 1 > npitchs[0] < 360
                IF NOT ies_calibrate() THEN ies_widget_plot
               END

   'CAL' :  BEGIN
              output_info.cal = 1 - output_info.cal
              IF NOT ies_calibrate() THEN ies_widget_plot
            END

   'RAT' :  BEGIN
              output_info.rat = 1 - output_info.rat
              IF NOT ies_calibrate() THEN ies_widget_plot
            END

   'LOG' :  BEGIN
              output_info.log = 1 - output_info.log
              IF NOT ies_calibrate() THEN ies_widget_plot
            END

   'DIV' :  BEGIN
              output_info.div = 1 - output_info.div
              IF NOT ies_calibrate() THEN ies_widget_plot
            END

   'SUB' : BEGIN
              output_info.subtract = 1 - output_info.subtract
              IF NOT ies_calibrate() THEN ies_widget_plot
            END

   'SHIFT' : BEGIN
              output_info.shift = 1 - output_info.shift
              IF NOT ies_calibrate() THEN ies_widget_plot
            END

   'TRACK' : BEGIN
               output_info.track = 1 - output_info.track
               WIDGET_CONTROL, ids.draw_ids(0), DRAW_MOTION_EVENTS=output_info.track, DRAW_BUTTON_EVENTS=1-output_info.track
            END

   'LOGO' : IF event.press THEN dsp_strarr, ies_logo_info() ; only respond to button press

    ELSE  : BEGIN

              ; check if one of the program button labels

              FOR button = 0, N_ELEMENTS(output_info.button_labels)-1 DO BEGIN

                IF uvalue NE '' AND uvalue EQ output_info.button_labels(button) THEN BEGIN

                  output_info.button_values(button) = 1 - output_info.button_values(button)

                  ; dont respond when exclusive buttons are unset

                  IF STRPOS(output_info.button_labels(button),'/') NE 0 OR $
                     output_info.button_values(button) THEN BEGIN

                    IF button EQ 0 THEN BEGIN

                      IF NOT ies_calibrate() THEN ies_widget_plot

                    ENDIF ELSE ies_widget_plot

                  ENDIF

                ENDIF

              ENDFOR

              ; check if need to change any draw widget characteristics

              IF output_info.procname EQ 'ANGLE/ANGLE' THEN BEGIN

                ; set band labels

                band_labels = ies_band_labels(header, output_info, output_chans)

                ; loop through widgets

                FOR k = 1, output_info.noutputs DO WIDGET_CONTROL, ids.title_ids[k], SET_VALUE = band_labels[k-1]

              ENDIF

           END

  ENDCASE

END

;**********************************************************
; main routine for widget
;**********************************************************

PRO ies_plot

   ; ensure only single version of ies_plot running
   ; brings to the fornt if registered

   IF XREGISTERED('ies_plot') THEN RETURN

   ; come back out to top level

;   ON_ERROR, 1

   ; set up no of colors and color table

   dummy = ies_ncolors(/SET, SHARED=39)

   ; extract data, set up buttons and program

   IF ies_do_prelims () THEN RETURN

   ; set up and realize display widget

   base = ies_draw_widget ()

   ; plot data in display widget

   ies_widget_plot

   ; make display widget visible

   XMANAGER, "ies_plot", base

END
