;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_PLOT_IMG
;
; Purpose     :	Draws polar/azimuth image.
;
; Explanation :	Draws polar/azimuth image.
;               Believe that on POLAR the electron sensor numbering gives sensors from
;               anti-parallel to parallel to the SC spin axis for the small pixels (of which there are six) and the opposite
;               way around for the large pixels (of which there are nine). Similarly IPS detector
;               numbering as used in header.detectors gives detectors from anti-parallel to parallel.
;               Mag field is given relative to the spin axis.
;
;               Angle/Angle plot :
;               Image is set up as sector vs detector.
;               Image is plotted using ORDER=0 so that image is drawn from the bottom up,
;               Image is indexed by dpositions in the y direction.
;
;               Believe that SAT table gives small pixel orientation so that detectors 0-8 are
;               anti-parallel to spin axis. Detector 1 occurs at bottom of image and plot
;               corresponding to the anti-parallel spin direction.
;
;               For CEPPAD/IES large pixels dpositions = 8->0 so that detector 8 is first element of image drawn at bottom.
;               CEPPAD/IES large detector 0 is parallel to spin axis at top of plot.
;
;               For CEPPAD/IPS dpositions = header.detectors-1.
;               header.detectors indexes detectors from anti-parallel to parallel to spin axis.
;               So top of plot is parallel to spin axis.
;
;               When plotting things in SC coord system such as sun position, earth position or magnetic field,
;               invert because up is parallel to spin.
;
; Use         : < ies_plot_img, header, datum, vol, type, XMARGIN=xmargin, YMARGIN=ymargin >
;
; Inputs      : header       : STRUCTURE giving header information
;               datum        : STRUCTURE for data at single sample point
;               vol          : ARRAY(nx,ny) covering 360 by 180 degrees for display
;               type         : INT indicating type of output device
;                                 0 : Plot window, non-scaleable
;                                 1 : Plot window, scaleable
;                                 2 : Whole device
;
; Opt. Inputs : pedestals    : FLTARR(nsamples) bin positions of pedestals for each sample.
;
; Outputs     : y            : FLTARR(ysize,nsamples) interpolated image
;
; Opt. Outputs:	None.
;
; Keywords    : XMARGIN : allows xmargin control of map window.
;               YMARGIN : allows ymargin control of map window.
;               MASK    : Flag indicating should generate mask.
;               PROJ    : Flag indicating should do projection.
;               MVALS   : Flag indicating should show MVALS..
;
; Calls       :	None.
;
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 23/10/98
;                 Adapted from ies_plot.
;
; Modified    :	Version 0.1, 12/11/98
;                 Modified button assignments.
;               Version 0.2, 01/02/99
;                 Made more compatible with non-CEPPAD data.
;                 Fixed bug in displaying postscript images with PROJ on.
;               Version 0.3, 24/02/99
;                 Fixed bug in displaying IES detectors.
;               Version 0.4, 20/8/00
;                 deal with lut=-1 for RAPID data.
;               Version 0.5, 28/02/02
;                 Added mvals plot to timeline and angle/angle
;               Version 0.6, 12/07/02
;                 Used ies_tag_exists.
;               Version 0.7, 3/02/03
;                 Made coding slightly more explicit.
;                 Added MVALS keyword.
;                 Used ies_detector_polar_angle.
;               Version 0.8, 06/03/03
;                 Added some comments.
;
; Version     :	Version 0.8, 06/03/03
;-
;**********************************************************

PRO ies_plot_img, header, datum, vol, type, XMARGIN=xmargin, YMARGIN=ymargin, MASK=mask, PROJ=proj, MVALS=mvals

  ; set up output device

  CASE type OF

    0 : BEGIN

          ; get device coords of plot window

          vector = CONVERT_COORD ( !X.WINDOW, !Y.WINDOW, /NORMAL, /TO_DEVICE )

          ; get origin and size in device co-ords (inclusive of axes)


          xorig = vector(0,0)
          yorig = vector(1,0)
          xsize = FIX(vector(0,1) - xorig + 1) ; need the  + 1 else lose lines 30/10/98
          ysize = FIX(vector(1,1) - yorig + 1)

          ; check xsize and ysize are reasonable

          IF xsize LE 0 OR ysize LE 0 THEN BEGIN
            PRINT, 'IMAGE WINDOW INVALID SIZE :', title
            RETURN
          ENDIF

        END

    1 : BEGIN
          xorig = 0
          yorig = 0
          xsize = 360
          ysize = 180
        END

    2 : BEGIN
          xorig = 0
          yorig = 0
          xsize = !D.X_SIZE
          ysize = !D.Y_SIZE
        END

  ENDCASE

  ; dpositions indexes position in image and gives position of detector relative to anti-spin direction
  ; image then goes from anti-spin to spin direction : bottom to top
  ; dlist indexes position in data array and header.detectors ie detectors used

  ies_detector_polar_angle, header, datum.lut, dlist, dcount, dsize, dpositions

  ; generate polar/azimuth projection

  img = BYTARR(header.nsectors,dsize)

  FOR d = 0, dcount-1 DO $
    img[*,dpositions[d]] = vol[*,dlist[d]]

  ; if projection is used and quick look then do not expand img

  IF NOT (type EQ 2 AND KEYWORD_SET(proj)) THEN BEGIN

    ; expand image

    img = img[header.nsectors*FINDGEN(xsize)/xsize,*]

    img = img[*,dsize*FINDGEN(ysize)/ysize]

  ENDIF

  ; do earth mask

  IF ies_tag_exists(datum,'EARTH') AND KEYWORD_SET(mask) THEN $
    ies_mask, datum.earth[0], datum.earth[1], datum.earth[2], img, 100b

  ; do sun mask

  IF ies_tag_exists(datum,'SUN') AND KEYWORD_SET(mask) THEN $
    ies_mask, datum.sun[0], datum.sun[1], datum.sun[2], img, 150b

  ; check if sinusoidal projection required

  IF KEYWORD_SET(proj) AND ies_tag_exists(datum,'MF') THEN BEGIN

    ; sinusoidal projection

    centre_lat = 180*(0.5-datum.mf[1]/!PI)
    centre_lon = 180*(((datum.mf[0]/!PI+1) MOD 2)-1)
    centre_rot = 0

    ; put B-field at north pole

    IF centre_lat GE 0 THEN BEGIN
      centre_lat = centre_lat - 90
    ENDIF ELSE BEGIN
      centre_lat = -centre_lat-90
      IF centre_lon GE 0 THEN centre_lon = centre_lon - 180 ELSE centre_lon = centre_lon + 180
      centre_rot = 180
    ENDELSE

    IF NOT KEYWORD_SET(xmargin) THEN xmargin = [0,0]
    IF NOT KEYWORD_SET(ymargin) THEN ymargin = [0,0]

    MAP_SET, centre_lat, centre_lon, centre_rot, /SIN, XMARGIN=xmargin, YMARGIN=ymargin, NOERASE=type NE 2

    ; use low compression if widget display

    IF type EQ 2 THEN compress=4 ELSE compress=1

    ; MAP_IMAGE expects centre of img[*,0] to be at LATMIN
    ; MAP_IMAGE expects centre of img[0,*] to be at LONMIN

    img = MAP_IMAGE(img, xorig, yorig, xsize, ysize, $
                    LATMIN=180./ysize/2.-90, LATMAX=90-180./ysize/2., $
                    LONMIN=360./xsize/2., LONMAX = 360-360./xsize/2., COMPRESS=compress)

    IF type EQ 1 THEN $
      TV, img, xorig, yorig, ORDER=0, XSIZE=xsize, YSIZE=ysize $
    ELSE $
      TV, img, xorig, yorig, ORDER=0

    ; do sun position

    IF ies_tag_exists(datum,'SUN') THEN $
      PLOTS, 180*(((datum.sun[0]/!PI+1) MOD 2)-1), 180*(0.5-datum.sun[1]/!PI), PSYM=2, COLOR=50

    ; do earth position

    IF ies_tag_exists(datum,'EARTH') THEN $
      PLOTS, 180*(((datum.earth[0]/!PI+1) MOD 2)-1), 180*(0.5-datum.earth[1]/!PI), PSYM=1, COLOR=50

  ENDIF ELSE BEGIN

    IF type EQ 1 THEN BEGIN

      TV, img, xorig, yorig, ORDER=0, XSIZE=xsize, YSIZE=ysize, /DATA

      data_coords = 1

    ENDIF ELSE BEGIN

      TV, img, xorig, yorig, ORDER=0

      device_coords = 1

    ENDELSE

    ; do magnetic field
    ; invert because Mag field is given relative to the spin axis corresponding to top of plot

    IF ies_tag_exists(datum,'MF') THEN BEGIN

      PLOTS, xorig+xsize*datum.mf[0]/2./!PI, yorig+ysize*(1-datum.mf[1]/!PI), PSYM=4, SYMSIZE=0.7, DATA=data_coords, DEVICE=device_coords

      PLOTS, xorig+xsize*((datum.mf[0]/2./!PI+0.5) MOD 1.), yorig+ysize*datum.mf[1]/!PI, PSYM=4, DATA=data_coords, DEVICE=device_coords

    ENDIF

    ; do mvals

    IF ies_tag_exists(datum,'MVALS') AND KEYWORD_SET(mvals) THEN BEGIN

      ; get m values for each sector

      m = rcs_decode_m(datum.mvals, /MVAL)

      ; get msigns for each sector

      msigns = rcs_decode_msigns(datum.mvals)

      ; plot perp and parallel for each sector

      FOR s = 0, 15 DO BEGIN

        ; perp direction
        ; invert because m=0 is parallel to spin axis

        PLOTS, xorig+xsize*(s+0.5)/16., yorig+ysize*(16-m[s]+0.5)/16., PSYM=2, SYMSIZE=0.5, DATA=data_coords, DEVICE=device_coords

        ; parallel direction (actually 90 deg to perp direction which is projection of B onto plane)

        IF msigns[s] THEN BEGIN

          PLOTS, xorig+xsize*(s+0.5)/16., yorig+ysize*(((24-m[s]) MOD 16)+0.5)/16., SYMSIZE=0.5, PSYM=7, DATA=data_coords, DEVICE=device_coords

        ENDIF ELSE BEGIN

          PLOTS, xorig+xsize*(s+0.5)/16., yorig+ysize*(((24-m[s]) MOD 16)+0.5)/16., SYMSIZE=0.5, PSYM=1, DATA=data_coords, DEVICE=device_coords

        ENDELSE

      ENDFOR

      IF ies_tag_exists(datum,'MF') THEN BEGIN

        ; convert direction to xyz in RAPID SC coords

        dirb = [COS(datum.mf[0])*SIN(datum.mf[1]),SIN(datum.mf[0])*SIN(datum.mf[1]),COS(datum.mf[1])]

        FOR s = 0, 15 DO BEGIN

          phi = (s+0.5)/16*2*!PI

          ; take dot product with vector along perp to spin axis
          ; and derive angle wrt spin axis of projection of B field onto sector plane ie closest parallel to B

          ; ang = ATAN((COS(phi)*dirb[0]+SIN(phi)*dirb[1]),dirb[2])

          ; find 90 deg to this
          ; this is where perpendicular to B field crosses sector plane

          ang = ATAN(-dirb[2],COS(phi)*dirb[0]+SIN(phi)*dirb[1])

          ang = (ang + !PI) MOD !PI

          ; b direction
          ; top is parallel to spin direction

          PLOTS, xorig+xsize*(s+0.5)/16., yorig+ysize*(1-ang/!PI), PSYM=6, SYMSIZE=1.0, DATA=data_coords, DEVICE=device_coords

        ENDFOR

      ENDIF

    ENDIF

    ; do sun position
    ; sun position is given relative to the spin axis corresponding to top of plot

    IF ies_tag_exists(datum,'SUN') THEN $
      PLOTS, xorig+xsize*datum.sun[0]/2./!PI, yorig+ysize*(1-datum.sun[1]/!PI), PSYM=2, COLOR=50, DATA=data_coords, DEVICE=device_coords

    ; do earth position
    ; earth position is given relative to the spin axis corresponding to top of plot

    IF ies_tag_exists(datum,'EARTH') THEN $
       PLOTS, xorig+xsize*datum.earth[0]/2./!PI, yorig+ysize*(1-datum.earth[1]/!PI), PSYM=1, COLOR=50, DATA=data_coords, DEVICE=device_coords

  ENDELSE

END