;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_PLOT_MISC
;
; Purpose     :	Does plot of miscellaneous data.
;
; Explanation :	Does plot of miscellaneous data.
;
; Use         : < ies_plot_misc, time, data, /WINDOWS, /PS, /PNG, /GIF, /BWPS, $
;                               T0=start_time, T1=end_time >
;
; Inputs      : None.

; Opt. Inputs : None.
;
; Outputs     : Draws plot on current graphics device.
;
; Opt. Outputs:	None.
;
; Keywords    : /WINDOWS    = plots output in x window.
;               /PS         = plots output as postscript file
;                                if string then assumes it is a file name
;               /PNG        = plots output as PNG file
;                                if string then assumes it is a file name
;               /GIF        = plots output as GIF file
;                                if string then assumes it is a file name
;               /BWPS       = plots output as black and white postscript file
;                                if string then assumes it is a file name
;               T0          = See cep_l1toif.
;               T1          = See cep_l1toif.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 04/02/02
;
; Modified    :	Version 0.1, 10/06/02
;                 Changed POSTSCRIPT keyword to PS keyword.
;               Version 0.2, 09/07/02
;                 Removed ncolors from argument lists.
;               Version 0.3, 28/03/03
;                 Changed definition of logo positions.
;               Version 0.4, 01/10/03
;                 Added POLAR switch to ies set plot.
;
; Version     :	Version 0.4, 01/10/03
;-
;**********************************************************

PRO ies_plot_misc, itime, idata, iluts, WINDOWS=windows, PS=postscript, PNG=png, GIF=gif, BWPS=bwps, $
      T0=start_time, T1=end_time, HEADING=heading, YTITLE=ytitle, _EXTRA=extra

  ; come back out to top level

 ; ON_ERROR, 1

  ON_IOERROR, IOERROR

  ; set current device name

  devname = !D.NAME

  npoints = N_ELEMENTS(itime)

  ; get calendar day corresponding to midpoint

  midday = anytim2cal ( (itime(0)+itime(npoints-1))/2, FORM = 9, /DATE)

  md = anytim2utc(midday)

  ; select time interval

  IF KEYWORD_SET(start_time) THEN BEGIN
    start_time = anytim2utc(start_time)
    start_time.mjd = md.mjd
    start_time = utc2tai(start_time)
  ENDIF ELSE start_time = itime(0)

  IF KEYWORD_SET(end_time) THEN BEGIN
    end_time = anytim2utc(end_time)
    end_time.mjd = md.mjd
    end_time = utc2tai(end_time)
  ENDIF ELSE end_time = itime(npoints-1)

  time_list = WHERE ( itime GE start_time AND itime LE end_time, npoints)

  IF npoints EQ 0 THEN MESSAGE, 'NO DATA IN TIME RANGE'

  time = itime[time_list]

  IF (SIZE(idata))[0] EQ 1 THEN data = REFORM(idata[time_list], npoints, 1, 1) $
  ELSE data = idata[time_list,*,*]

  ndetectors = (SIZE(data))[3]

  IF KEYWORD_SET(iluts) THEN luts = iluts[time_list]

  IF NOT KEYWORD_SET(heading) THEN heading  = ['PLOT for ']

  heading = heading + midday

  IF NOT KEYWORD_SET(ytitle) THEN ytitle = ''

  ; set default plot type

  IF NOT KEYWORD_SET(postscript) AND NOT KEYWORD_SET(png) AND NOT KEYWORD_SET(gif) AND NOT KEYWORD_SET(bwps) THEN windows=1

  ; set output device

  IF ies_set_plot( 0, devname, unit, file, /POLAR, LOGO=2, $
                   WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps) THEN RETURN

  ; get character sizes in normal coords

  x_chsize = get_chsize(/X)
  y_chsize = get_chsize(/Y)

  ; do title

  XYOUTS, 0.5 - STRLEN(heading)*x_chsize*1.5/2.0, 1-2*y_chsize, heading, CHARSIZE=1.5*!P.CHARSIZE, /NORMAL

  ytop = 1-2.5*y_chsize

  ; do footnotes

  get_utc, utc

  footnote = 'Page 1 : ' + anytim2cal(utc, FORM=9)

  XYOUTS, 1.0 - STRLEN(footnote)*0.6*x_chsize, y_chsize*0.1, footnote, /NORMAL, CHARSIZE=0.6*!P.CHARSIZE

  footnote = strip_dirname(file)
  XYOUTS, 0.01, y_chsize*0.1, footnote, /NORMAL, CHARSIZE=0.6*!P.CHARSIZE

  ; set up plot region parameters

  ysiz = (ytop-y_chsize*1.7)/10.0

  ; adjust charsize for plots

  !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]

  ies_charsize

  ; get new character sizes in normal coords

  x_chsize = get_chsize(/X)
  y_chsize = get_chsize(/Y)

  ; fix margins so that in postcript format plot size is 16 cm

  xmargin = [10.0,16.0]

  xmargin = 3.9/19.9/x_chsize * xmargin/TOTAL(xmargin)

  ymargin = [0,0]

  ; loop over detectors

  FOR pg_pos = 0, ndetectors-1 DO BEGIN

    !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - pg_pos*[ 0.0, ysiz, 0.0, ysiz]

    ; do line plot of data

    ies_draw_pedestal, time, data[*,*,pg_pos], xmargin, ymargin, TITLE='Det. ' + STRTRIM(pg_pos+1,2), $
      NO_XAXIS = KEYWORD_SET(luts) OR (pg_pos NE ndetectors-1), YTITLE = ytitle, _EXTRA=extra

  ENDFOR

  ; plot luts

  IF KEYWORD_SET(luts) THEN BEGIN

    !P.REGION = [ 0.0, ytop-ysiz, 1.0, ytop]  - ndetectors*[ 0.0, ysiz, 0.0, ysiz]

    ies_draw_luts, time, luts, xmargin, ymargin, 'IES'

  ENDIF

  ; finish off

  ies_end_plot, file, devname, WINDOWS=windows, POSTSCRIPT=postscript, PNG=png, GIF=gif, BWPS=bwps

  RETURN

IOERROR:

  MESSAGE, !ERR_STRING, /TRACEBACK

END
