;+
; Project     :	CLUSTER - RAPID
;
; Name        :	IES_PLOT_PROFILES
;
; Purpose     :	Plots PROFILES.
;
; Explanation : Plots PROFILES.
;
; Use         : < ies_plot_profiles, xsiz, ysiz, ytop, ncolors, ncolumns, WIDGET=widget >
;
; Inputs      : None.
;
; Opt. Inputs : xsiz : FLOAT
;               ysiz : FLOAT
;               ytop : FLOAT
;               ncolors : INT
;               ncolumns : INT
;
; Outputs     : Displays plots in window.
;
; Opt. Outputs:
;
; Keywords    : WIDGET  : Flag for widget rather than window output.
;
; Prev. Hist. :	Combined IES_PLOT and IES_WINDOW_PLOT versions.
;
; Written     :	Version 0.0, Martin Carter, RAL, 08/05/02
;
; Modified    :	Version 0.1, 07/02/03
;                 Dealt with pitch angle data containing -1.
;                 Added GYRORADIUS vs ENERGY
;                 Added OVERLAY keyword.
;                 brought some of processing outside loop.
;
; Version     :	Version 0.1, 07/02/03
;-
;**********************************************************

PRO ies_plot_profiles, xsiz, ysiz, ytop, ncolumns, WIDGET=widget, EVENTX=eventx, OVERLAY=overlay

  ; define common blocks used internally for draw widgets
  ; full_header   = structure containing original data header information
  ; full_data     = structure array containing original data
  ; header        = structure containing extracted data header information
  ; data          = structure array containing extracted data
  ; output_info   = structure containing plotting and processing info
  ; output_chans  = array containing processed channel positions
  ; output_array  = array containing processed data for plotting
  ; ids           = structure giving widget IDs
  ; dnfull        = co-ordinate conversion array for full window
  ; dnscroll      = co-ordinate conversion array for magnified windows
  ; files         = STRARR giving data file names
  ; ephem         = structure containing ephemeris data
  ; mf            = structure containing magnetic field data
  ; at            = structure containing attitude data

  COMMON ies_draw_widget_block, full_header, full_data, header, data, output_info, output_chans, output_array, $
                                ids, dnfull, dnscroll, files, ephem, mf, at

  ; get list of good data

  IF output_info.button_values[2] THEN good_list = ies_good_data(data.flag, COUNT=count) $
  ELSE BEGIN
    count = header.npoints
    good_list = LINDGEN(count)
  ENDELSE

  IF count EQ 0 THEN RETURN

  ; check time in range

  IF ( output_info.time GE data(0).time ) AND $
     ( output_info.time LE data(N_ELEMENTS(data)-1).time ) THEN BEGIN

     ; get corresponding samples

     list_a = WHERE ( data[good_list].time LT (output_info.time-output_info.rng), pos_a)
     list_b = WHERE ( data[good_list].time LT (output_info.time+output_info.rng), pos_b)
     list_c = WHERE ( data[good_list].time LT output_info.time, pos_c)

     IF pos_a GT 0 AND pos_b GT 0 THEN BEGIN

       ; get sample indeces

       a = list_a(pos_a-1)
       b = list_b(pos_b-1)
       c = list_c(pos_c-1)

       ; get lut of central point

       lut = data(good_list[c]).lut

       ; get luts

       luts = data(good_list[a:b]).lut

       ; get indeces where luts=lut

       list = good_list[a + WHERE(luts EQ lut)]

       IF KEYWORD_SET(widget) THEN BEGIN

         WIDGET_CONTROL, ids.title_ids[0], $
           SET_VALUE = 'Profile plot start time = '+anytim2cal(data(good_list[c]).time)

         save_x = [data(list).time-header.tstart]

         save_y = REPLICATE ( (CONVERT_COORD( [ eventx, 1], /DEVICE, /TO_DATA))(1), N_ELEMENTS(save_x) )

         xx = !X
         yy = !Y

         OPLOT, save_x, save_y, PSYM=3, /NOCLIP

       ENDIF

       ; get data for plot

       plot_data = REFORM(output_array(0:output_info.noutputs-1,*,list), $
                          output_info.noutputs, output_info.ysamples, N_ELEMENTS(list))

       ; special case for pitch angle data

       IF output_info.format EQ 'PITCH ANGLE vs ENERGY' OR output_info.format EQ 'GYRORADIUS vs ENERGY' THEN BEGIN

         ; ie sector/detector data for which no pitch angle present designated by NaN

         pdata = FLTARR(output_info.noutputs, output_info.ysamples)

         FOR o = 0, output_info.noutputs-1 DO FOR s = 0, output_info.ysamples-1 DO BEGIN

           ; check for data containing -1

           lis = WHERE( plot_data[o,s,*] GE 0, co)

           IF co GT 0 THEN pdata[o,s] = TOTAL(plot_data[o,s,lis])/co

         ENDFOR

         plot_data = pdata

       ENDIF ELSE BEGIN

         plot_data = TOTAL(plot_data,3) /N_ELEMENTS(list)

       ENDELSE

       ; set global plot min and max

       plot_min = output_info.minimum
       plot_max = output_info.maximum

       ; set default plotting symbol
       ; avoid joining to invalid data

       psym = 4

       psign = -1
       IF output_info.format EQ 'PITCH ANGLE vs ENERGY' OR output_info.format EQ 'GYRORADIUS vs ENERGY' THEN psign = 1

       ; set x sampling

       IF output_info.button_values(3) THEN xmin = 0.5 ELSE xmin = 0.0 ; check if xlog button set

       xmax = output_info.ysamples

       centres = FINDGEN(xmax) > xmin

       IF output_info.format EQ 'SECTOR vs DETECTOR'  THEN BEGIN
         xtitle  = 'Sector'
       ENDIF ELSE IF output_info.format EQ 'DETECTOR vs SECTOR'  THEN BEGIN
         xtitle  = 'Detector'
       ENDIF ELSE IF output_info.format EQ 'PITCH ANGLE vs ENERGY'  THEN BEGIN
         IF output_info.button_values(0) THEN BEGIN
           xtitle  = 'Pitch angle [deg]'
           xmin = MIN(output_chans)
           IF output_info.button_values(3) THEN xmin = xmin > 0.5 ; check if xlog button set
           xmax = MAX(output_chans)
         ENDIF ELSE BEGIN
           xtitle  = 'Pitch angle bin'
         ENDELSE
       ENDIF ELSE IF output_info.format EQ 'GYRORADIUS vs ENERGY'  THEN BEGIN
         IF output_info.button_values(0) THEN BEGIN
           xtitle  = 'Gyro-radius [km]'
           xmin = MIN(output_chans)
           xmax = MAX(output_chans)
         ENDIF ELSE BEGIN
           xtitle  = 'Gyro-radius bin'
         ENDELSE
       ENDIF ELSE IF output_info.button_values(0) THEN BEGIN
         xtitle  = 'Energy [keV]'
         xmin = MIN(output_chans)
         IF output_info.button_values(3) THEN xmin = xmin > 0.5 ; check if xlog button set
         xmax = MAX(output_chans)
       ENDIF ELSE BEGIN
         xtitle  = 'Energy [Channel]'
         xmax    = header.nbands
         centres = FINDGEN(xmax) > xmin
       ENDELSE

       ; adjust x axis if explicitly set

       IF output_info.elo THEN xmin = output_info.elow
       IF output_info.ehi THEN xmax = output_info.ehigh

       ; define z title

       ztitle = ies_ctitle(header,output_info)

       IF KEYWORD_SET(overlay) THEN BEGIN

         IF KEYWORD_SET(widget) THEN BEGIN

           WIDGET_CONTROL, ids.draw_ids[1], GET_VALUE=win

           WSET, win

           !P.REGION   = [0.0, 0.0, 1.0, 1.0]

         ENDIF ELSE BEGIN

           !P.REGION = [ 0, ytop-ysiz, xsiz, ytop]

         ENDELSE

         PLOT, [0,1], /NODATA, XMARGIN=[8,20], $
                      XLOG=output_info.button_values(3), $
                      YLOG=output_info.log, $
                      XSTYLE = 1, $
                      YSTYLE = 1, $
                      TITLE  = output_info.format, $
                      XTITLE = xtitle, $
                      YTITLE = ztitle, $
                      YRANGE=[plot_min, plot_max], $
                      XRANGE = [xmin,xmax]

         !P.POSITION = [!X.WINDOW[0], !Y.WINDOW[0], !X.WINDOW[1], !Y.WINDOW[1]]

       ENDIF

       ; loop through outputs

       FOR k = 0, output_info.noutputs-1 DO BEGIN

         IF KEYWORD_SET(widget) THEN BEGIN

            IF NOT KEYWORD_SET(overlay) THEN BEGIN

              WIDGET_CONTROL, ids.draw_ids[k+1], GET_VALUE=win

              WSET, win

            ENDIF

         ENDIF ELSE BEGIN

            ; set plot region

            IF NOT KEYWORD_SET(overlay) THEN BEGIN

              !P.REGION = [ (k MOD ncolumns)*xsiz, ytop-ysiz*(k/ncolumns+1), $
                            (k MOD ncolumns)*xsiz+xsiz, ytop-ysiz*(k/ncolumns)]

            ENDIF

         ENDELSE

         ; adjust centres

         IF output_info.format EQ 'SECTOR vs DETECTOR' OR output_info.format EQ 'DETECTOR vs SECTOR'  THEN BEGIN
         ENDIF ELSE IF output_info.format EQ 'PITCH ANGLE vs ENERGY'  OR output_info.format EQ 'GYRORADIUS vs ENERGY'  THEN BEGIN
           IF output_info.button_values(0) THEN BEGIN
             centres = REFORM([TOTAL(output_chans(*,*,k,lut>0), 1)] )/2.0
           ENDIF
         ENDIF ELSE IF output_info.button_values(0) THEN BEGIN
           centres = REFORM([TOTAL(output_chans(*,*,k,lut>0), 1)] )/2.0
         ENDIF

         ; NB [] ensures still array if only one energy

         IF KEYWORD_SET(widget) THEN noerase=0 ELSE noerase=1

         IF KEYWORD_SET(overlay) THEN BEGIN

           psym=(k MOD 6) + 1
           IF psym GE 3 THEN psym = psym+1

           OPLOT, centres, [plot_data(k,*)], PSYM=psign*psym, COLOR=ies_get_color(k)

         ENDIF ELSE BEGIN

           PLOT, NOERASE=noerase, centres, $
             [plot_data(k,*)], $
             XLOG=output_info.button_values(3), $
             YLOG=output_info.log, $
             XSTYLE = 1, $
             YSTYLE = 1, $
             TITLE  = output_info.ptitle[k], $
             XTITLE = xtitle, $
             YTITLE = ztitle, $
             YRANGE=[plot_min, plot_max], $
             XRANGE = [xmin,xmax], $
             PSYM=psign*psym

         ENDELSE

       ENDFOR

       IF KEYWORD_SET(overlay) THEN ies_do_labels, output_info.noutputs, output_info.ptitle, TITLE='Key'

       IF KEYWORD_SET(widget) THEN BEGIN

         WIDGET_CONTROL, ids.draw_ids(0), GET_VALUE=win
         WSET, win

         ; reset co-ord conversion

         !X = xx
         !Y = yy

         OPLOT, save_x, save_y, PSYM=3, COLOR=0, /NOCLIP

       ENDIF

     ENDIF ; pos_a,pos_b gt 0

  ENDIF ; time in range

END