;+
; Project     :	CLUSTER - RAPID, POLAR - CEPPAD
;
; Name        :	IES_RAT
;
; Purpose     :	Converts data from counts to counts per sec and the reverse.
;
; Explanation :	Converts counts to count rate and the reverse.
;               Note that the output_array is not actually count rate but count rate / nsectors.
;
; Use         : < ies_rat, data, nsectors, nbands, ndetectors, output_array, REVERSE=reverse  >
;
; Inputs      : data          : STRUCTURE[] for each sample.
;               nsectors      : INT no. of sectors.
;               nbands        : INT no. of energy channels.
;               ndetectors    : INT no. of detectors.
;               output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing input data
;
; Opt. Inputs : None.
;
; Outputs     : output_array  : FLTARR(nsectors, ndetectors, nbands, npoints) containing processed data.
;
; Opt. Outputs:	None.
;
; Keywords    : REVERSE : flag indicating conversion back to counts required.
;
; Calls       :	None.
;
; Common      : None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 13/11/96
;                 Adapted from ies_cal.
;
; Modified    :	Version 0.1, 15/4/97
;                 Allowed reverse change.
;               Version 0.2, 8/5/01
;                 Added ENDTIMES keyword.
;               Version 0.3, 14/04/03
;                 Removed ENDTIMEs keyword.
;                 Changed to use data structure aray rather than times array argument.
;                 Used ies_endtimes.
;               Version 0.4 07/07/03
;                 Improved time differences when bad times encountered.
;                 Corrected bug in calculating time differenes introduced in version 0.3 which made all time differences
;                 negative !
;               Version 0.5, 21/10/03
;                 Added MESSAGE if bad times encountered.
;               Version 0.6, 09/01/04
;                 Added more info if bad times encountered.
;
; Version     :	Version 0.6, 09/01/04
;-
;**********************************************************

PRO ies_rat, data, nsectors, nbands, ndetectors, output_array, REVERSE=reverse

  npoints = N_ELEMENTS(data)

  ; convert from counts to counts per sec

  ; get time difference array
  ; NB last point is dummy point if full range
  ;    will always be at least two points unless only one point in l1 file

  end_times = ies_endtimes(data)

  time_diffs =  end_times - data.time

  ; fix any duff time differences

  bad_list = WHERE ( time_diffs LE 0.001, bcount )

  IF bcount GT 0 THEN BEGIN

    PRINT, 'WARNING, bad times encountered in ' + TAG_NAMES(data,/STRUCTURE_NAME)
    MESSAGE, 'WARNING, bad times encountered in ' + TAG_NAMES(data,/STRUCTURE_NAME), /INFORMATIONAL

    IF ies_debug() THEN BEGIN

      PRINT, 'Bad times = ' + bcount + '/' + npoints

      FOR k=0, bcount-1 DO PRINT, anytim2cal(data[bad_list[k]].time) + ' to ' + anytim2cal(end_times[bad_list[k]])

    ENDIF

    good_list = WHERE ( time_diffs GT 0.001, gcount )

    If gcount EQ 0 THEN MESSAGE, 'FAILED TO CORRECT BAD TIMES'

    mean_diff = TOTAL(time_diffs[good_list])/gcount

    time_diffs[bad_list] = mean_diff

  ENDIF

  IF KEYWORD_SET(reverse) THEN BEGIN

    ; do time difference multiplication for each energy and each output
    ; remember rates are stored as count rate/nsectors, counts are stored as counts

    FOR s = 0L, nsectors-1 DO FOR d = 0L, ndetectors-1 DO FOR e = 0L, nbands-1 DO $
        output_array(s,d,e,*) = output_array(s,d,e,*) * time_diffs

  ENDIF ELSE BEGIN

    ; do time difference division for each energy and each output
    ; remember rates are stored as count rate/nsectors, counts are stored as counts

    FOR s = 0L, nsectors-1 DO FOR d = 0L, ndetectors-1 DO FOR e = 0L, nbands-1 DO $
        output_array(s,d,e,*) = output_array(s,d,e,*) / time_diffs

  ENDELSE

END
