;+
; Project     :	CLUSTER - PEACE
;
; Name        :	IES_REBIN_Y
;
; Purpose     :	Rebins y axis within a 2-D image. Allows pedestal shift and interpolation. 
;
; Explanation :	Associated with each column of the input image is a LUT number which defines the
;               bin positions of the y samples. This routine maps the input image yaxis into a new 
;               size optionally taking into account the different column to column y mapping i.e the
;               different channel sizes.  
;               It also allows an option to shift each column to match pedestal positions.   
;
; Use         : < ies_rebin_y, y, luts, ch_positions, nchannels, yp, shifted_pedestals, $
;                   SHOWCH=showch, SHFTCH=shftch, INTERP=interp >
;
; Inputs      : y            : FLTARR(nchannels,nsamples) data values at each sample point for each channel
;               luts         : BYTARR(nsamples) LUT used for each sample point
;               ch_positions : FLTARR(2,nchannels,1,nluts) channel bin positions for each LUT.
;               nchannels    : LONG no. of y channels            
;               yp           : FLTARR channel or bin boundaries for each new y sample
;               shifted_pedestals : FLTARR(nsamples) pedestal position in kev
;
; Opt. Inputs : None.
;
; Outputs     : y            : FLTARR(ysize,nsamples) rebinned image
;
; Opt. Outputs:	None.
;
; Keywords    :/SHOWCH   : indicates whether to show real channel sizes or not
;              /SHFTCH   : indicates whether to shift data to match pedestal positions.
;                          Only available if showch is also set.
;
; Calls       :	ies_resample.
;                
; Common      :	None.
;
; Restrictions:	None.
;
; Side effects:	None.
;
; Category    :	Display.
;
; Prev. Hist. :	None.
;
; Written     :	Version 0.0, Martin Carter, RAL, 26/9/96
;                 Adapted from ies_mk_image.       
;
; Modified    :	Version 0.1, 30/10/96.
;                 Changed order of processing channels so integral channel does not overwrite lower channels.
;              	Version 0.2, 11/11/96
;                 Removed pedestal position calculation.
;              	Version 0.3, 2/4/97
;                 Changed yp to bin boundaries rather than bin centres.
;              	Version 0.4, 2/3/98
;                 Removed interpolation option. Removed minimum argument.
;
; Version     :	Version 0.4, 2/3/98
;-
;**********************************************************

PRO ies_rebin_y, y, luts, ch_positions, nchannels, yp, shifted_pedestals, $
                 SHOWCH=showch, SHFTCH=shftch

  ; fiddle y range of data to fit into plot box

  ; get maps for each unique lut

  IF NOT KEYWORD_SET(showch) THEN BEGIN

    ; get new no. of y samples

    ysize = N_ELEMENTS(yp)-1

    ; get channel centres

    yp = (yp(1:ysize)+yp(0:ysize-1))/2.0

    ; get bin or channel no.s

    yp = FIX(yp)

    ; expand y values 

    y = y(yp,*)

  ENDIF ELSE BEGIN ; end NOT showch

    ; resample data

    ies_resample, y, luts, ch_positions, shifted_pedestals, nchannels, yp, SHFTCH=shftch

  ENDELSE ; end showch

END 