;**********************************************************
;+
; Project     :	CLUSTER II - RAPID
;
; Name        :	IES_REPROCESS
;
; Purpose     :	Reprocesses RAPID and POLAR IF data files to produce new IF data files.
;
; Explanation : CEPPAD/RAPID IF files are selected.
;               Each of these is processed appropriately and a new IF file produced.
;               IF file sub-directory named appropriately.
;               Preserves SW version number of original IF data file.
;
; Use         : < ies_reprocess, FILES=files, INTEGRAL=integral >
;
; Inputs      : None.
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    : FILES    : STRARR[] list of IF files for reprocessing.
;               INTEGRAL : FLAG indicating integral data set created.
;                          If present then creates IF data file with flux integrated over energies(ie cts/s/cm2/sr),
;                          sectors and detectors.
;               ENDTIME  : Flag indicating should explicitly add data structure endtime values
;               GSE      : FLAG add GSE coords
;               GSM      : FLAG add GSM coords
;               SM       : FLAG add SM  coords
;               MF       : FLAG add a magnetic field tag to header and data structure
;                          If tag already exists then will update.
;               PA       : FLAG add pitch angle
;                          If MF tag already exist then will use.
;               POLAR    : FLAG indicating POLAR S/C.
;               RAPID    : FLAG indicating RAPID S/C.
;
; Written     :	Version 0.0, 01/01/01
;
; Modified    : Version 0.1, 11/10/2002
;                 Removed ephemeris info for now so dont need pgp and geo routines
;               Version 0.2, 06/01/03
;                 Modified routine to be more generic.
;               Version 0.3, 07/03/03
;                 Fixed bug with invalid argument list for rcs_restore_ifdata.
;                 Fixed bug with energy ranges in call to rcs_sum.
;               Version 0.4, 14/03/03
;                 Fixed bug to do with energy efficiencies not being set in INTEGRAL structure
;               Version 0.5, 26/09/03
;                 Fixed bug if INTEGRAL used for IF data sets with more than detector or sector.
;               Version 0.6, 08/10/03
;                 Used ies get dir.
;               Version 0.7, 09/10/03
;                 Combined with cep reprocess and renamed from reprocess_rapid.
;                 Added POLAR and RAPID keywords.
;                 Renamed routine rcs sum to ies sum.
;               Version 0.8, 24/10/03
;                 Defaulted to RAPID
;                 Corrected bug whereby mf tag set up in header if /pa used (but not in data)
;               Version 3.3, 31/10/03
;                 Changed name of ies magnetic field routine.
;                 Changed so that uses new magnetic field if /MF or /PA selected.
;
; Version     :	Version 3.3, 31/10/03
;-
;**********************************************************

PRO ies_reprocess, FILES=files, INTEGRAL=integral, GSE=gse, GSM=gsm, SM=sm, PA=pa, MF=mf, $
                   ENDTIME=endtime, POLAR=polar, RAPID=rapid

  ; default to RAPID

  IF NOT KEYWORD_SET(polar) AND NOT KEYWORD_SET(rapid) THEN rapid=1

  ; get IF files

  IF NOT KEYWORD_SET(files) THEN BEGIN

    files = ies_pickfile ( PATH=ies_get_dir('IF',RAPID=rapid,POLAR=polar), COUNT=fcount, /MULTIPLE, $
                           TITLE='Select IF files for processing or CANCEL',/MUST_EXIST  )

    ; deal with CANCEL

    IF fcount EQ 0 THEN BEGIN
      PRINT, 'No files selected for restoring'
      RETURN
    END

  ENDIF ELSE BEGIN

    fcount = N_ELEMENTS(files)

  ENDELSE

  ; sort files

  files = files[SORT(files)]

  ; loop through files

  FOR f = 0, fcount-1 DO BEGIN

    ; print some info

    PRINT, 'Processing IF file : '+files[f]

    ; restore IF file

    ies_restore_gzip, files[f], input_header, input_data, /CHECK

    ; set up version to form a_b
    ; early sw which had float version

    version = STRING(input_header.version)

    IF STRPOS(version,'.') GT 0 THEN STRPUT, version, '_', STRPOS(version,'.')

    ; set up data size

    no_of_points = input_header.npoints

    IF KEYWORD_SET(integral) THEN BEGIN

      no_of_detectors = 1

      no_of_energies = 1

      no_of_sectors = 1

    ENDIF ELSE BEGIN

      no_of_detectors = input_header.ndetectors

      no_of_energies = input_header.nbands

      no_of_sectors = input_header.nsectors

    ENDELSE

    no_of_luts = input_header.nluts

    ; set up new input_header data type
    ; structure name is defined by EXPERIMENT keyword
    ; structure datatype tag is defined by DATATYPE keyword

    IF ies_instrument(input_header.datatype,/POLAR) THEN BEGIN

      datatype = 'POLAR' + '_'+cep_ifname_datatype(files[f])

    ENDIF ELSE IF ies_instrument(input_header.datatype,/RAPID) THEN BEGIN

      datatype = input_header.datatype

    ENDIF ELSE BEGIN

      MESSAGE, 'IMPLEMENTATION ERROR', /TRACEBACK

    ENDELSE

    ; need to change datatype since this determines the sub-directory
    ; dont change twice though

    IF KEYWORD_SET(integral) AND STRPOS(datatype,'_INTEGRAL') LT 0 THEN datatype = datatype + '_INTEGRAL'

    IF KEYWORD_SET(gse) AND STRPOS(datatype,'_GSE') LT 0  THEN datatype = datatype + '_GSE'

    IF KEYWORD_SET(gsm) AND STRPOS(datatype,'_GSM') LT 0  THEN datatype = datatype + '_GSM'

    IF KEYWORD_SET(sm) AND STRPOS(datatype,'_SM') LT 0  THEN datatype = datatype + '_SM'

    IF KEYWORD_SET(mf) AND STRPOS(datatype,'_MF') LT 0  THEN datatype = datatype + '_MF'

    IF KEYWORD_SET(pa) AND STRPOS(datatype,'_PA') LT 0  THEN datatype = datatype + '_PA'

    IF KEYWORD_SET(endtime) AND STRPOS(datatype,'_ENDTIME') LT 0  THEN datatype = datatype + '_ENDTIME'

    ; set up new input_header structure name

    experiment = datatype

    ; set up optional tags to ensure passed through to new structure

    ; set up lut_times only if present
    ; Nb change structure name to indicate no of luts

    IF ies_tag_exists(input_header,'LUT_TIMES') THEN BEGIN

      lut_times = input_header.lut_times

      ; rapid has variable no .of luts
      ; lut_times implies variable no of luts
      ; if add to CEPPAD structure name will get reflected in the IF directory

      experiment = experiment + '_' + STRTRIM(no_of_luts,2)

    ENDIF

    ; explicitly set up lut_map if not present
    ; (for instance not present in CEPPAD IPS, HIST data sets)
    ; since needed in the summation (NB ies extdata explicilty sets up also)

    IF ies_tag_exists(input_header,'LUT_MAP') THEN BEGIN

      lut_map = input_header.lut_map

    ENDIF ELSE BEGIN

      lut_map = REPLICATE(1,input_header.nluts)

    ENDELSE

    ; get energy cal if present (NB if not set up then wont be put into output header)

    IF ies_tag_exists(input_header,'ENERGY_CAL') THEN energy_cal = input_header.energy_cal

    ; get energy cal if present (NB if not set up then wont be put into output header)

    IF ies_tag_exists(input_header,'COUNT_CAL') THEN count_cal = input_header.count_cal

    ; get mf source tag

    IF ies_tag_exists(input_header,'MF') THEN mf_source = input_header.mf

    ; get pa source tag

    IF ies_tag_exists(input_header,'PA') THEN pa_source = input_header.pa

    ; set up magnetic field structure and source for mf

    IF KEYWORD_SET(mf) THEN BEGIN

      ; get magnetic field structure array
      ; force magnetic field to be re-read

      mag_field = get_sc_mf ( input_header, input_data, /FORCE )

      IF NOT KEYWORD_SET(mag_field) THEN BEGIN

        MESSAGE, 'MISSING MF DATA', /INFORMATIONAL

        RETURN

      ENDIF

      ; set up magnetic field source

      mf_source = ies_magnetic_field_source(mag_field)

    ENDIF

    ; set up magnetic field structure and source for pa

    IF KEYWORD_SET(pa) THEN BEGIN

      ; check if magnetic field already read
      ; get magnetic field structure array
      ; if mf tag exists then will use


      IF NOT KEYWORD_SET(mf) THEN mag_field = get_sc_mf ( input_header, input_data )


      IF NOT KEYWORD_SET(mag_field) THEN BEGIN

        MESSAGE, 'MISSING MF DATA', /INFORMATIONAL

        RETURN

      ENDIF

      ; set up magnetic field source

      pa_source = ies_magnetic_field_source(mag_field)

    ENDIF

    ; create new output header structure
    ; previously copied structrue to structure with new name
    ; set energy efficiencies to 1
    ; this is valid provided /CAL is used
    ; may have other tags ie input_header.lut_times that want to transfer
    ; add nluts to structure name else will fail if processing old IF files

    output_header = ies_header_structure ( datatype, no_of_points, no_of_detectors, no_of_energies, no_of_sectors, no_of_luts, $
                                           EXPERIMENT=experiment, $
                                           ENERGY_CALIBRATION=energy_cal, $
                                           COUNT_CALIBRATION=count_cal, $
                                           ENERGY_EFFICIENCY=REPLICATE(1.0,no_of_energies,no_of_luts), $
                                           LUT_MAP=lut_map, LUT_TIMES=lut_times, $
                                           MF=mf_source, PA=pa_source )

    ; copy everything from old to new header structure except exceptions

    IF KEYWORD_SET(integral) THEN BEGIN

      ; set up exceptions

      exceptions = ['datatype','ch_positions','energy_efficiency','detectors','ptitle']

      IF KEYWORD_SET(polar) THEN BEGIN
        e0 = 1
        e1 = input_header.nbands-2
      ENDIF ELSE BEGIN
        e0 = 0
        e1 = input_header.nbands-1
      ENDELSE

      ; set energy channels

      output_header.ch_positions[0,0,0,*] = input_header.ch_positions[0,e0,0,*]
      output_header.ch_positions[1,0,0,*] = input_header.ch_positions[1,e1,0,*]

      output_header.cal = 1

      output_header.rat = 1

      output_header.detectors = 1

      output_header.ptitle = 'Sum'

    ENDIF ELSE BEGIN

      exceptions = ['datatype']

    ENDELSE

    ; copy all header structure tags except exceptions

    copy_struct, input_header, output_header, EXCEPT=exceptions

    ; do data
    ; check for mvals and other tags and pass them on

    output_data = ies_data_structure ( no_of_points, no_of_detectors, no_of_energies, no_of_sectors, $
                                       EXPERIMENT=experiment, $
                                       GSE=KEYWORD_SET(gse) OR ies_tag_exists(input_data[0],'GSE'), $
                                       GSM=KEYWORD_SET(gsm) OR ies_tag_exists(input_data[0],'GSM'), $
                                       SM=KEYWORD_SET(sm) OR ies_tag_exists(input_data[0],'SM'), $
                                       PA=KEYWORD_SET(pa) OR ies_tag_exists(input_data[0],'PA'), $
                                       MF=KEYWORD_SET(mf) OR ies_tag_exists(input_data[0],'MF'), $
                                       MVALS=ies_tag_exists(input_data[0],'MVALS') )

    ; set up exceptions

    exceptions = ''

    IF KEYWORD_SET(integral) THEN exceptions = [exceptions,'data']

    IF KEYWORD_SET(endtime) THEN exceptions = [exceptions,'endtime']

    ; copy from input data to output data

    copy_struct, input_data, output_data, EXCEPT=exceptions

    ; set up integral data

    IF KEYWORD_SET(integral) THEN BEGIN

      ; use [*] to avoid data structure conflicts (under windows ?)
      ; blasted idl and its stupid array indexing

        output_data.data[*] = ies_sum(input_header, input_data, /CAL, /RAT, /NOBAD, E0=e0, E1=e1)

    ENDIF

    ; set up end time tag from start times

    IF KEYWORD_SET(endtime) THEN output_data.endtime = ies_endtimes(input_data)

    ; set up mf tag
    ; if no .mf tag in output_data then will just create mfield

    IF KEYWORD_SET(mf) OR KEYWORD_SET(pa) THEN BEGIN

      ; get sample times and offsets

      midtimes = ies_midtimes(output_data)

      ; get sun offsets

      offsets = ies_sun_offsets(output_header.datatype, midtimes)

      ; get magnetic field array

      mfield = ies_get_mf(mag_field, midtimes, offsets)

      ; set up magnetic field
      ; overwrite any existing .mf tag with new magnetic field data

      IF KEYWORD_SET(mf) THEN output_data.mf = mfield

      ; set up pitch angles
      ; overwrite any existing .pa tag with pitch angles derived with new magnetic field data

      IF KEYWORD_SET(pa) THEN BEGIN

        ; get pitch angles
        ; pas is [nsectors,ndetectors,npoints]
        ; pass mf explicitly since output_data does not necessarily have a .mf tag

        ies_pitch_angle, output_header, output_data, pas,  MF=mfield

        ; NB have not applied SUN or EARTH masks

        output_data.pa = reform(pas)

      ENDIF

    ENDIF

    ; set up gse, gsm or sm tags

    IF KEYWORD_SET(gse) OR KEYWORD_SET(gsm) OR KEYWORD_SET(sm) THEN BEGIN

      ephem = get_sc_ephem ( input_header )

      gse_x = INTERPOL(ephem.gse_xyz[0], ephem.time, output_data.time)/6378

      gse_y = INTERPOL(ephem.gse_xyz[1], ephem.time, output_data.time)/6378

      gse_z = INTERPOL(ephem.gse_xyz[2], ephem.time, output_data.time)/6378

      IF KEYWORD_SET(gse) THEN BEGIN

        output_data.gse_xyz[0] = gse_x
        output_data.gse_xyz[1] = gse_y
        output_data.gse_xyz[2] = gse_z

      ENDIF

      IF KEYWORD_SET(gsm) OR KEYWORD_SET(sm) THEN BEGIN

        FOR k =0, input_header.npoints-1 DO BEGIN

          time = anytim2utc(input_data[k].time,/external)

          recalc, time.year, utc2doy(time), time.hour, time.minute, time.second

          gsmgse, xgsm, ygsm, zgsm, gse_x[k], gse_y[k], gse_z[k], -1

          IF KEYWORD_SET(gsm) THEN output_data[k].gsm_xyz=[xgsm,ygsm,zgsm]

          IF KEYWORD_SET(sm) THEN BEGIN

            smgsm, xsm, ysm, zsm, xgsm, ygsm, zgsm, -1

            output_data[k].sm_xyz=[xsm,ysm,zsm]

          ENDIF

        ENDFOR

      ENDIF

    ENDIF

    IF ies_instrument(output_header.datatype,/POLAR) THEN BEGIN

      cep_save_ifset, output_header, output_data, cep_ifname_day(files[f])

    ENDIF ELSE IF ies_instrument(input_header.datatype,/RAPID) THEN BEGIN

      rcs_save_ifset, output_header, output_data, rcs_ifname_day(files[f])

    ENDIF ELSE BEGIN

      MESSAGE, 'IMPLEMENTATION ERROR', /TRACEBACK

    ENDELSE

  ENDFOR

END
